/**
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { ACCESS_TOKEN, AUTHORIZATION_CODE_TYPE, Hooks, OIDC_SCOPE, REFRESH_TOKEN, Storage } from "./constants";
import { isWebWorkerConfig } from "./helpers";
import { HttpClient } from "./http-client";
import { customGrant as customGrantUtil, endAuthenticatedSession, getAccessToken as getAccessTokenUtil, getDecodedIDToken, getServiceEndpoints, getSessionParameter, getUserInfo as getUserInfoUtil, handleSignIn, handleSignOut, isLoggedOut, resetOPConfiguration, sendRefreshTokenRequest, sendRevokeTokenRequest } from "./utils";
import { WebWorkerClient } from "./worker";
/**
 * Default configurations.
 */
const DefaultConfig = {
    authorizationType: AUTHORIZATION_CODE_TYPE,
    clientHost: origin,
    clientSecret: null,
    clockTolerance: 60,
    consentDenied: false,
    enablePKCE: true,
    responseMode: null,
    scope: [OIDC_SCOPE],
    sendCookiesInRequests: false,
    validateIDToken: true
};
/**
 * IdentityClient class constructor.
 *
 * @export
 * @class IdentityClient
 * @implements {ConfigInterface} - Configuration interface.
 */
export class IdentityClient {
    // eslint-disable-next-line @typescript-eslint/no-empty-function
    constructor() {
        this._startedInitialize = false;
        this._onCustomGrant = new Map();
    }
    static getInstance() {
        if (this._instance) {
            return this._instance;
        }
        this._instance = new IdentityClient();
        return this._instance;
    }
    /**
     * This method initializes the `IdentityClient` instance.
     *
     * @preserve
     *
     * @param {ConfigInterface | WebWorkerConfigInterface} config The config object to initialize with.
     *
     * @return {Promise<boolean>} Resolves to true if initialization is successful.
     */
    initialize(config) {
        var _a;
        if (!config.signOutRedirectURL) {
            config.signOutRedirectURL = config.signInRedirectURL;
        }
        this._storage = (_a = config.storage) !== null && _a !== void 0 ? _a : Storage.SessionStorage;
        this._initialized = false;
        this._startedInitialize = true;
        const startCallback = (request) => {
            request.headers = Object.assign(Object.assign({}, request.headers), { Authorization: `Bearer ${getSessionParameter(ACCESS_TOKEN, config)}` });
            this._onHttpRequestStart && typeof this._onHttpRequestStart === "function" && this._onHttpRequestStart();
        };
        if (!isWebWorkerConfig(config)) {
            this._authConfig = Object.assign(Object.assign({}, DefaultConfig), config);
            this._initialized = true;
            this._httpClient = HttpClient.getInstance();
            this._httpClient.init(true, startCallback, this._onHttpRequestSuccess, this._onHttpRequestError, this._onHttpRequestFinish);
            if (this._onInitialize) {
                this._onInitialize(true);
            }
            return Promise.resolve(true);
        }
        else {
            this._client = WebWorkerClient.getInstance();
            return this._client
                .initialize(Object.assign(Object.assign({}, DefaultConfig), config))
                .then(() => {
                if (this._onInitialize) {
                    this._onInitialize(true);
                }
                this._initialized = true;
                return Promise.resolve(true);
            })
                .catch((error) => {
                return Promise.reject(error);
            });
        }
    }
    getUserInfo() {
        if (this._storage === Storage.WebWorker) {
            return this._client.getUserInfo();
        }
        return Promise.resolve(getUserInfoUtil(this._authConfig));
    }
    validateAuthentication() {
        // TODO: Implement
        return;
    }
    /**
     * Sign-in method.
     *
     * @param {() => void} [callback] - Callback method to run on successful sign-in
     * @returns {Promise<any>} promise.
     * @memberof IdentityClient
     */
    signIn(fidp) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this._startedInitialize) {
                return Promise.reject("The object has not been initialized yet.");
            }
            let iterationToWait = 0;
            const sleep = () => {
                return new Promise((resolve) => setTimeout(resolve, 500));
            };
            while (!this._initialized) {
                if (iterationToWait === 21) {
                    // eslint-disable-next-line no-console
                    console.warn("It is taking longer than usual for the object to be initialized");
                }
                yield sleep();
                iterationToWait++;
            }
            if (this._storage === Storage.WebWorker) {
                return this._client
                    .signIn(fidp)
                    .then((response) => {
                    if (this._onSignInCallback) {
                        if (response.allowedScopes || response.displayName || response.email || response.username) {
                            this._onSignInCallback(response);
                        }
                    }
                    return Promise.resolve(response);
                })
                    .catch((error) => {
                    return Promise.reject(error);
                });
            }
            return handleSignIn(this._authConfig, fidp)
                .then(() => {
                if (this._onSignInCallback) {
                    const userInfo = getUserInfoUtil(this._authConfig);
                    if (userInfo.allowedScopes || userInfo.displayName || userInfo.email || userInfo.username) {
                        this._onSignInCallback(getUserInfoUtil(this._authConfig));
                    }
                }
                return Promise.resolve(getUserInfoUtil(this._authConfig));
            })
                .catch((error) => {
                return Promise.reject(error);
            });
        });
    }
    /**
     * Sign-out method.
     *
     * @param {() => void} [callback] - Callback method to run on successful sign-in
     * @returns {Promise<any>} promise.
     * @memberof IdentityClient
     */
    signOut() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this._storage === Storage.WebWorker) {
                return this._client
                    .signOut()
                    .then((response) => {
                    return Promise.resolve(response);
                })
                    .catch((error) => {
                    return Promise.reject(error);
                });
            }
            return handleSignOut(this._authConfig)
                .then((response) => {
                return Promise.resolve(response);
            })
                .catch((error) => {
                return Promise.reject(error);
            });
        });
    }
    httpRequest(config) {
        return __awaiter(this, void 0, void 0, function* () {
            if (this._storage === Storage.WebWorker) {
                return this._client.httpRequest(config);
            }
            return this._httpClient.request(config);
        });
    }
    httpRequestAll(config) {
        return __awaiter(this, void 0, void 0, function* () {
            if (this._storage === Storage.WebWorker) {
                return this._client.httpRequestAll(config);
            }
            const requests = [];
            config.forEach((request) => {
                requests.push(this._httpClient.request(request));
            });
            return this._httpClient.all(requests);
        });
    }
    customGrant(requestParams) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!requestParams.id) {
                throw Error("No ID specified for the custom grant.");
            }
            if (this._storage === Storage.WebWorker) {
                return this._client
                    .customGrant(requestParams)
                    .then((response) => {
                    if (this._onCustomGrant.get(requestParams.id)) {
                        this._onCustomGrant.get(requestParams.id)(response);
                    }
                    return Promise.resolve(response);
                })
                    .catch((error) => {
                    return Promise.reject(error);
                });
            }
            return customGrantUtil(requestParams, this._authConfig)
                .then((response) => {
                if (this._onCustomGrant.get(requestParams.id)) {
                    this._onCustomGrant.get(requestParams.id)(response);
                }
                return Promise.resolve(response);
            })
                .catch((error) => {
                return Promise.reject(error);
            });
        });
    }
    endUserSession() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this._storage === Storage.WebWorker) {
                return this._client
                    .endUserSession()
                    .then((response) => {
                    if (this._onEndUserSession) {
                        this._onEndUserSession(response);
                        return Promise.resolve(response);
                    }
                })
                    .catch((error) => {
                    return Promise.reject(error);
                });
            }
            return sendRevokeTokenRequest(this._authConfig, getSessionParameter(ACCESS_TOKEN, this._authConfig))
                .then((response) => {
                resetOPConfiguration(this._authConfig);
                endAuthenticatedSession(this._authConfig);
                if (this._onEndUserSession) {
                    this._onEndUserSession(response);
                    return Promise.resolve(true);
                }
            })
                .catch((error) => {
                return Promise.reject(error);
            });
        });
    }
    getServiceEndpoints() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this._storage === Storage.WebWorker) {
                return this._client.getServiceEndpoints();
            }
            return Promise.resolve(getServiceEndpoints(this._authConfig));
        });
    }
    getHttpClient() {
        if (this._initialized) {
            return this._httpClient;
        }
        throw Error("Identity Client has not been initialized yet");
    }
    getDecodedIDToken() {
        if (this._storage === Storage.WebWorker) {
            return this._client.getDecodedIDToken();
        }
        return Promise.resolve(getDecodedIDToken(this._authConfig));
    }
    getAccessToken() {
        if (this._storage === Storage.WebWorker) {
            return Promise.reject("The access token cannot be obtained when the storage type is set to webWorker.");
        }
        return getAccessTokenUtil(this._authConfig);
    }
    refreshToken() {
        if (this._storage === Storage.WebWorker) {
            return Promise.reject("The token is automatically refreshed when the storage type is set to webWorker.");
        }
        return sendRefreshTokenRequest(this._authConfig, getSessionParameter(REFRESH_TOKEN, this._authConfig));
    }
    on(hook, callback, id) {
        if (callback && typeof callback === "function") {
            switch (hook) {
                case Hooks.SignIn:
                    this._onSignInCallback = callback;
                    break;
                case Hooks.SignOut:
                    this._onSignOutCallback = callback;
                    if (isLoggedOut()) {
                        this._onSignOutCallback();
                    }
                    break;
                case Hooks.EndUserSession:
                    this._onEndUserSession = callback;
                    break;
                case Hooks.Initialize:
                    this._onInitialize = callback;
                    break;
                case Hooks.HttpRequestError:
                    if (this._storage === Storage.WebWorker) {
                        this._client.onHttpRequestError(callback);
                    }
                    this._onHttpRequestError = callback;
                    break;
                case Hooks.HttpRequestFinish:
                    if (this._storage === Storage.WebWorker) {
                        this._client.onHttpRequestFinish(callback);
                    }
                    this._onHttpRequestFinish = callback;
                    break;
                case Hooks.HttpRequestStart:
                    if (this._storage === Storage.WebWorker) {
                        this._client.onHttpRequestStart(callback);
                    }
                    this._onHttpRequestStart = callback;
                    break;
                case Hooks.HttpRequestSuccess:
                    if (this._storage === Storage.WebWorker) {
                        this._client.onHttpRequestSuccess(callback);
                    }
                    this._onHttpRequestSuccess = callback;
                    break;
                case Hooks.CustomGrant:
                    this._onCustomGrant.set(id, callback);
                    break;
                default:
                    throw Error("No such hook found");
            }
        }
        else {
            throw Error("The callback function is not a valid function.");
        }
    }
}
//# sourceMappingURL=client.js.map