/**
 * Copyright (c) 2019, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import axios from "axios";
import { getSessionParameter, removeSessionParameter, setSessionParameter } from "./session-storage";
import { AUTHORIZATION_ENDPOINT, CLIENT_ID, END_SESSION_ENDPOINT, ISSUER, JWKS_ENDPOINT, OIDC_SESSION_IFRAME_ENDPOINT, OPEN_ID_CONFIG, OP_CONFIG_INITIATED, REVOKE_TOKEN_ENDPOINT, SERVICE_RESOURCES, SIGN_IN_REDIRECT_URL, SIGN_OUT_REDIRECT_URL, Storage, TENANT, TOKEN_ENDPOINT, USERNAME } from "../constants";
/**
 * Checks whether openid configuration initiated.
 *
 * @returns {boolean}
 */
export const isOPConfigInitiated = (requestParams) => {
    if (requestParams.storage !== Storage.WebWorker) {
        return (getSessionParameter(OP_CONFIG_INITIATED, requestParams) &&
            getSessionParameter(OP_CONFIG_INITIATED, requestParams) === "true");
    }
    else {
        return (requestParams.session.get(OP_CONFIG_INITIATED) && requestParams.session.get(OP_CONFIG_INITIATED) === "true");
    }
};
/**
 * Set OAuth2 authorize endpoint.
 *
 * @param {string} authorizationEndpoint
 */
export const setAuthorizeEndpoint = (authorizationEndpoint, requestParams) => {
    setSessionParameter(AUTHORIZATION_ENDPOINT, authorizationEndpoint, requestParams);
};
/**
 * Sets the open id config data on the session.
 *
 * @param {OpenIDConfig} data - The Open ID Config.
 * @param {ConfigInterface} requestParams - Initial Config.
 */
export const setOPConfig = (data, requestParams) => {
    setSessionParameter(OPEN_ID_CONFIG, JSON.stringify(data), requestParams);
};
/**
 * Returns the Open ID config from the session.
 *
 * @param {ConfigInterface} requestParams - Initial Config.
 *
 * @return {OpenIDConfig} - Open ID Config.
 */
export const getOPConfig = (requestParams) => {
    return JSON.parse(getSessionParameter(OPEN_ID_CONFIG, requestParams));
};
/**
 * Set OAuth2 token endpoint.
 *
 * @param {string} tokenEndpoint
 */
export const setTokenEndpoint = (tokenEndpoint, requestParams) => {
    setSessionParameter(TOKEN_ENDPOINT, tokenEndpoint, requestParams);
};
/**
 * Set OIDC end session endpoint.
 *
 * @param {string} endSessionEndpoint
 */
export const setEndSessionEndpoint = (endSessionEndpoint, requestParams) => {
    setSessionParameter(END_SESSION_ENDPOINT, endSessionEndpoint, requestParams);
};
/**
 * Set JWKS URI.
 *
 * @param jwksEndpoint
 */
export const setJwksUri = (jwksEndpoint, requestParams) => {
    setSessionParameter(JWKS_ENDPOINT, jwksEndpoint, requestParams);
};
/**
 * Set OAuth2 revoke token endpoint.
 *
 * @param {string} revokeTokenEndpoint
 */
export const setRevokeTokenEndpoint = (revokeTokenEndpoint, requestParams) => {
    setSessionParameter(REVOKE_TOKEN_ENDPOINT, revokeTokenEndpoint, requestParams);
};
/**
 * Set openid configuration initiated.
 */
export const setOPConfigInitiated = (requestParams) => {
    setSessionParameter(OP_CONFIG_INITIATED, "true", requestParams);
};
/**
 * Set sign-in redirect URL.
 */
export const setSignInRedirectURL = (url, requestParams) => {
    setSessionParameter(SIGN_IN_REDIRECT_URL, url, requestParams);
};
/**
 * Set sign-out redirect URL.
 */
export const setSignOutRedirectURL = (url, requestParams) => {
    setSessionParameter(SIGN_OUT_REDIRECT_URL, url, requestParams);
};
/**
 * Set OIDC Session IFrame URL.
 */
export const setOIDCSessionIFrameURL = (url, requestParams) => {
    setSessionParameter(OIDC_SESSION_IFRAME_ENDPOINT, url, requestParams);
};
/**
 * Set id_token issuer.
 *
 * @param issuer id_token issuer.
 */
export const setIssuer = (issuer, requestParams) => {
    setSessionParameter(ISSUER, issuer, requestParams);
};
/**
 * Set Client ID.
 *
 * @param {string} clientID - Client ID of the application.
 */
export const setClientID = (requestParams) => {
    setSessionParameter(CLIENT_ID, requestParams.clientID, requestParams);
};
/**
 * Initialize openid provider configuration.
 *
 * @param {string} wellKnownEndpoint openid provider configuration.
 * @param {boolean} forceInit whether to initialize the configuration again.
 * @returns {Promise<any>} promise.
 */
/* eslint-disable @typescript-eslint/no-explicit-any */
export const initOPConfiguration = (requestParams, forceInit) => {
    var _a;
    if (!forceInit && isValidOPConfig(requestParams)) {
        return Promise.resolve();
    }
    const serverHost = requestParams.serverOrigin;
    let wellKnownEndpoint = serverHost + SERVICE_RESOURCES.wellKnown;
    if ((_a = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _a === void 0 ? void 0 : _a.wellKnown) {
        wellKnownEndpoint = requestParams.endpoints.wellKnown;
    }
    return axios
        .get(wellKnownEndpoint)
        .then((response) => {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        if (response.status !== 200) {
            return Promise.reject(new Error("Failed to load OpenID provider configuration from: " + serverHost + SERVICE_RESOURCES.wellKnown));
        }
        setOPConfig(response.data, requestParams);
        if ((_a = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _a === void 0 ? void 0 : _a.authorize) {
            setAuthorizeEndpoint((_b = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _b === void 0 ? void 0 : _b.authorize, requestParams);
        }
        else {
            setAuthorizeEndpoint(response.data.authorization_endpoint, requestParams);
        }
        if ((_c = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _c === void 0 ? void 0 : _c.token) {
            setTokenEndpoint((_d = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _d === void 0 ? void 0 : _d.token, requestParams);
        }
        else {
            setTokenEndpoint(response.data.token_endpoint, requestParams);
        }
        if ((_e = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _e === void 0 ? void 0 : _e.jwks) {
            setJwksUri((_f = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _f === void 0 ? void 0 : _f.jwks, requestParams);
        }
        else {
            setJwksUri(response.data.jwks_uri, requestParams);
        }
        if ((_g = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _g === void 0 ? void 0 : _g.logout) {
            setEndSessionEndpoint((_h = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _h === void 0 ? void 0 : _h.logout, requestParams);
        }
        else {
            setEndSessionEndpoint(response.data.end_session_endpoint, requestParams);
        }
        if ((_j = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _j === void 0 ? void 0 : _j.revoke) {
            setRevokeTokenEndpoint((_k = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _k === void 0 ? void 0 : _k.revoke, requestParams);
        }
        else {
            setRevokeTokenEndpoint(response.data.token_endpoint.substring(0, response.data.token_endpoint.lastIndexOf("token")) +
                "revoke", requestParams);
        }
        setIssuer(response.data.issuer, requestParams);
        setClientID(requestParams);
        if ((_l = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _l === void 0 ? void 0 : _l.oidcSessionIFrame) {
            setOIDCSessionIFrameURL((_m = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _m === void 0 ? void 0 : _m.oidcSessionIFrame, requestParams);
        }
        else {
            setOIDCSessionIFrameURL(response.data.check_session_iframe, requestParams);
        }
        setSignInRedirectURL(requestParams.signInRedirectURL, requestParams);
        setSignOutRedirectURL(requestParams.signOutRedirectURL, requestParams);
        setOPConfigInitiated(requestParams);
        return Promise.resolve("Initialized OpenID Provider configuration from: " + serverHost + SERVICE_RESOURCES.wellKnown);
    })
        .catch(() => {
        var _a, _b, _c, _d, _e, _f, _g;
        setAuthorizeEndpoint(((_a = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _a === void 0 ? void 0 : _a.authorize) || (requestParams.serverOrigin + SERVICE_RESOURCES.authorize), requestParams);
        setTokenEndpoint(((_b = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _b === void 0 ? void 0 : _b.token) || (requestParams.serverOrigin + SERVICE_RESOURCES.token), requestParams);
        setRevokeTokenEndpoint(((_c = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _c === void 0 ? void 0 : _c.revoke) || (requestParams.serverOrigin + SERVICE_RESOURCES.revoke), requestParams);
        setEndSessionEndpoint(((_d = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _d === void 0 ? void 0 : _d.logout) || (requestParams.serverOrigin + SERVICE_RESOURCES.logout), requestParams);
        setJwksUri(((_e = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _e === void 0 ? void 0 : _e.jwks) || (serverHost + SERVICE_RESOURCES.jwks), requestParams);
        setIssuer(((_f = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _f === void 0 ? void 0 : _f.token) || (requestParams.serverOrigin + SERVICE_RESOURCES.token), requestParams);
        setClientID(requestParams);
        setOIDCSessionIFrameURL(((_g = requestParams === null || requestParams === void 0 ? void 0 : requestParams.endpoints) === null || _g === void 0 ? void 0 : _g.oidcSessionIFrame) || (requestParams.serverOrigin + SERVICE_RESOURCES.oidcSessionIFrame), requestParams);
        setSignInRedirectURL(requestParams.signInRedirectURL, requestParams);
        setSignOutRedirectURL(requestParams.signOutRedirectURL, requestParams);
        setOPConfigInitiated(requestParams);
        return Promise.resolve(new Error("Initialized OpenID Provider configuration from default configuration." +
            "Because failed to access wellknown endpoint: " +
            serverHost +
            SERVICE_RESOURCES.wellKnown));
    });
};
/**
 * Reset openid provider configuration.
 */
export const resetOPConfiguration = (requestParams) => {
    if (requestParams.storage !== Storage.WebWorker) {
        removeSessionParameter(AUTHORIZATION_ENDPOINT, requestParams);
        removeSessionParameter(TOKEN_ENDPOINT, requestParams);
        removeSessionParameter(END_SESSION_ENDPOINT, requestParams);
        removeSessionParameter(JWKS_ENDPOINT, requestParams);
        removeSessionParameter(REVOKE_TOKEN_ENDPOINT, requestParams);
        removeSessionParameter(OP_CONFIG_INITIATED, requestParams);
        removeSessionParameter(ISSUER, requestParams);
        removeSessionParameter(CLIENT_ID, requestParams);
        removeSessionParameter(TENANT, requestParams);
        removeSessionParameter(SIGN_IN_REDIRECT_URL, requestParams);
        removeSessionParameter(SIGN_OUT_REDIRECT_URL, requestParams);
        removeSessionParameter(OIDC_SESSION_IFRAME_ENDPOINT, requestParams);
    }
    else {
        requestParams.session.clear();
    }
};
export const getServiceEndpoints = (authConfig) => {
    return {
        authorize: getAuthorizeEndpoint(authConfig),
        jwks: getJwksUri(authConfig),
        logout: getEndSessionEndpoint(authConfig),
        oidcSessionIFrame: getOIDCSessionIFrameURL(authConfig),
        revoke: getRevokeTokenEndpoint(authConfig),
        token: getTokenEndpoint(authConfig),
        wellKnown: SERVICE_RESOURCES.wellKnown
    };
};
/**
 * Get OAuth2 authorize endpoint.
 *
 * @returns {string|null}
 */
export const getAuthorizeEndpoint = (requestParams) => {
    return getSessionParameter(AUTHORIZATION_ENDPOINT, requestParams);
};
/**
 * Get OAuth2 token endpoint.
 *
 * @returns {string|null}
 */
export const getTokenEndpoint = (requestParams) => {
    return getSessionParameter(TOKEN_ENDPOINT, requestParams);
};
/**
 * Get OAuth2 revoke token endpoint.
 *
 * @returns {string|null}
 */
export const getRevokeTokenEndpoint = (requestParams) => {
    return getSessionParameter(REVOKE_TOKEN_ENDPOINT, requestParams);
};
export const getOIDCSessionIFrameURL = (requestParams) => {
    return getSessionParameter(OIDC_SESSION_IFRAME_ENDPOINT, requestParams);
};
/**
 * Get OIDC end session endpoint.
 *
 * @returns {string|null}
 */
export const getEndSessionEndpoint = (requestParams) => {
    return getSessionParameter(END_SESSION_ENDPOINT, requestParams);
};
/**
 * Get JWKS URI.
 *
 * @returns {string|null}
 */
export const getJwksUri = (requestParams) => {
    return getSessionParameter(JWKS_ENDPOINT, requestParams);
};
/**
 * Get authenticated user's username
 *
 * @returns {string|null}
 */
export const getUsername = (requestParams) => {
    return getSessionParameter(USERNAME, requestParams);
};
/**
 * Get tenant name
 *
 * @returns {any}
 */
export const getTenant = (requestParams) => {
    return getSessionParameter(TENANT, requestParams);
};
/**
 * Get id_token issuer.
 *
 * @returns {any}
 */
export const getIssuer = (requestParams) => {
    return getSessionParameter(ISSUER, requestParams);
};
/**
 * Get Client ID.
 *
 * @return {string}
 */
export const getClientID = (requestParams) => {
    return getSessionParameter(CLIENT_ID, requestParams);
};
/**
 * Checks whether openid configuration initiated is valid.
 *
 * @param {string} tenant - Tenant of the logged in user.
 * @param {string} clientID - Client ID of the application.
 * @return {boolean}
 */
export const isValidOPConfig = (requestParams) => {
    return (isOPConfigInitiated(requestParams) &&
        !!getClientID(requestParams) &&
        getClientID(requestParams) === requestParams.clientID);
};
//# sourceMappingURL=op-config.js.map