/**
 * Copyright (c) 2019, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { AxiosResponse } from "axios";
import { AuthenticatedUserInterface, ConfigInterface, CustomGrantRequestParams, DecodedIdTokenPayloadInterface, SignInResponse, TokenRequestHeader, TokenResponseInterface, UserInfo, WebWorkerConfigInterface } from "../models";
/**
 * Checks whether authorization code is present.
 *
 * @returns {boolean} true if authorization code is present.
 */
export declare function hasAuthorizationCode(requestParams: ConfigInterface | WebWorkerConfigInterface): boolean;
/**
 * Resolves the authorization code.
 * If response mode is `form_post` authorization code is taken from the session storage.
 * And if the response mode is not defined or set to `query`, the code will be extracted from
 * the URL params.
 *
 * @returns {string} Resolved authorization code.
 */
export declare function getAuthorizationCode(requestParams?: ConfigInterface | WebWorkerConfigInterface): string;
/**
 * Get token request headers.
 *
 * @param {string} clientHost
 * @returns {TokenRequestHeader}
 */
export declare const getTokenRequestHeaders: () => TokenRequestHeader;
/**
 * Send authorization request.
 *
 * @param {ConfigInterface | WebWorkerConfigInterface} requestParams
 * request parameters required for authorization request.
 */
export declare function sendAuthorizationRequest(requestParams: ConfigInterface | WebWorkerConfigInterface, fidp?: string): Promise<SignInResponse | never>;
/**
 * Validate id_token.
 *
 * @param {string} clientID client ID.
 * @param {string} idToken id_token received from the IdP.
 * @returns {Promise<boolean>} whether token is valid.
 */
export declare function validateIdToken(idToken: string, requestParams: ConfigInterface | WebWorkerConfigInterface): Promise<any>;
/**
 * Send token request.
 *
 * @param {ConfigInterface | WebWorkerConfigInterface} requestParams request parameters required for token request.
 * @returns {Promise<TokenResponseInterface>} token response data or error.
 */
export declare function sendTokenRequest(requestParams: ConfigInterface | WebWorkerConfigInterface): Promise<TokenResponseInterface>;
/**
 * Send refresh token request.
 *
 * @param {ConfigInterface | WebWorkerConfigInterface} requestParams request parameters required for token request.
 * @param {string} refreshToken
 * @returns {Promise<TokenResponseInterface>} refresh token response data or error.
 */
export declare function sendRefreshTokenRequest(requestParams: ConfigInterface | WebWorkerConfigInterface, refreshToken: string): Promise<any>;
/**
 * Send revoke token request.
 *
 * @param {ConfigInterface | WebWorkerConfigInterface} requestParams
 * request parameters required for revoke token request.
 * @param {string} accessToken access token
 * @returns {any}
 */
export declare function sendRevokeTokenRequest(requestParams: ConfigInterface | WebWorkerConfigInterface, accessToken: string): Promise<any>;
/**
 * Get authenticated user from the id_token.
 *
 * @param idToken id_token received from the IdP.
 * @returns {AuthenticatedUserInterface} authenticated user.
 */
export declare const getAuthenticatedUser: (idToken: string) => AuthenticatedUserInterface;
/**
 * Execute user sign in request
 *
 * @param {object} requestParams
 * @param {function} callback
 * @returns {Promise<any>} sign out request status
 */
export declare function sendSignInRequest(requestParams: ConfigInterface | WebWorkerConfigInterface, fidp?: string): Promise<SignInResponse>;
export declare function handleSignIn(requestParams: ConfigInterface | WebWorkerConfigInterface, fidp?: string): Promise<any>;
/**
 * Allows using custom grant types.
 *
 * @param {CustomGrantRequestParams} requestParams - The request parameters.
 * @param {ConfigInterface | WebWorkerConfigInterface} authConfig - Auth config object.
 * @returns {Promise<boolean|AxiosResponse>} A promise that resolves with a boolean value or the request response
 * if the the `returnResponse` attribute in the `requestParams` object is set to `true`.
 */
export declare const customGrant: (requestParams: CustomGrantRequestParams, authConfig: ConfigInterface | WebWorkerConfigInterface | WebWorkerConfigInterface | WebWorkerConfigInterface) => Promise<SignInResponse | boolean | AxiosResponse>;
/**
 * Returns email, username, display name and allowed scopes.
 *
 * @returns {UserInfo} User information.
 */
export declare const getUserInfo: (config: ConfigInterface | WebWorkerConfigInterface) => UserInfo;
/**
 *
 * @param {ConfigInterface} config - The configuration parameters.
 *
 * @return {DecodedIdTokenPayloadInterface} The decoded ID Token payload.
 */
export declare const getDecodedIDToken: (config: ConfigInterface | WebWorkerConfigInterface) => DecodedIdTokenPayloadInterface;
//# sourceMappingURL=sign-in.d.ts.map