/**
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { WebWorker } from "./web-worker";
import { API_CALL, API_CALL_ALL, AUTH_REQUIRED, CUSTOM_GRANT, END_USER_SESSION, GET_DECODED_ID_TOKEN, GET_SERVICE_ENDPOINTS, GET_USER_INFO, INIT, LOGOUT, REQUEST_ERROR, REQUEST_FINISH, REQUEST_START, REQUEST_SUCCESS, SIGNED_IN, SIGN_IN } from "../constants";
import { generateFailureDTO, generateSuccessDTO } from "../utils";
const ctx = self;
let webWorker;
ctx.onmessage = ({ data, ports }) => {
    var _a, _b, _c, _d;
    const port = ports[0];
    switch (data.type) {
        case INIT:
            try {
                const config = Object.assign({}, data.data);
                config.httpClient = Object.assign(Object.assign({}, config.httpClient), { requestErrorCallback: onRequestErrorCallback, requestFinishCallback: onRequestFinishCallback, requestStartCallback: onRequestStartCallback, requestSuccessCallback: onRequestSuccessCallback });
                webWorker = WebWorker.getInstance(config);
                port.postMessage(generateSuccessDTO());
            }
            catch (error) {
                port.postMessage(generateFailureDTO(error));
            }
            break;
        case SIGN_IN:
            if (!webWorker) {
                port.postMessage(generateFailureDTO("Worker has not been initiated."));
            }
            else {
                if (((_a = data === null || data === void 0 ? void 0 : data.data) === null || _a === void 0 ? void 0 : _a.code) || ((_b = data === null || data === void 0 ? void 0 : data.data) === null || _b === void 0 ? void 0 : _b.pkce)) {
                    webWorker.setAuthCode(data.data.code, (_c = data === null || data === void 0 ? void 0 : data.data) === null || _c === void 0 ? void 0 : _c.sessionState, (_d = data === null || data === void 0 ? void 0 : data.data) === null || _d === void 0 ? void 0 : _d.pkce);
                }
                webWorker
                    .signIn(typeof (data === null || data === void 0 ? void 0 : data.data) === "string" && (data === null || data === void 0 ? void 0 : data.data))
                    .then((response) => {
                    if (response.type === SIGNED_IN) {
                        port.postMessage(generateSuccessDTO({
                            data: response.data,
                            type: SIGNED_IN
                        }));
                    }
                    else {
                        port.postMessage(generateSuccessDTO({
                            code: response.code,
                            pkce: response.pkce,
                            type: AUTH_REQUIRED
                        }));
                    }
                })
                    .catch((error) => {
                    port.postMessage(generateFailureDTO(error));
                });
            }
            break;
        case API_CALL:
            if (!webWorker) {
                port.postMessage(generateFailureDTO("Worker has not been initiated."));
                break;
            }
            if (!webWorker.isSignedIn()) {
                port.postMessage(generateFailureDTO("You have not signed in yet."));
            }
            else {
                webWorker
                    .httpRequest(data.data)
                    .then((response) => {
                    port.postMessage(generateSuccessDTO(response));
                })
                    .catch((error) => {
                    port.postMessage(generateFailureDTO(error));
                });
            }
            break;
        case API_CALL_ALL:
            if (!webWorker) {
                port.postMessage(generateFailureDTO("Worker has not been initiated."));
                break;
            }
            if (!webWorker.isSignedIn()) {
                port.postMessage(generateFailureDTO("You have not signed in yet."));
            }
            else {
                webWorker
                    .httpRequestAll(data.data)
                    .then((response) => {
                    port.postMessage(generateSuccessDTO(response));
                })
                    .catch((error) => {
                    port.postMessage(generateFailureDTO(error));
                });
            }
            break;
        case LOGOUT:
            if (!webWorker) {
                port.postMessage(generateFailureDTO("Worker has not been initiated."));
                break;
            }
            if (!webWorker.isSignedIn()) {
                port.postMessage(generateFailureDTO("You have not signed in yet."));
            }
            else {
                webWorker
                    .signOut()
                    .then((response) => {
                    if (response) {
                        port.postMessage(generateSuccessDTO(response));
                    }
                    else {
                        port.postMessage(generateFailureDTO("Received no response"));
                    }
                })
                    .catch((error) => {
                    port.postMessage(generateFailureDTO(error));
                });
            }
            break;
        case CUSTOM_GRANT:
            if (!webWorker) {
                port.postMessage(generateFailureDTO("Worker has not been initiated."));
                break;
            }
            if (!webWorker.isSignedIn() && data.data.signInRequired) {
                port.postMessage(generateFailureDTO("You have not signed in yet."));
                break;
            }
            webWorker
                .customGrant(data.data)
                .then((response) => {
                port.postMessage(generateSuccessDTO(response));
            })
                .catch((error) => {
                port.postMessage(generateFailureDTO(error));
            });
            break;
        case END_USER_SESSION:
            if (!webWorker) {
                port.postMessage(generateFailureDTO("Worker has not been initiated."));
                break;
            }
            if (!webWorker.isSignedIn() && data.data.signInRequired) {
                port.postMessage(generateFailureDTO("You have not signed in yet."));
                break;
            }
            webWorker
                .endUserSession()
                .then((response) => {
                port.postMessage(generateSuccessDTO(response));
            })
                .catch((error) => {
                port.postMessage(generateFailureDTO(error));
            });
            break;
        case GET_SERVICE_ENDPOINTS:
            if (!webWorker) {
                port.postMessage(generateFailureDTO("Worker has not been initiated."));
                break;
            }
            webWorker.getServiceEndpoints().then(response => {
                port.postMessage(generateSuccessDTO(response));
            }).catch(error => {
                port.postMessage(generateFailureDTO(error));
            });
            break;
        case GET_USER_INFO:
            if (!webWorker) {
                port.postMessage(generateFailureDTO("Worker has not been initiated."));
                break;
            }
            if (!webWorker.isSignedIn() && data.data.signInRequired) {
                port.postMessage(generateFailureDTO("You have not signed in yet."));
                break;
            }
            try {
                port.postMessage(generateSuccessDTO(webWorker.getUserInfo()));
            }
            catch (error) {
                port.postMessage(generateFailureDTO(error));
            }
            break;
        case GET_DECODED_ID_TOKEN:
            if (!webWorker) {
                port.postMessage(generateFailureDTO("Worker has not been initiated."));
                break;
            }
            if (!webWorker.isSignedIn() && data.data.signInRequired) {
                port.postMessage(generateFailureDTO("You have not signed in yet."));
                break;
            }
            try {
                port.postMessage(generateSuccessDTO(webWorker.getDecodedIDToken()));
            }
            catch (error) {
                port.postMessage(generateFailureDTO(error));
            }
            break;
        default:
            port === null || port === void 0 ? void 0 : port.postMessage(generateFailureDTO(`Unknown message type ${data === null || data === void 0 ? void 0 : data.type}`));
    }
};
const onRequestStartCallback = () => {
    ctx.postMessage({ type: REQUEST_START });
};
const onRequestSuccessCallback = (response) => {
    ctx.postMessage({ data: JSON.stringify(response !== null && response !== void 0 ? response : ""), type: REQUEST_SUCCESS });
};
const onRequestFinishCallback = () => {
    ctx.postMessage({ type: REQUEST_FINISH });
};
const onRequestErrorCallback = (error) => {
    const errorObject = Object.assign({}, error);
    delete errorObject.toJSON;
    ctx.postMessage({ data: JSON.stringify(errorObject !== null && errorObject !== void 0 ? errorObject : ""), type: REQUEST_ERROR });
};
export default {};
//# sourceMappingURL=oidc.worker.js.map