"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getSize = exports.exists = exports.getCwd = exports.rmDist = exports.cleanup = exports.readFile = exports.readJson = exports.listFiles = exports.checkFilesExist = exports.renameFile = exports.updateFile = exports.expectTestsPass = exports.copyNodeModules = exports.fileExists = exports.directoryExists = void 0;
const fs_extra_1 = require("fs-extra");
const path_1 = require("path");
const paths_1 = require("./paths");
const devkit_1 = require("@nrwl/devkit");
const fileutils_1 = require("nx/src/utils/fileutils");
Object.defineProperty(exports, "directoryExists", { enumerable: true, get: function () { return fileutils_1.directoryExists; } });
Object.defineProperty(exports, "fileExists", { enumerable: true, get: function () { return fileutils_1.fileExists; } });
/**
 * Copies module folders from the working directory to the e2e directory
 * @param modules a list of module names or scopes to copy
 */
function copyNodeModules(modules) {
    modules.forEach((module) => {
        (0, fs_extra_1.removeSync)(`${(0, paths_1.tmpProjPath)()}/node_modules/${module}`);
        (0, fs_extra_1.copySync)(`./node_modules/${module}`, `${(0, paths_1.tmpProjPath)()}/node_modules/${module}`);
    });
}
exports.copyNodeModules = copyNodeModules;
/**
 * Assert test output from a asynchronous CLI command.
 *
 * @param output Output from an asynchronous command
 */
function expectTestsPass(output) {
    expect(output.stderr).toContain('Ran all test suites');
    expect(output.stderr).not.toContain('fail');
}
exports.expectTestsPass = expectTestsPass;
// type callback =
/**
 * Update a file's content in the e2e directory.
 *
 * If the `content` param is a callback, it will provide the original file content as an argument.
 *
 * @param file Path of the file in the e2e directory
 * @param content Content to replace the original content with
 */
function updateFile(file, content) {
    (0, fs_extra_1.ensureDirSync)((0, path_1.dirname)((0, paths_1.tmpProjPath)(file)));
    if (typeof content === 'string') {
        (0, fs_extra_1.writeFileSync)((0, paths_1.tmpProjPath)(file), content);
    }
    else {
        (0, fs_extra_1.writeFileSync)((0, paths_1.tmpProjPath)(file), content((0, fs_extra_1.readFileSync)((0, paths_1.tmpProjPath)(file)).toString()));
    }
}
exports.updateFile = updateFile;
/**
 * Rename a file or directory within the e2e directory.
 * @param path Original path
 * @param newPath New path
 */
function renameFile(path, newPath) {
    (0, fs_extra_1.ensureDirSync)((0, path_1.dirname)((0, paths_1.tmpProjPath)(newPath)));
    (0, fs_extra_1.renameSync)((0, paths_1.tmpProjPath)(path), (0, paths_1.tmpProjPath)(newPath));
}
exports.renameFile = renameFile;
/**
 * Check if the file or directory exists.
 *
 * If a path starts with `/` or `C:/`, it will check it as absolute. Otherwise it will check within the e2e directory.
 *
 * @param expectedPaths Files or directories to check
 * @usage `checkFileExists('file1', 'file2', '/var/user/file')`
 */
function checkFilesExist(...expectedPaths) {
    expectedPaths.forEach((path) => {
        const filePath = (0, path_1.isAbsolute)(path) ? path : (0, paths_1.tmpProjPath)(path);
        if (!exists(filePath)) {
            throw new Error(`'${filePath}' does not exist`);
        }
    });
}
exports.checkFilesExist = checkFilesExist;
/**
 * Get a list of all files within a directory.
 * @param dirName Directory name within the e2e directory.
 */
function listFiles(dirName) {
    return (0, fs_extra_1.readdirSync)((0, paths_1.tmpProjPath)(dirName));
}
exports.listFiles = listFiles;
/**
 * Read a JSON file.
 * @param path Path to the JSON file. Absolute or relative to the e2e directory.
 * @param options JSON parse options
 */
function readJson(path, options) {
    return (0, devkit_1.parseJson)(readFile(path), options);
}
exports.readJson = readJson;
/**
 * Read a file.
 * @param path Path to the file. Absolute or relative to the e2e directory.
 */
function readFile(path) {
    const filePath = (0, path_1.isAbsolute)(path) ? path : (0, paths_1.tmpProjPath)(path);
    return (0, fs_extra_1.readFileSync)(filePath, 'utf-8');
}
exports.readFile = readFile;
/**
 * Deletes the e2e directory
 */
function cleanup() {
    (0, fs_extra_1.removeSync)((0, paths_1.tmpProjPath)());
}
exports.cleanup = cleanup;
/**
 * Remove the dist folder from the e2e directory
 */
function rmDist() {
    (0, fs_extra_1.removeSync)(`${(0, paths_1.tmpProjPath)()}/dist`);
}
exports.rmDist = rmDist;
/**
 * Get the currend `cwd` in the process
 */
function getCwd() {
    return process.cwd();
}
exports.getCwd = getCwd;
/**
 * Check if a file or directory exists.
 * @param path Path to file or directory
 */
function exists(path) {
    return (0, fileutils_1.directoryExists)(path) || (0, fileutils_1.fileExists)(path);
}
exports.exists = exists;
/**
 * Get the size of a file on disk
 * @param filePath Path to the file
 */
function getSize(filePath) {
    return (0, fs_extra_1.statSync)(filePath).size;
}
exports.getSize = getSize;
//# sourceMappingURL=utils.js.map