"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findBuilderInMainJsTs = exports.resolveCommonStorybookOptionMapper = exports.runStorybookSetupCheck = exports.getStorybookFrameworkPath = void 0;
const devkit_1 = require("@nrwl/devkit");
const find_nodes_1 = require("@nrwl/workspace/src/utilities/typescript/find-nodes");
require("dotenv/config");
const fs_1 = require("fs");
const path_1 = require("path");
const semver_1 = require("semver");
const ts = require("typescript");
const utilities_1 = require("../utils/utilities");
function getStorybookFrameworkPath(uiFramework) {
    const serverOptionsPaths = {
        '@storybook/react': '@storybook/react/dist/cjs/server/options',
        '@storybook/html': '@storybook/html/dist/cjs/server/options',
        '@storybook/vue': '@storybook/vue/dist/cjs/server/options',
        '@storybook/vue3': '@storybook/vue3/dist/cjs/server/options',
        '@storybook/web-components': '@storybook/web-components/dist/cjs/server/options',
        '@storybook/svelte': '@storybook/svelte/dist/cjs/server/options',
    };
    if (isStorybookV62onwards(uiFramework)) {
        return serverOptionsPaths[uiFramework];
    }
    else {
        return `${uiFramework}/dist/server/options`;
    }
}
exports.getStorybookFrameworkPath = getStorybookFrameworkPath;
function isStorybookV62onwards(uiFramework) {
    const storybookPackageVersion = require((0, path_1.join)(uiFramework, 'package.json')).version;
    return (0, semver_1.gte)(storybookPackageVersion, '6.2.0-rc.4');
}
function runStorybookSetupCheck(options) {
    webpackFinalPropertyCheck(options);
    reactWebpack5Check(options);
}
exports.runStorybookSetupCheck = runStorybookSetupCheck;
function reactWebpack5Check(options) {
    if (options.uiFramework === '@storybook/react') {
        let storybookConfigFilePath = (0, devkit_1.joinPathFragments)(options.config.configFolder, 'main.js');
        if (!(0, fs_1.existsSync)(storybookConfigFilePath)) {
            storybookConfigFilePath = (0, devkit_1.joinPathFragments)(options.config.configFolder, 'main.ts');
        }
        if (!(0, fs_1.existsSync)(storybookConfigFilePath)) {
            // looks like there's no main config file, so skip
            return;
        }
        // check whether the current Storybook configuration has the webpack 5 builder enabled
        const storybookConfig = (0, fs_1.readFileSync)(storybookConfigFilePath, {
            encoding: 'utf8',
        });
        const source = ts.createSourceFile(storybookConfigFilePath, storybookConfig, ts.ScriptTarget.Latest, true);
        findBuilderInMainJsTs(source);
    }
}
function webpackFinalPropertyCheck(options) {
    let placesToCheck = [
        {
            path: (0, devkit_1.joinPathFragments)('.storybook', 'webpack.config.js'),
            result: false,
        },
        {
            path: (0, devkit_1.joinPathFragments)(options.config.configFolder, 'webpack.config.js'),
            result: false,
        },
    ];
    placesToCheck = placesToCheck
        .map((entry) => {
        return Object.assign(Object.assign({}, entry), { result: (0, fs_1.existsSync)(entry.path) });
    })
        .filter((x) => x.result === true);
    if (placesToCheck.length > 0) {
        devkit_1.logger.warn(`
  You have a webpack.config.js files in your Storybook configuration:
  ${placesToCheck.map((x) => `- "${x.path}"`).join('\n  ')}

  Consider switching to the "webpackFinal" property declared in "main.js" (or "main.ts") instead.
  ${options.uiFramework === '@storybook/react'
            ? 'https://nx.dev/storybook/migrate-webpack-final-react'
            : 'https://nx.dev/storybook/migrate-webpack-final-angular'}
    `);
    }
}
function resolveCommonStorybookOptionMapper(builderOptions, frameworkOptions, context) {
    const storybookConfig = (0, utilities_1.findOrCreateConfig)(builderOptions.config, context);
    const storybookOptions = Object.assign(Object.assign({ workspaceRoot: context.root, configDir: storybookConfig }, frameworkOptions), { frameworkPresets: [...(frameworkOptions.frameworkPresets || [])], watch: false });
    return storybookOptions;
}
exports.resolveCommonStorybookOptionMapper = resolveCommonStorybookOptionMapper;
function findBuilderInMainJsTs(storybookConfig) {
    const importArray = (0, find_nodes_1.findNodes)(storybookConfig, [
        ts.SyntaxKind.PropertyAssignment,
    ]);
    let builderIsSpecified = false;
    importArray.forEach((parent) => {
        var _a, _b;
        const identifier = (0, find_nodes_1.findNodes)(parent, ts.SyntaxKind.Identifier);
        const sbBuilder = (0, find_nodes_1.findNodes)(parent, ts.SyntaxKind.StringLiteral);
        const builderText = (_b = (_a = sbBuilder === null || sbBuilder === void 0 ? void 0 : sbBuilder[0]) === null || _a === void 0 ? void 0 : _a.getText()) !== null && _b !== void 0 ? _b : '';
        if (identifier[0].getText() === 'builder') {
            builderIsSpecified = true;
            if (builderText.includes('webpack') &&
                !builderText.includes('webpack5')) {
                builderIsSpecified = false;
            }
        }
    });
    if (!builderIsSpecified) {
        devkit_1.logger.warn(`
    It looks like you use Webpack 5 but your Storybook setup is not configured to leverage that
    and thus falls back to Webpack 4.
    Make sure you upgrade your Storybook config to use Webpack 5.
    
      - https://gist.github.com/shilman/8856ea1786dcd247139b47b270912324#upgrade
          
    `);
    }
}
exports.findBuilderInMainJsTs = findBuilderInMainJsTs;
//# sourceMappingURL=utils.js.map