/**
 * Copyright (c) 2021, WSO2 LLC. (https://www.wso2.com). All Rights Reserved.
 *
 * WSO2 LLC. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { IdentifiableComponentInterface } from "@wso2is/core/models";
import { FC, PropsWithChildren } from "react";
import { ButtonGroupProps, ButtonProps } from "semantic-ui-react";
interface StrictSwitcher {
    /**
     * You can specify maximum of 3 options and a minimum of 2 options.
     * Optionally you can edit the interface props to support your use-case.
     * However, you should consider making the button group `vertical` or
     * `compact` based on the space it takes.
     */
    options: [SwitcherOptionProps, SwitcherOptionProps, SwitcherOptionProps?];
    /**
     * This function callback will trigger each time when the value
     * changes in the toggle group.
     *
     * @param selectedValue - will return all the props you have passed
     * to this interface and the current state of the option itself.
     */
    onChange?: (selectedValue: SwitcherOptionProps) => void;
    /**
     * The initial selected value. This is different from default option
     * value. Where default option specify the sort order and the selected
     * value describes the initial selected value.
     */
    selectedValue?: string;
}
interface StrictSwitcherOption {
    /**
     * The item value for the switcher option.
     * Usually this should be unique for every option
     * in the switcher.
     */
    value: string;
    /**
     * The label of this option. This is usually a
     * readable string that can be rendered to the screen.
     */
    label: string;
    /**
     * Determines whether this switch option is disabled or not.
     * If theres only two options available and one of them are
     * disabled then the other value will be selected by default
     * if the switcher marked as required.
     */
    disabled?: boolean;
    /**
     * {@link onSelect} will be triggered when this button
     * state has been changed to active.
     *
     * @param self - switcher options
     */
    onSelect?: (self: StrictSwitcherOption) => void;
    /**
     * Disabled popup content.
     */
    disabledMessage?: string;
}
/**
 * Switcher props intersected with {@link ButtonGroupProps}
 * to enable more configuration options to the exposing
 * interface.
 */
export declare type SwitcherProps = PropsWithChildren<StrictSwitcher & ButtonGroupProps & IdentifiableComponentInterface>;
/**
 * Example on extended ButtonProps
 * ===============================
 *
 * Optionally specify an icon to give the individual
 * selection to give more accessibility. For available
 * icons refer: {@link https://react.semantic-ui.com/elements/icon/}
 *
 * This can be done through ...rest props as well. However,
 * this property is explicitly added to this interface to
 * give more clarity. Same goes to {@link color} as well.
 *
 * Example: `{ ... icon: 'file' }`
 *
 * Optionally specify the color to each of the {@link StrictSwitcherOption}
 * to distinguish from other options. For a color palette refer:
 * {@link https://react.semantic-ui.com/elements/button/#types-basic-shorthand}
 *
 * Example: `{ ... color: 'grey' }`
 *
 * Optionally specify the label position of the labeled
 * icon button. It can be either left or right depending on the
 * option. By default {@link Switcher} has enabled the
 * {@link StrictButtonGroupProps.labeled} property to true.
 *
 * Example: `{ ... labelPosition?: 'right' | 'left' }`
 */
export declare type SwitcherOptionProps = StrictSwitcherOption & ButtonProps & IdentifiableComponentInterface;
/**
 * This is a dynamic switcher component that offers below
 * functionality:-
 *
 *  1) Given the options, it can render a grouped set of
 *     buttons that is selectable.
 *
 *  2) A {@link StrictSwitcher.defaultOptionValue} value
 *     can be set to enforce the matching value to be
 *     selected initially. If the defaultOptionValue is
 *     unavailable the first element will be the default.
 *
 *  3) Support three switch options. Optionally this can be
 *     adjusted if the use-case requires more options. Ideally
 *     it should be little because its a group and overflows
 *     will not be handle by this component at this time.
 *
 *  4) Fire a initial event for the default value. This will
 *     be triggered when provided/not-provided {@link defaultOptionValue}
 *     nonetheless.
 *
 * How can I use this interface? As a example refer below block.
 *
 * ```
 *  <Switcher
 *      defaultOptionValue="option2"
 *      onChange={ (selectedOption: SwitcherOptionProps) => {
 *          // Will be triggered initially and on value change.
 *          console.log(selectedOption.value);
 *      } }
 *      options=\{ [
 *          {
 *              value: "option1",
 *              label: "My option 1",
 *              icon: "cogs"
 *          },
 *          {
 *              value: "option2",
 *              label: "My option 2",
 *              icon: "file"
 *          }
 *      ] }
 *  />
 *```
 * @param props - props injected to the component
 */
export declare const Switcher: FC<SwitcherProps>;
export {};
