"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.conversionSchematic = exports.conversionGenerator = void 0;
const tslib_1 = require("tslib");
const devkit_1 = require("@nrwl/devkit");
const linter_1 = require("@nrwl/linter");
const cypress_project_1 = require("../cypress-project/cypress-project");
function conversionGenerator(host, options) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        /**
         * The ProjectConverter instance encapsulates all the standard operations we need
         * to perform in order to convert a project from TSLint to ESLint, as well as some
         * extensibility points for adjusting the behavior on a per package basis.
         *
         * E.g. @nrwl/angular projects might need to make different changes to the final
         * ESLint config when compared with @nrwl/next projects.
         *
         * See the ProjectConverter implementation for a full breakdown of what it does.
         */
        const projectConverter = new linter_1.ProjectConverter({
            host,
            projectName: options.project,
            ignoreExistingTslintConfig: options.ignoreExistingTslintConfig,
            eslintInitializer: ({ projectName, projectConfig }) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                yield (0, cypress_project_1.addLinter)(host, {
                    linter: 'eslint',
                    projectName,
                    projectRoot: projectConfig.root,
                    /**
                     * We set the parserOptions.project config just in case the converted config uses
                     * rules which require type-checking. Later in the conversion we check if it actually
                     * does and remove the config again if it doesn't, so that it is most efficient.
                     */
                    setParserOptionsProject: true,
                    skipFormat: options.skipFormat,
                });
            }),
        });
        /**
         * If root eslint configuration already exists it will not be recreated
         * but we also don't want to re-run the tslint config conversion
         * as it was likely already done
         */
        const rootEslintConfigExists = host.exists('.eslintrc.json');
        /**
         * Create the standard (which is applicable to the current package) ESLint setup
         * for converting the project.
         */
        const eslintInitInstallTask = yield projectConverter.initESLint();
        /**
         * Convert the root tslint.json and apply the converted rules to the root .eslintrc.json.
         */
        const rootConfigInstallTask = yield projectConverter.convertRootTSLintConfig((json) => removeCodelyzerRelatedRules(json), rootEslintConfigExists);
        /**
         * Convert the project's tslint.json to an equivalent ESLint config.
         */
        const projectConfigInstallTask = yield projectConverter.convertProjectConfig((json) => json);
        /**
         * Clean up the original TSLint configuration for the project.
         */
        projectConverter.removeProjectTSLintFile();
        // Only project shouldn't be added as a default
        const { project } = options, defaults = tslib_1.__rest(options, ["project"]);
        /**
         * Store user preferences for the collection
         */
        projectConverter.setDefaults('@nrwl/cypress', defaults);
        /**
         * Based on user preference and remaining usage, remove TSLint from the workspace entirely.
         */
        let uninstallTSLintTask = () => Promise.resolve(undefined);
        if (options.removeTSLintIfNoMoreTSLintTargets &&
            !projectConverter.isTSLintUsedInWorkspace()) {
            uninstallTSLintTask = projectConverter.removeTSLintFromWorkspace();
        }
        if (!options.skipFormat) {
            yield (0, devkit_1.formatFiles)(host);
        }
        return () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield eslintInitInstallTask();
            yield rootConfigInstallTask();
            yield projectConfigInstallTask();
            yield uninstallTSLintTask();
        });
    });
}
exports.conversionGenerator = conversionGenerator;
exports.conversionSchematic = (0, devkit_1.convertNxGenerator)(conversionGenerator);
/**
 * Remove any @angular-eslint rules that were applied as a result of converting prior codelyzer
 * rules, because they are only relevant for Angular projects.
 */
function removeCodelyzerRelatedRules(json) {
    for (const ruleName of Object.keys(json.rules)) {
        if (ruleName.startsWith('@angular-eslint')) {
            delete json.rules[ruleName];
        }
    }
    if (json.plugins) {
        json.plugins = json.plugins.filter((plugin) => !plugin.startsWith('@angular-eslint'));
    }
    return json;
}
//# sourceMappingURL=convert-tslint-to-eslint.js.map