function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

function _objectWithoutPropertiesLoose(source, excluded) { if (source == null) return {}; var target = {}; var sourceKeys = Object.keys(source); var key, i; for (i = 0; i < sourceKeys.length; i++) { key = sourceKeys[i]; if (excluded.indexOf(key) >= 0) continue; target[key] = source[key]; } return target; }

import React, { useCallback, useEffect, useMemo, useState } from 'react';
import { HexColorPicker, HslaStringColorPicker, RgbaStringColorPicker } from 'react-colorful';
import convert from 'color-convert';
import throttle from 'lodash/throttle';
import { styled } from '@storybook/theming';
import { TooltipNote } from '../tooltip/TooltipNote';
import { WithTooltip } from '../tooltip/lazy-WithTooltip';
import { Form } from '../form';
import { Icons } from '../icon/icon';
import { getControlId } from './helpers';
const Wrapper = styled.div({
  position: 'relative',
  maxWidth: 250
});
const PickerTooltip = styled(WithTooltip)({
  position: 'absolute',
  zIndex: 1,
  top: 4,
  left: 4
});
const TooltipContent = styled.div({
  width: 200,
  margin: 5,
  '.react-colorful__saturation': {
    borderRadius: '4px 4px 0 0'
  },
  '.react-colorful__hue': {
    boxShadow: 'inset 0 0 0 1px rgb(0 0 0 / 5%)'
  },
  '.react-colorful__last-control': {
    borderRadius: '0 0 4px 4px'
  }
});
const Note = styled(TooltipNote)(({
  theme
}) => ({
  fontFamily: theme.typography.fonts.base
}));
const Swatches = styled.div({
  display: 'grid',
  gridTemplateColumns: 'repeat(9, 16px)',
  gap: 6,
  padding: 3,
  marginTop: 5,
  width: 200
});
const SwatchColor = styled.div(({
  theme,
  active
}) => ({
  width: 16,
  height: 16,
  boxShadow: active ? `${theme.appBorderColor} 0 0 0 1px inset, ${theme.color.mediumdark}50 0 0 0 4px` : `${theme.appBorderColor} 0 0 0 1px inset`,
  borderRadius: theme.appBorderRadius
}));
const swatchBackground = `url('data:image/svg+xml;charset=utf-8,<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill-opacity=".05"><path d="M8 0h8v8H8zM0 8h8v8H0z"/></svg>')`;

const Swatch = (_ref) => {
  let {
    value,
    active,
    onClick,
    style
  } = _ref,
      props = _objectWithoutPropertiesLoose(_ref, ["value", "active", "onClick", "style"]);

  const backgroundImage = `linear-gradient(${value}, ${value}), ${swatchBackground}, linear-gradient(#fff, #fff)`;
  return /*#__PURE__*/React.createElement(SwatchColor, _extends({}, props, {
    active,
    onClick,
    style: Object.assign({}, style, {
      backgroundImage
    })
  }));
};

Swatch.displayName = "Swatch";
const Input = styled(Form.Input)(({
  theme
}) => ({
  width: '100%',
  paddingLeft: 30,
  paddingRight: 30,
  boxSizing: 'border-box',
  fontFamily: theme.typography.fonts.base
}));
const ToggleIcon = styled(Icons)(({
  theme
}) => ({
  position: 'absolute',
  zIndex: 1,
  top: 6,
  right: 7,
  width: 20,
  height: 20,
  padding: 4,
  boxSizing: 'border-box',
  cursor: 'pointer',
  color: theme.input.color
}));
var ColorSpace;

(function (ColorSpace) {
  ColorSpace["RGB"] = "rgb";
  ColorSpace["HSL"] = "hsl";
  ColorSpace["HEX"] = "hex";
})(ColorSpace || (ColorSpace = {}));

const COLOR_SPACES = Object.values(ColorSpace);
const COLOR_REGEXP = /\(([0-9]+),\s*([0-9]+)%?,\s*([0-9]+)%?,?\s*([0-9.]+)?\)/;
const RGB_REGEXP = /^\s*rgba?\(([0-9]+),\s*([0-9]+),\s*([0-9]+),?\s*([0-9.]+)?\)\s*$/i;
const HSL_REGEXP = /^\s*hsla?\(([0-9]+),\s*([0-9]+)%,\s*([0-9]+)%,?\s*([0-9.]+)?\)\s*$/i;
const HEX_REGEXP = /^\s*#?([0-9a-f]{3}|[0-9a-f]{6})\s*$/i;
const SHORTHEX_REGEXP = /^\s*#?([0-9a-f]{3})\s*$/i;
const ColorPicker = {
  [ColorSpace.HEX]: HexColorPicker,
  [ColorSpace.RGB]: RgbaStringColorPicker,
  [ColorSpace.HSL]: HslaStringColorPicker
};
const fallbackColor = {
  [ColorSpace.HEX]: 'transparent',
  [ColorSpace.RGB]: 'rgba(0, 0, 0, 0)',
  [ColorSpace.HSL]: 'hsla(0, 0%, 0%, 0)'
};

const stringToArgs = value => {
  const match = value === null || value === void 0 ? void 0 : value.match(COLOR_REGEXP);
  if (!match) return [0, 0, 0, 1];
  const [, x, y, z, a = 1] = match;
  return [x, y, z, a].map(Number);
};

const parseValue = value => {
  if (!value) return undefined;
  let valid = true;

  if (RGB_REGEXP.test(value)) {
    const [r, g, b, a] = stringToArgs(value);
    const [h, s, l] = convert.rgb.hsl([r, g, b]) || [0, 0, 0];
    return {
      valid,
      value,
      keyword: convert.rgb.keyword([r, g, b]),
      colorSpace: ColorSpace.RGB,
      [ColorSpace.RGB]: value,
      [ColorSpace.HSL]: `hsla(${h}, ${s}%, ${l}%, ${a})`,
      [ColorSpace.HEX]: `#${convert.rgb.hex([r, g, b]).toLowerCase()}`
    };
  }

  if (HSL_REGEXP.test(value)) {
    const [h, s, l, a] = stringToArgs(value);
    const [r, g, b] = convert.hsl.rgb([h, s, l]) || [0, 0, 0];
    return {
      valid,
      value,
      keyword: convert.hsl.keyword([h, s, l]),
      colorSpace: ColorSpace.HSL,
      [ColorSpace.RGB]: `rgba(${r}, ${g}, ${b}, ${a})`,
      [ColorSpace.HSL]: value,
      [ColorSpace.HEX]: `#${convert.hsl.hex([h, s, l]).toLowerCase()}`
    };
  }

  const plain = value.replace('#', '');
  const rgb = convert.keyword.rgb(plain) || convert.hex.rgb(plain);
  const hsl = convert.rgb.hsl(rgb);
  let mapped = value;
  if (/[^#a-f0-9]/i.test(value)) mapped = plain;else if (HEX_REGEXP.test(value)) mapped = `#${plain}`;

  if (mapped.startsWith('#')) {
    valid = HEX_REGEXP.test(mapped);
  } else {
    try {
      convert.keyword.hex(mapped);
    } catch (e) {
      valid = false;
    }
  }

  return {
    valid,
    value: mapped,
    keyword: convert.rgb.keyword(rgb),
    colorSpace: ColorSpace.HEX,
    [ColorSpace.RGB]: `rgba(${rgb[0]}, ${rgb[1]}, ${rgb[2]}, 1)`,
    [ColorSpace.HSL]: `hsla(${hsl[0]}, ${hsl[1]}%, ${hsl[2]}%, 1)`,
    [ColorSpace.HEX]: mapped
  };
};

const getRealValue = (value, color, colorSpace) => {
  if (!value || !(color !== null && color !== void 0 && color.valid)) return fallbackColor[colorSpace];
  if (colorSpace !== ColorSpace.HEX) return (color === null || color === void 0 ? void 0 : color[colorSpace]) || fallbackColor[colorSpace];

  if (!color.hex.startsWith('#')) {
    try {
      return `#${convert.keyword.hex(color.hex)}`;
    } catch (e) {
      return fallbackColor.hex;
    }
  }

  const short = color.hex.match(SHORTHEX_REGEXP);
  if (!short) return HEX_REGEXP.test(color.hex) ? color.hex : fallbackColor.hex;
  const [r, g, b] = short[1].split('');
  return `#${r}${r}${g}${g}${b}${b}`;
};

const useColorInput = (initialValue, onChange) => {
  const [value, setValue] = useState(initialValue || '');
  const [color, setColor] = useState(() => parseValue(value));
  const [colorSpace, setColorSpace] = useState((color === null || color === void 0 ? void 0 : color.colorSpace) || ColorSpace.HEX); // Reset state when initialValue becomes undefined (when resetting controls)

  useEffect(() => {
    if (initialValue !== undefined) return;
    setValue('');
    setColor(undefined);
    setColorSpace(ColorSpace.HEX);
  }, [initialValue]);
  const realValue = useMemo(() => getRealValue(value, color, colorSpace).toLowerCase(), [value, color, colorSpace]);
  const updateValue = useCallback(update => {
    const parsed = parseValue(update);
    setValue((parsed === null || parsed === void 0 ? void 0 : parsed.value) || update || '');
    if (!parsed) return;
    setColor(parsed);
    setColorSpace(parsed.colorSpace);
    onChange(parsed.value);
  }, [onChange]);
  const cycleColorSpace = useCallback(() => {
    let next = COLOR_SPACES.indexOf(colorSpace) + 1;
    if (next >= COLOR_SPACES.length) next = 0;
    setColorSpace(COLOR_SPACES[next]);
    const update = (color === null || color === void 0 ? void 0 : color[COLOR_SPACES[next]]) || '';
    setValue(update);
    onChange(update);
  }, [color, colorSpace, onChange]);
  return {
    value,
    realValue,
    updateValue,
    color,
    colorSpace,
    cycleColorSpace
  };
};

const id = value => value.replace(/\s*/, '').toLowerCase();

const usePresets = (presetColors, currentColor, colorSpace) => {
  const [selectedColors, setSelectedColors] = useState(currentColor !== null && currentColor !== void 0 && currentColor.valid ? [currentColor] : []); // Reset state when currentColor becomes undefined (when resetting controls)

  useEffect(() => {
    if (currentColor !== undefined) return;
    setSelectedColors([]);
  }, [currentColor]);
  const presets = useMemo(() => {
    const initialPresets = (presetColors || []).map(preset => {
      if (typeof preset === 'string') return parseValue(preset);
      if (preset.title) return Object.assign({}, parseValue(preset.color), {
        keyword: preset.title
      });
      return parseValue(preset.color);
    });
    return initialPresets.concat(selectedColors).filter(Boolean).slice(-27);
  }, [presetColors, selectedColors]);
  const addPreset = useCallback(color => {
    if (!(color !== null && color !== void 0 && color.valid)) return;
    if (presets.some(preset => id(preset[colorSpace]) === id(color[colorSpace]))) return;
    setSelectedColors(arr => arr.concat(color));
  }, [colorSpace, presets]);
  return {
    presets,
    addPreset
  };
};

export const ColorControl = ({
  name,
  value: initialValue,
  onChange,
  onFocus,
  onBlur,
  presetColors,
  startOpen
}) => {
  const {
    value,
    realValue,
    updateValue,
    color,
    colorSpace,
    cycleColorSpace
  } = useColorInput(initialValue, throttle(onChange, 200));
  const {
    presets,
    addPreset
  } = usePresets(presetColors, color, colorSpace);
  const Picker = ColorPicker[colorSpace];
  return /*#__PURE__*/React.createElement(Wrapper, null, /*#__PURE__*/React.createElement(PickerTooltip, {
    trigger: "click",
    startOpen: startOpen,
    closeOnClick: true,
    onVisibilityChange: () => addPreset(color),
    tooltip: /*#__PURE__*/React.createElement(TooltipContent, null, /*#__PURE__*/React.createElement(Picker, {
      color: realValue === 'transparent' ? '#000000' : realValue,
      onChange: updateValue,
      onFocus,
      onBlur
    }), presets.length > 0 && /*#__PURE__*/React.createElement(Swatches, null, presets.map((preset, index) => /*#__PURE__*/React.createElement(WithTooltip // eslint-disable-next-line react/no-array-index-key
    , {
      key: `${preset.value}-${index}`,
      hasChrome: false,
      tooltip: /*#__PURE__*/React.createElement(Note, {
        note: preset.keyword || preset.value
      })
    }, /*#__PURE__*/React.createElement(Swatch, {
      value: preset[colorSpace],
      active: color && id(preset[colorSpace]) === id(color[colorSpace]),
      onClick: () => updateValue(preset.value)
    })))))
  }, /*#__PURE__*/React.createElement(Swatch, {
    value: realValue,
    style: {
      margin: 4
    }
  })), /*#__PURE__*/React.createElement(Input, {
    id: getControlId(name),
    value: value,
    onChange: e => updateValue(e.target.value),
    onFocus: e => e.target.select(),
    placeholder: "Choose color..."
  }), value ? /*#__PURE__*/React.createElement(ToggleIcon, {
    icon: "markup",
    onClick: cycleColorSpace
  }) : null);
};
ColorControl.displayName = "ColorControl";
export default ColorControl;