/**
 * Copyright (c) 2020, WSO2 LLC. (https://www.wso2.com). All Rights Reserved.
 *
 * WSO2 LLC. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { AnnouncementBannerInterface, ProductReleaseTypes } from "../models";
/**
 * Class containing common utility methods used across application.
 */
export declare class CommonUtils {
    /**
     * Private constructor to avoid object instantiation from outside
     * the class.
     */
    private constructor();
    /**
     * A util method to humanize the last modified date.
     *
     * @param date - Date string which needs to be humanize
     */
    static humanizeDateDifference: (date: string) => string;
    /**
     * Parses the product version.
     * @example
     * // returns [ version = "5.11.0", release = "m23", type = "milestone" ]
     * const [ version, release, type ] = CommonUtils.parseProductVersion("5.11.0-m23-SNAPSHOT");
     *
     * @param version- Raw version in the form of `5.11.0-m23-SNAPSHOT`.
     * @param allowSnapshot - Show the SNAPSHOT label.
     * @returns Parsed product version.
     */
    static parseProductVersion(version: string, allowSnapshot?: boolean): [string, string, ProductReleaseTypes];
    /**
     * Iterates through the announcements array and gets the valid announcement to be displayed.
     *
     * @param announcements - Array of announcements.
     * @param seen - Set of seen announcements.
     * @returns Valid announcement.
     */
    static getValidAnnouncement(announcements: AnnouncementBannerInterface[], seen: string[]): AnnouncementBannerInterface;
    /**
     * Get user image from gravatar.com.
     *
     * @param emailAddress - email address received authenticated user.
     * @returns Gravatar image path.
     */
    static getGravatar(emailAddress: string): string;
    /**
     * Reloads the current document.
     */
    static refreshPage(): void;
    /**
     * Scroll page to a specific target element.
     *
     * @param element - target element.
     * @param offset - scroll stop offset value.
     */
    static scrollToTarget(element: Element, offset?: number): void;
    /**
     * Copy to clipboard with fallback.
     *
     * @param text - Text to copy.
     * @returns A promise of type boolean.
     */
    static copyTextToClipboard(text: string): Promise<boolean>;
    /**
     * Get the list of countries to be added to the country dropdown input field.
     *
     * @returns List of country objects.
     */
    static getCountryList(): {
        flag: string;
        key: number;
        text: string;
        value: string;
    }[];
    /**
     * Returns boolean value for both boolean strings and primitive boolean value.
     *
     * @param booleanResult - Result that needs to be converted to boolean.
     * Can be a string or a boolean value.
     * @returns Parsed boolean value.
     */
    static parseBoolean(booleanResult: string | unknown): boolean;
}
