/**
 * Copyright (c) 2020, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { AsgardeoSPAClient, AuthClientConfig, BasicUserInfo, Config, DecodedIDTokenPayload, FetchResponse, Hooks, HttpClientInstance, HttpRequestConfig, HttpResponse, OIDCEndpoints, SignInConfig } from "@asgardeo/auth-spa";
import { SPACustomGrantConfig } from "@asgardeo/auth-spa/src/models/request-custom-grant";
import { AuthStateInterface } from "./models";
declare class AuthAPI {
    static DEFAULT_STATE: AuthStateInterface;
    private _authState;
    private _client;
    constructor(spaClient?: AsgardeoSPAClient);
    /**
     * Method to return Auth Client instance authentication state.
     *
     * @return {AuthStateInterface} Authentication State.
     */
    getState(): AuthStateInterface;
    /**
     * Method to initialize the AuthClient instance.
     *
     * @param {Config} config - `dispatch` function from React Auth Context.
     */
    init(config: AuthClientConfig<Config>): Promise<boolean>;
    /**
     * Method to handle user Sign In requests.
     *
     * @param {any} dispatch - `dispatch` function from React Auth Context.
     * @param {AuthStateInterface} state - Current authentication state in React Auth Context.
     * @param {any} callback - Action to trigger on successful sign in.
     */
    signIn(dispatch: (state: AuthStateInterface) => void, state: AuthStateInterface, config: SignInConfig, authorizationCode: string, sessionState: string, authState?: string, callback?: (response: BasicUserInfo) => void): Promise<BasicUserInfo>;
    /**
     * Method to handle user Sign Out requests.
     *
     * @param {any} dispatch - `dispatch` function from React Auth Context.
     * @param {AuthStateInterface} state - Current authentication state in React Auth Context.
     * @param {any} callback - Action to trigger on successful sign out.
     */
    signOut(dispatch: (state: AuthStateInterface) => void, state: AuthStateInterface, callback?: (response?: boolean) => void): Promise<boolean>;
    /**
     * Method to update Auth Client instance authentication state.
     *
     * @param {AuthStateInterface} state - State values to update in authentication state.
     */
    updateState(state: AuthStateInterface): void;
    /**
     * This method returns a Promise that resolves with the basic user information obtained from the ID token.
     *
     * @return {Promise<BasicUserInfo>} - A promise that resolves with the user information.
     */
    getBasicUserInfo(): Promise<BasicUserInfo>;
    /**
     * This method sends an API request to a protected endpoint.
     * The access token is automatically attached to the header of the request.
     * This is the only way by which protected endpoints can be accessed
     * when the web worker is used to store session information.
     *
     * @param {HttpRequestConfig} config -  The config object containing attributes necessary to send a request.
     *
     * @return {Promise<FetchResponse>} - Returns a Promise that resolves with the response to the request.
     */
    httpRequest(config: HttpRequestConfig): Promise<HttpResponse<any>>;
    /**
     * This method sends multiple API requests to a protected endpoint.
     * The access token is automatically attached to the header of the request.
     * This is the only way by which multiple requests can be sent to protected endpoints
     * when the web worker is used to store session information.
     *
     * @param {HttpRequestConfig[]} config -  The config object containing attributes necessary to send a request.
     *
     * @return {Promise<FetchResponse[]>} - Returns a Promise that resolves with the responses to the requests.
     */
    httpRequestAll(configs: HttpRequestConfig[]): Promise<HttpResponse<any>[]>;
    /**
     * This method allows you to send a request with a custom grant.
     *
     * @param {CustomGrantRequestParams} config - The request parameters.
     *
     * @return {Promise<FetchResponse<any> | SignInResponse>} - A Promise that resolves with
     * the value returned by the custom grant request.
     */
    requestCustomGrant(config: SPACustomGrantConfig, callback: (response: BasicUserInfo | FetchResponse<any>) => void, dispatch: (state: AuthStateInterface) => void): Promise<BasicUserInfo | FetchResponse<any>>;
    /**
     * This method ends a user session. The access token is revoked and the session information is destroyed.
     *
     * @return {Promise<boolean>} - A promise that resolves with `true` if the process is successful.
     */
    revokeAccessToken(dispatch: (state: AuthStateInterface) => void): Promise<boolean>;
    /**
     * This method returns a Promise that resolves with an object containing the service endpoints.
     *
     * @return {Promise<ServiceResourcesType} - A Promise that resolves with an object containing the service endpoints.
     */
    getOIDCServiceEndpoints(): Promise<OIDCEndpoints>;
    /**
     * This methods returns the Axios http client.
     *
     * @return {HttpClientInstance} - The Axios HTTP client.
     */
    getHttpClient(): Promise<HttpClientInstance>;
    /**
     * This method decodes the payload of the id token and returns it.
     *
     * @return {Promise<DecodedIDTokenPayloadInterface>} - A Promise that resolves with
     * the decoded payload of the id token.
     */
    getDecodedIDToken(): Promise<DecodedIDTokenPayload>;
    /**
     * This method decodes the payload of the idp id token and returns it.
     *
     * @return {Promise<DecodedIDTokenPayloadInterface>} - A Promise that resolves with
     * the decoded payload of the idp id token.
     */
    getDecodedIDPIDToken(): Promise<DecodedIDTokenPayload>;
    /**
     * This method returns the ID token.
     *
     * @return {Promise<string>} - A Promise that resolves with the id token.
     */
    getIDToken(): Promise<string>;
    /**
     * This method return a Promise that resolves with the access token.
     *
     * **This method will not return the access token if the storage type is set to `webWorker`.**
     *
     * @return {Promise<string>} - A Promise that resolves with the access token.
     */
    getAccessToken(): Promise<string>;
    /**
     * This method return a Promise that resolves with the idp access token.
     *
     * **This method will not return the idp access token if the storage type is set to `webWorker`.**
     * **This can be used to access the IDP access token when custom auth grant functionalities are used**
     *
     * @return {Promise<string>} - A Promise that resolves with the idp access token.
     */
    getIDPAccessToken(): Promise<string>;
    /**
     * This method refreshes the access token.
     *
     * @return {TokenResponseInterface} - A Promise that resolves with an object containing
     * information about the refreshed access token.
     */
    refreshAccessToken(): Promise<BasicUserInfo>;
    /**
     * This method specifies if the user is authenticated or not.
     *
     * @return {Promise<boolean>} - A Promise that resolves with `true` if teh user is authenticated.
     */
    isAuthenticated(): Promise<boolean>;
    /**
     * This method enables callback functions attached to the http client.
     *
     * @return {Promise<boolean>} - A promise that resolves with True.
     *
     */
    enableHttpHandler(): Promise<boolean>;
    /**
     * This method disables callback functions attached to the http client.
     *
     * @return {Promise<boolean>} - A promise that resolves with True.
     */
    disableHttpHandler(): Promise<boolean>;
    /**
     * This method updates the configuration that was passed into the constructor when instantiating this class.
     *
     * @param {Partial<AuthClientConfig<T>>} config - A config object to update the SDK configurations with.
     */
    updateConfig(config: Partial<AuthClientConfig<Config>>): Promise<void>;
    /**
     * This method attaches a callback function to an event hook that fires the callback when the event happens.
     *
     * @param {Hooks.CustomGrant} hook - The name of the hook.
     * @param {(response?: any) => void} callback - The callback function.
     * @param {string} id (optional) - The id of the hook. This is used when multiple custom grants are used.
     *
     */
    on(hook: Hooks.CustomGrant, callback: (response?: any) => void, id: string): Promise<void>;
    on(hook: Exclude<Hooks, Hooks.CustomGrant>, callback: (response?: any) => void): Promise<void>;
    /**
     * This method allows you to sign in silently.
     * First, this method sends a prompt none request to see if there is an active user session in the identity server.
     * If there is one, then it requests the access token and stores it. Else, it returns false.
     *
     * @return {Promise<BasicUserInfo | boolean>} - A Promise that resolves with the user information after signing in
     * or with `false` if the user is not signed in.
     *
     * @example
     *```
     * client.trySignInSilently()
     *```
     */
    trySignInSilently(state: AuthStateInterface, dispatch: (state: AuthStateInterface) => void): Promise<BasicUserInfo | boolean | undefined>;
}
export default AuthAPI;
//# sourceMappingURL=api.d.ts.map