/**
 * Copyright (c) 2019, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 */
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var HttpClient_1;
import axios from "axios";
import { staticDecorator } from "../helpers";
/**
 * An Http Http client to perform Http requests.
 *
 * @remarks
 * Typescript doesn't support static functions in interfaces. Therefore,
 * a decorator i.e `staticDecorator` was written to add static support.
 * Follow {@link https://github.com/Microsoft/TypeScript/issues/13462}
 * for more info.
 *
 * @example
 * Example usage.
 * ```
 *
 * const httpClient = HttpClient.getInstance();
 * httpClient.init(true, onRequestStart, onRequestSuccess, onRequestError, onRequestFinish);
 * ```
 */
let HttpClient = HttpClient_1 = class HttpClient {
    /**
     * Private constructor to avoid object instantiation from outside
     * the class.
     *
     * @hideconstructor
     */
    constructor() {
        this.attachToken = () => Promise.resolve();
        this.requestStartCallback = () => null;
        this.requestSuccessCallback = () => null;
        this.requestErrorCallback = () => null;
        this.requestFinishCallback = () => null;
        this.init = this.init.bind(this);
        this.setHttpRequestErrorCallback = this.setHttpRequestErrorCallback.bind(this);
        this.setHttpRequestFinishCallback = this.setHttpRequestFinishCallback.bind(this);
        this.setHttpRequestStartCallback = this.setHttpRequestStartCallback.bind(this);
        this.setHttpRequestSuccessCallback = this.setHttpRequestSuccessCallback.bind(this);
    }
    /**
     * Returns an aggregated instance of type `HttpInstance` of `HttpClient`.
     *
     * @return {any}
     */
    static getInstance() {
        if (this.axiosInstance) {
            return this.axiosInstance;
        }
        this.axiosInstance = axios.create({
            withCredentials: true
        });
        if (!this.clientInstance) {
            this.clientInstance = new HttpClient_1();
        }
        // Register request interceptor
        this.axiosInstance.interceptors.request.use((request) => __awaiter(this, void 0, void 0, function* () { return yield this.clientInstance.requestHandler(request); }));
        // Register response interceptor
        this.axiosInstance.interceptors.response.use((response) => this.clientInstance.successHandler(response), (error) => this.clientInstance.errorHandler(error));
        // Add the missing helper methods from axios
        this.axiosInstance.all = axios.all;
        this.axiosInstance.spread = axios.spread;
        // Add the init method from the `HttpClient` instance.
        this.axiosInstance.init = this.clientInstance.init;
        // Add the handler enabling & disabling methods to the instance.
        this.axiosInstance.enableHandler = this.clientInstance.enableHandler;
        this.axiosInstance.disableHandler = this.clientInstance.disableHandler;
        this.axiosInstance.disableHandlerWithTimeout = this.clientInstance.disableHandlerWithTimeout;
        this.axiosInstance.setHttpRequestStartCallback = this.clientInstance.setHttpRequestStartCallback;
        this.axiosInstance.setHttpRequestSuccessCallback = this.clientInstance.setHttpRequestSuccessCallback;
        this.axiosInstance.setHttpRequestErrorCallback = this.clientInstance.setHttpRequestErrorCallback;
        this.axiosInstance.setHttpRequestFinishCallback = this.clientInstance.setHttpRequestFinishCallback;
        return this.axiosInstance;
    }
    /**
     * Intercepts all the requests.
     * If the `isHandlerEnabled` flag is set to true, fires the `requestStartCallback`
     * and retrieves the access token from the server and attaches it to the request.
     * Else, just returns the original request.
     *
     * @param {HttpRequestConfig} request - Original request.
     * @return {HttpRequestConfig}
     */
    requestHandler(request) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.attachToken(request);
            if (request === null || request === void 0 ? void 0 : request.shouldEncodeToFormData) {
                const data = request === null || request === void 0 ? void 0 : request.data;
                const formData = new FormData();
                Object.keys(data).forEach((key) => {
                    formData.append(key, data[key]);
                });
                request.data = formData;
            }
            request.startTimeInMs = new Date().getTime();
            if (HttpClient_1.isHandlerEnabled) {
                if (this.requestStartCallback && typeof this.requestStartCallback === "function") {
                    this.requestStartCallback(request);
                }
            }
            return request;
        });
    }
    /**
     * Handles response errors.
     * If the `isHandlerEnabled` flag is set to true, fires the `requestErrorCallback`
     * and the `requestFinishCallback` functions. Else, just returns the original error.
     *
     * @param {HttpError} error - Original error.
     * @return {HttpError}
     */
    errorHandler(error) {
        if (HttpClient_1.isHandlerEnabled) {
            if (this.requestErrorCallback && typeof this.requestErrorCallback === "function") {
                this.requestErrorCallback(error);
            }
            if (this.requestFinishCallback && typeof this.requestFinishCallback === "function") {
                this.requestFinishCallback();
            }
        }
        throw error;
    }
    /**
     * Handles response success.
     * If the `isHandlerEnabled` flag is set to true, fires the `requestSuccessCallback`
     * and the `requestFinishCallback` functions. Else, just returns the original response.
     *
     * @param {HttpResponse} response - Original response.
     * @return {HttpResponse}
     */
    successHandler(response) {
        if (HttpClient_1.isHandlerEnabled) {
            if (this.requestSuccessCallback && typeof this.requestSuccessCallback === "function") {
                this.requestSuccessCallback(response);
            }
            if (this.requestFinishCallback && typeof this.requestFinishCallback === "function") {
                this.requestFinishCallback();
            }
        }
        return response;
    }
    /**
     * Initializes the Http client.
     *
     * @param isHandlerEnabled - Flag to toggle handler enablement.
     * @param requestStartCallback - Callback function to be triggered on request start.
     * @param requestSuccessCallback - Callback function to be triggered on request success.
     * @param requestErrorCallback - Callback function to be triggered on request error.
     * @param requestFinishCallback - Callback function to be triggered on request error.
     */
    init(isHandlerEnabled = true, attachToken) {
        return __awaiter(this, void 0, void 0, function* () {
            HttpClient_1.isHandlerEnabled = isHandlerEnabled;
            this.attachToken = attachToken;
        });
    }
    /**
     * Enables the handler.
     */
    enableHandler() {
        HttpClient_1.isHandlerEnabled = true;
    }
    /**
     * Disables the handler.
     */
    disableHandler() {
        HttpClient_1.isHandlerEnabled = false;
    }
    /**
     * Disables the handler for a given period of time.
     *
     * @param {number} timeout - Timeout in milliseconds.
     */
    disableHandlerWithTimeout(timeout = HttpClient_1.DEFAULT_HANDLER_DISABLE_TIMEOUT) {
        HttpClient_1.isHandlerEnabled = false;
        setTimeout(() => {
            HttpClient_1.isHandlerEnabled = true;
        }, timeout);
    }
    setHttpRequestStartCallback(callback) {
        this.requestStartCallback = callback;
    }
    setHttpRequestSuccessCallback(callback) {
        this.requestSuccessCallback = callback;
    }
    setHttpRequestErrorCallback(callback) {
        this.requestErrorCallback = callback;
    }
    setHttpRequestFinishCallback(callback) {
        this.requestFinishCallback = callback;
    }
};
HttpClient.DEFAULT_HANDLER_DISABLE_TIMEOUT = 1000;
HttpClient = HttpClient_1 = __decorate([
    staticDecorator()
], HttpClient);
export { HttpClient };
//# sourceMappingURL=axios-http-client.js.map