"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getEsLintOptions = exports.addMigrationJsonChecks = void 0;
const tslib_1 = require("tslib");
const devkit_1 = require("@nrwl/devkit");
const versions_1 = require("../../utils/versions");
const logger_1 = require("nx/src/utils/logger");
const package_json_1 = require("nx/src/utils/package-json");
function pluginLintCheckGenerator(host, options) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const project = (0, devkit_1.readProjectConfiguration)(host, options.projectName);
        const packageJson = (0, devkit_1.readJson)(host, (0, devkit_1.joinPathFragments)(project.root, 'package.json'));
        // This rule is eslint **only**
        if (projectIsEsLintEnabled(project)) {
            updateRootEslintConfig(host);
            updateProjectEslintConfig(host, project, packageJson);
            updateProjectTarget(host, options, packageJson);
            // Project is setup for vscode
            if (host.exists('.vscode')) {
                setupVsCodeLintingForJsonFiles(host);
            }
            // Project contains migrations.json
            const migrationsPath = (0, package_json_1.readNxMigrateConfig)(packageJson).migrations;
            if (migrationsPath &&
                host.exists((0, devkit_1.joinPathFragments)(project.root, migrationsPath))) {
                addMigrationJsonChecks(host, options, packageJson);
            }
        }
        else {
            devkit_1.logger.error(`${logger_1.NX_PREFIX} plugin lint checks can only be added to plugins which use eslint for linting`);
        }
        const installTask = (0, devkit_1.addDependenciesToPackageJson)(host, {}, { 'jsonc-eslint-parser': versions_1.jsoncEslintParserVersion });
        return () => installTask;
    });
}
exports.default = pluginLintCheckGenerator;
function addMigrationJsonChecks(host, options, packageJson) {
    var _a, _b;
    const projectConfiguration = (0, devkit_1.readProjectConfiguration)(host, options.projectName);
    const [eslintTarget, eslintTargetConfiguration] = getEsLintOptions(projectConfiguration);
    const relativeMigrationsJsonPath = (0, package_json_1.readNxMigrateConfig)(packageJson).migrations;
    if (!relativeMigrationsJsonPath) {
        return;
    }
    const migrationsJsonPath = (0, devkit_1.joinPathFragments)(projectConfiguration.root, relativeMigrationsJsonPath);
    if (eslintTarget &&
        !((_b = (_a = eslintTargetConfiguration.options) === null || _a === void 0 ? void 0 : _a.lintFilePatterns) === null || _b === void 0 ? void 0 : _b.includes(migrationsJsonPath))) {
        // Add to lintFilePatterns
        eslintTargetConfiguration.options.lintFilePatterns.push(migrationsJsonPath);
        (0, devkit_1.updateProjectConfiguration)(host, options.projectName, projectConfiguration);
        // Update project level eslintrc
        (0, devkit_1.updateJson)(host, `${projectConfiguration.root}/.eslintrc.json`, (c) => {
            const override = c.overrides.find((o) => { var _a, _b; return (_b = Object.keys((_a = o.rules) !== null && _a !== void 0 ? _a : {})) === null || _b === void 0 ? void 0 : _b.includes('@nrwl/nx/nx-plugin-checks'); });
            if (Array.isArray(override === null || override === void 0 ? void 0 : override.files) &&
                !override.files.includes(relativeMigrationsJsonPath)) {
                override.files.push(relativeMigrationsJsonPath);
            }
            return c;
        });
    }
}
exports.addMigrationJsonChecks = addMigrationJsonChecks;
function updateProjectTarget(host, options, packageJson) {
    var _a, _b;
    const project = (0, devkit_1.readProjectConfiguration)(host, options.projectName);
    if (!project.targets) {
        return;
    }
    for (const [target, configuration] of Object.entries(project.targets)) {
        if (configuration.executor === '@nrwl/linter:eslint') {
            const opts = (_a = configuration.options) !== null && _a !== void 0 ? _a : {};
            (_b = opts.lintFilePatterns) !== null && _b !== void 0 ? _b : (opts.lintFilePatterns = []);
            if (packageJson.generators) {
                opts.lintFilePatterns.push((0, devkit_1.joinPathFragments)(project.root, packageJson.generators));
            }
            if (packageJson.schematics &&
                packageJson.schematics !== packageJson.generators) {
                opts.lintFilePatterns.push((0, devkit_1.joinPathFragments)(project.root, packageJson.schematics));
            }
            if (packageJson.executors) {
                opts.lintFilePatterns.push((0, devkit_1.joinPathFragments)(project.root, packageJson.executors));
            }
            if (packageJson.builders &&
                packageJson.builders !== packageJson.executors) {
                opts.lintFilePatterns.push((0, devkit_1.joinPathFragments)(project.root, packageJson.builders));
            }
            opts.lintFilePatterns.push(`${project.root}/package.json`);
            opts.lintFilePatterns = [...new Set(opts.lintFilePatterns)];
            project.targets[target].options = opts;
        }
    }
    (0, devkit_1.updateProjectConfiguration)(host, options.projectName, project);
}
function updateProjectEslintConfig(host, options, packageJson) {
    var _a;
    // Update the project level lint configuration to specify
    // the plugin schema rule for generated files
    const eslintPath = `${options.root}/.eslintrc.json`;
    const eslintConfig = (0, devkit_1.readJson)(host, eslintPath);
    (_a = eslintConfig.overrides) !== null && _a !== void 0 ? _a : (eslintConfig.overrides = []);
    if (!eslintConfig.overrides.some((x) => { var _a; return Object.keys((_a = x.rules) !== null && _a !== void 0 ? _a : {}).includes('@nrwl/nx/nx-plugin-checks'); })) {
        eslintConfig.overrides.push({
            files: [
                './package.json',
                packageJson.generators,
                packageJson.executors,
                packageJson.schematics,
                packageJson.builders,
            ].filter((f) => !!f),
            parser: 'jsonc-eslint-parser',
            rules: {
                '@nrwl/nx/nx-plugin-checks': 'error',
            },
        });
    }
    (0, devkit_1.writeJson)(host, eslintPath, eslintConfig);
}
// Update the root eslint to specify a parser for json files
// This is required, otherwise every json file that is not overriden
// will display false errors in the IDE
function updateRootEslintConfig(host) {
    var _a;
    const rootESLint = (0, devkit_1.readJson)(host, '.eslintrc.json');
    (_a = rootESLint.overrides) !== null && _a !== void 0 ? _a : (rootESLint.overrides = []);
    if (!eslintConfigContainsJsonOverride(rootESLint)) {
        rootESLint.overrides.push({
            files: '*.json',
            parser: 'jsonc-eslint-parser',
            rules: {},
        });
        (0, devkit_1.writeJson)(host, '.eslintrc.json', rootESLint);
    }
}
function setupVsCodeLintingForJsonFiles(host) {
    var _a;
    let existing = {};
    if (host.exists('.vscode/settings.json')) {
        existing = (0, devkit_1.readJson)(host, '.vscode/settings.json');
    }
    else {
        devkit_1.logger.info(`${logger_1.NX_PREFIX} We've updated the vscode settings for this repository to ensure that plugin lint checks show up inside your IDE. This created .vscode/settings.json. To read more about this file, check vscode's documentation. It is frequently not commited, so other developers may need to add similar settings if they'd like to see the lint checks in the IDE rather than only during linting.`);
    }
    // setup eslint validation for json files
    const eslintValidate = (_a = existing['eslint.validate']) !== null && _a !== void 0 ? _a : [];
    if (!eslintValidate.includes('json')) {
        existing['eslint.validate'] = [...eslintValidate, 'json'];
    }
    (0, devkit_1.writeJson)(host, '.vscode/settings.json', existing);
}
function eslintConfigContainsJsonOverride(eslintConfig) {
    return eslintConfig.overrides.some((x) => {
        if (typeof x.files === 'string' && x.files.includes('.json')) {
            return true;
        }
        return Array.isArray(x.files) && x.files.some((f) => f.includes('.json'));
    });
}
function projectIsEsLintEnabled(project) {
    return !!getEsLintOptions(project);
}
function getEsLintOptions(project) {
    return Object.entries(project.targets || {}).find(([, x]) => x.executor === '@nrwl/linter:eslint');
}
exports.getEsLintOptions = getEsLintOptions;
//# sourceMappingURL=generator.js.map