"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.reduxSchematic = exports.reduxGenerator = void 0;
const tslib_1 = require("tslib");
const path = require("path");
const ts = require("typescript");
const ast_utils_1 = require("../../utils/ast-utils");
const versions_1 = require("../../utils/versions");
const devkit_1 = require("@nrwl/devkit");
const typescript_1 = require("@nrwl/workspace/src/utilities/typescript");
function reduxGenerator(host, schema) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const options = normalizeOptions(host, schema);
        generateReduxFiles(host, options);
        addExportsToBarrel(host, options);
        const installTask = addReduxPackageDependencies(host);
        addStoreConfiguration(host, options);
        updateReducerConfiguration(host, options);
        yield (0, devkit_1.formatFiles)(host);
        return installTask;
    });
}
exports.reduxGenerator = reduxGenerator;
function generateReduxFiles(host, options) {
    (0, devkit_1.generateFiles)(host, (0, devkit_1.joinPathFragments)(__dirname, './files'), options.filesPath, Object.assign(Object.assign({}, options), { tmpl: '' }));
    if (options.js) {
        (0, devkit_1.toJS)(host);
    }
}
function addReduxPackageDependencies(host) {
    return (0, devkit_1.addDependenciesToPackageJson)(host, {
        '@reduxjs/toolkit': versions_1.reduxjsToolkitVersion,
        'react-redux': versions_1.reactReduxVersion,
    }, {});
}
function addExportsToBarrel(host, options) {
    const indexFilePath = path.join(options.projectSourcePath, options.js ? 'index.js' : 'index.ts');
    const indexSource = host.read(indexFilePath, 'utf-8');
    if (indexSource !== null) {
        const indexSourceFile = ts.createSourceFile(indexFilePath, indexSource, ts.ScriptTarget.Latest, true);
        const statePath = options.directory
            ? `./lib/${options.directory}/${options.fileName}`
            : `./lib/${options.fileName}`;
        const changes = (0, devkit_1.applyChangesToString)(indexSource, (0, ast_utils_1.addImport)(indexSourceFile, `export * from '${statePath}.slice';`));
        host.write(indexFilePath, changes);
    }
}
function addStoreConfiguration(host, options) {
    if (!options.appProjectSourcePath) {
        return;
    }
    const mainSource = host.read(options.appMainFilePath, 'utf-8');
    if (!mainSource.includes('redux')) {
        const mainSourceFile = ts.createSourceFile(options.appMainFilePath, mainSource, ts.ScriptTarget.Latest, true);
        const changes = (0, devkit_1.applyChangesToString)(mainSource, (0, ast_utils_1.addReduxStoreToMain)(options.appMainFilePath, mainSourceFile));
        host.write(options.appMainFilePath, changes);
    }
}
function updateReducerConfiguration(host, options) {
    if (!options.appProjectSourcePath) {
        return;
    }
    const mainSource = host.read(options.appMainFilePath, 'utf-8');
    const mainSourceFile = ts.createSourceFile(options.appMainFilePath, mainSource, ts.ScriptTarget.Latest, true);
    const changes = (0, devkit_1.applyChangesToString)(mainSource, (0, ast_utils_1.updateReduxStore)(options.appMainFilePath, mainSourceFile, {
        keyName: `${options.constantName}_FEATURE_KEY`,
        reducerName: `${options.propertyName}Reducer`,
        modulePath: `${options.projectModulePath}`,
    }));
    host.write(options.appMainFilePath, changes);
}
function normalizeOptions(host, options) {
    var _a;
    let appProjectSourcePath;
    let appMainFilePath;
    const extraNames = (0, devkit_1.names)(options.name);
    const projects = (0, devkit_1.getProjects)(host);
    const project = projects.get(options.project);
    const { sourceRoot, projectType } = project;
    const tsConfigJson = (0, devkit_1.readJson)(host, (0, typescript_1.getRootTsConfigPathInTree)(host));
    const tsPaths = tsConfigJson.compilerOptions
        ? tsConfigJson.compilerOptions.paths || {}
        : {};
    const modulePath = projectType === 'application'
        ? options.directory
            ? `./app/${options.directory}/${extraNames.fileName}.slice`
            : `./app/${extraNames.fileName}.slice`
        : Object.keys(tsPaths).find((k) => tsPaths[k].some((s) => s.includes(sourceRoot)));
    // If --project is set to an app, automatically configure store
    // for it without needing to specify --appProject.
    options.appProject =
        options.appProject ||
            (projectType === 'application' ? options.project : undefined);
    if (options.appProject) {
        const appConfig = projects.get(options.appProject);
        if (appConfig.projectType !== 'application') {
            throw new Error(`Expected ${options.appProject} to be an application but got ${appConfig.projectType}`);
        }
        appProjectSourcePath = appConfig.sourceRoot;
        appMainFilePath = path.join(appProjectSourcePath, options.js ? 'main.js' : 'main.tsx');
        if (!host.exists(appMainFilePath)) {
            throw new Error(`Could not find ${appMainFilePath} during store configuration`);
        }
    }
    return Object.assign(Object.assign(Object.assign({}, options), extraNames), { constantName: (0, devkit_1.names)(options.name).constantName.toUpperCase(), directory: (0, devkit_1.names)((_a = options.directory) !== null && _a !== void 0 ? _a : '').fileName, projectType, projectSourcePath: sourceRoot, projectModulePath: modulePath, appProjectSourcePath,
        appMainFilePath, filesPath: (0, devkit_1.joinPathFragments)(sourceRoot, projectType === 'application' ? 'app' : 'lib') });
}
exports.default = reduxGenerator;
exports.reduxSchematic = (0, devkit_1.convertNxGenerator)(reduxGenerator);
//# sourceMappingURL=redux.js.map