"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTsSourceFile = exports.isTheFileAStory = exports.findStorybookAndBuildTargetsAndCompiler = exports.dedupe = exports.findOrCreateConfig = exports.readCurrentWorkspaceStorybookVersionFromExecutor = exports.readCurrentWorkspaceStorybookVersionFromGenerator = exports.safeFileDelete = exports.isFramework = exports.Constants = void 0;
const devkit_1 = require("@nrwl/devkit");
const versions_1 = require("./versions");
const fs_1 = require("fs");
const os_1 = require("os");
const path_1 = require("path");
const find_nodes_1 = require("@nrwl/workspace/src/utilities/typescript/find-nodes");
const ts = require("typescript");
exports.Constants = {
    addonDependencies: ['@storybook/addons'],
    tsConfigExclusions: ['stories', '**/*.stories.ts'],
    pkgJsonScripts: {
        storybook: 'start-storybook -p 9001 -c .storybook',
    },
    jsonIndentLevel: 2,
    coreAddonPrefix: '@storybook/addon-',
    uiFrameworks: {
        angular: '@storybook/angular',
        react: '@storybook/react',
        html: '@storybook/html',
        'web-components': '@storybook/web-components',
        vue: '@storybook/vue',
        vue3: '@storybook/vue3',
        svelte: '@storybook/svelte',
        'react-native': '@storybook/react-native',
    },
};
function isFramework(type, schema) {
    if (type === 'angular' && schema.uiFramework === '@storybook/angular') {
        return true;
    }
    if (type === 'react' && schema.uiFramework === '@storybook/react') {
        return true;
    }
    if (type === 'html' && schema.uiFramework === '@storybook/html') {
        return true;
    }
    if (type === 'web-components' &&
        schema.uiFramework === '@storybook/web-components') {
        return true;
    }
    if (type === 'vue' && schema.uiFramework === '@storybook/vue') {
        return true;
    }
    if (type === 'vue3' && schema.uiFramework === '@storybook/vue3') {
        return true;
    }
    if (type === 'svelte' && schema.uiFramework === '@storybook/svelte') {
        return true;
    }
    if (type === 'react-native' &&
        schema.uiFramework === '@storybook/react-native') {
        return true;
    }
    return false;
}
exports.isFramework = isFramework;
function safeFileDelete(tree, path) {
    if (tree.exists(path)) {
        tree.delete(path);
        return true;
    }
    else {
        return false;
    }
}
exports.safeFileDelete = safeFileDelete;
function readCurrentWorkspaceStorybookVersionFromGenerator(tree) {
    const packageJsonContents = (0, devkit_1.readJson)(tree, 'package.json');
    return determineStorybookWorkspaceVersion(packageJsonContents);
}
exports.readCurrentWorkspaceStorybookVersionFromGenerator = readCurrentWorkspaceStorybookVersionFromGenerator;
function readCurrentWorkspaceStorybookVersionFromExecutor() {
    const packageJsonContents = (0, devkit_1.readJsonFile)('package.json');
    return determineStorybookWorkspaceVersion(packageJsonContents);
}
exports.readCurrentWorkspaceStorybookVersionFromExecutor = readCurrentWorkspaceStorybookVersionFromExecutor;
function determineStorybookWorkspaceVersion(packageJsonContents) {
    let workspaceStorybookVersion = versions_1.storybookVersion;
    if (packageJsonContents && packageJsonContents['devDependencies']) {
        if (packageJsonContents['devDependencies']['@storybook/angular']) {
            workspaceStorybookVersion =
                packageJsonContents['devDependencies']['@storybook/angular'];
        }
        if (packageJsonContents['devDependencies']['@storybook/react']) {
            workspaceStorybookVersion =
                packageJsonContents['devDependencies']['@storybook/react'];
        }
        if (packageJsonContents['devDependencies']['@storybook/core']) {
            workspaceStorybookVersion =
                packageJsonContents['devDependencies']['@storybook/core'];
        }
        if (packageJsonContents['devDependencies']['@storybook/react-native']) {
            workspaceStorybookVersion =
                packageJsonContents['devDependencies']['@storybook/react-native'];
        }
    }
    if (packageJsonContents && packageJsonContents['dependencies']) {
        if (packageJsonContents['dependencies']['@storybook/angular']) {
            workspaceStorybookVersion =
                packageJsonContents['dependencies']['@storybook/angular'];
        }
        if (packageJsonContents['dependencies']['@storybook/react']) {
            workspaceStorybookVersion =
                packageJsonContents['dependencies']['@storybook/react'];
        }
        if (packageJsonContents['dependencies']['@storybook/core']) {
            workspaceStorybookVersion =
                packageJsonContents['dependencies']['@storybook/core'];
        }
        if (packageJsonContents['dependencies']['@storybook/react-native']) {
            workspaceStorybookVersion =
                packageJsonContents['dependencies']['@storybook/react-native'];
        }
    }
    return workspaceStorybookVersion;
}
function findOrCreateConfig(config, context) {
    if ((config === null || config === void 0 ? void 0 : config.configFolder) && (0, fs_1.statSync)(config.configFolder).isDirectory()) {
        return config.configFolder;
    }
    else if ((0, fs_1.statSync)(config.configPath).isFile() &&
        (0, fs_1.statSync)(config.pluginPath).isFile() &&
        (0, fs_1.statSync)(config.srcRoot).isFile()) {
        return createStorybookConfig(config.configPath, config.pluginPath, config.srcRoot);
    }
    else {
        const sourceRoot = context.workspace.projects[context.projectName].root;
        if ((0, fs_1.statSync)((0, path_1.join)(context.root, sourceRoot, '.storybook')).isDirectory()) {
            return (0, path_1.join)(context.root, sourceRoot, '.storybook');
        }
    }
    throw new Error('No configuration settings');
}
exports.findOrCreateConfig = findOrCreateConfig;
function createStorybookConfig(configPath, pluginPath, srcRoot) {
    const tmpDir = (0, os_1.tmpdir)();
    const tmpFolder = `${tmpDir}${path_1.sep}`;
    (0, fs_1.mkdtempSync)(tmpFolder);
    (0, fs_1.copyFileSync)(configPath, `${tmpFolder}/${(0, path_1.basename)(configPath)}`, fs_1.constants.COPYFILE_EXCL);
    (0, fs_1.copyFileSync)(pluginPath, `${tmpFolder}/${(0, path_1.basename)(pluginPath)}`, fs_1.constants.COPYFILE_EXCL);
    (0, fs_1.copyFileSync)(srcRoot, `${tmpFolder}/${(0, path_1.basename)(srcRoot)}`, fs_1.constants.COPYFILE_EXCL);
    return tmpFolder;
}
function dedupe(arr) {
    return Array.from(new Set(arr));
}
exports.dedupe = dedupe;
/**
 * This function is used to find the build targets for Storybook
 * and the project (if it's buildable).
 *
 * The reason this function exists is because we cannot assume
 * that the user has not created a custom build target for the project,
 * or that they have not changed the name of the build target
 * from build to anything else.
 *
 * So, in order to find the correct name of the target,
 * we have to look into all the targets, check the executor
 * they are using, and infer from the executor that the target
 * is a build target.
 */
function findStorybookAndBuildTargetsAndCompiler(targets) {
    const returnObject = {};
    Object.entries(targets).forEach(([target, targetConfig]) => {
        var _a, _b, _c, _d, _e;
        switch (targetConfig.executor) {
            case '@storybook/angular:start-storybook':
                returnObject.storybookTarget = target;
                break;
            case '@storybook/angular:build-storybook':
                returnObject.storybookBuildTarget = target;
                break;
            // If it's a non-angular project, these will be the executors
            case '@nrwl/storybook:storybook':
                returnObject.storybookTarget = target;
                break;
            case '@nrwl/storybook:build':
                returnObject.storybookBuildTarget = target;
                break;
            case '@angular-devkit/build-angular:browser':
                /**
                 * Not looking for '@nrwl/angular:ng-packagr-lite' or any other
                 * angular executors.
                 * Only looking for '@angular-devkit/build-angular:browser'
                 * because the '@nrwl/angular:ng-packagr-lite' executor
                 * (and maybe the other custom exeucutors)
                 * does not support styles and extra options, so the user
                 * will be forced to switch to build-storybook to add extra options.
                 *
                 * So we might as well use the build-storybook by default to
                 * avoid any errors.
                 */
                returnObject.ngBuildTarget = target;
                returnObject.compiler = (_a = targetConfig === null || targetConfig === void 0 ? void 0 : targetConfig.options) === null || _a === void 0 ? void 0 : _a.compiler;
                break;
            case '@nrwl/next:build':
                returnObject.nextBuildTarget = target;
                returnObject.compiler = (_b = targetConfig === null || targetConfig === void 0 ? void 0 : targetConfig.options) === null || _b === void 0 ? void 0 : _b.compiler;
                break;
            case '@nrwl/web:webpack':
            case '@nrwl/web:rollup':
            case '@nrwl/js:tsc':
            case '@nrwl/angular:ng-packagr-lite':
                returnObject.otherBuildTarget = target;
                returnObject.compiler = (_c = targetConfig === null || targetConfig === void 0 ? void 0 : targetConfig.options) === null || _c === void 0 ? void 0 : _c.compiler;
                break;
            default:
                if ((_d = targetConfig.options) === null || _d === void 0 ? void 0 : _d.compiler) {
                    returnObject.otherBuildTarget = target;
                    returnObject.compiler = (_e = targetConfig === null || targetConfig === void 0 ? void 0 : targetConfig.options) === null || _e === void 0 ? void 0 : _e.compiler;
                }
                break;
        }
    });
    return returnObject;
}
exports.findStorybookAndBuildTargetsAndCompiler = findStorybookAndBuildTargetsAndCompiler;
function isTheFileAStory(tree, path) {
    const ext = path.slice(path.lastIndexOf('.'));
    let fileIsStory = false;
    if (ext === '.tsx' || ext === '.ts') {
        const file = getTsSourceFile(tree, path);
        const importArray = (0, find_nodes_1.findNodes)(file, [ts.SyntaxKind.ImportDeclaration]);
        let nodeContainsStorybookImport = false;
        let nodeContainsStoryImport = false;
        importArray.forEach((importNode) => {
            const importPath = (0, find_nodes_1.findNodes)(importNode, [ts.SyntaxKind.StringLiteral]);
            importPath.forEach((importPath) => {
                var _a;
                if ((_a = importPath.getText()) === null || _a === void 0 ? void 0 : _a.includes('@storybook/')) {
                    nodeContainsStorybookImport = true;
                }
            });
            const importSpecifiers = (0, find_nodes_1.findNodes)(importNode, [
                ts.SyntaxKind.ImportSpecifier,
            ]);
            importSpecifiers.forEach((importSpecifier) => {
                if (importSpecifier.getText() === 'Story' ||
                    importSpecifier.getText() === 'storiesOf' ||
                    importSpecifier.getText() === 'ComponentStory') {
                    nodeContainsStoryImport = true;
                }
            });
            // We place this check within the loop, because we want the
            // import combination of Story from @storybook/*
            if (nodeContainsStorybookImport && nodeContainsStoryImport) {
                fileIsStory = true;
            }
        });
    }
    else {
        fileIsStory =
            (path.endsWith('.js') && path.endsWith('.stories.js')) ||
                (path.endsWith('.jsx') && path.endsWith('.stories.jsx'));
    }
    return fileIsStory;
}
exports.isTheFileAStory = isTheFileAStory;
function getTsSourceFile(host, path) {
    const buffer = host.read(path);
    if (!buffer) {
        throw new Error(`Could not read TS file (${path}).`);
    }
    const content = buffer.toString();
    const source = ts.createSourceFile(path, content, ts.ScriptTarget.Latest, true);
    return source;
}
exports.getTsSourceFile = getTsSourceFile;
//# sourceMappingURL=utilities.js.map