/*
 * Decompiled with CFR 0.152.
 */
package org.xbib.netty.http.client.pool;

import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelInitializer;
import io.netty.channel.ChannelPromise;
import io.netty.channel.pool.ChannelPoolHandler;
import io.netty.channel.socket.SocketChannel;
import io.netty.handler.codec.http.HttpVersion;
import io.netty.handler.codec.http2.DefaultHttp2GoAwayFrame;
import io.netty.util.AttributeKey;
import io.netty.util.concurrent.GenericFutureListener;
import java.io.IOException;
import java.net.ConnectException;
import java.net.SocketAddress;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.xbib.netty.http.client.api.Pool;
import org.xbib.netty.http.common.PoolKey;

public class BoundedChannelPool<K extends PoolKey>
implements Pool<Channel> {
    private static final Logger logger = Logger.getLogger(BoundedChannelPool.class.getName());
    private final Semaphore semaphore;
    private final HttpVersion httpVersion;
    private final ChannelPoolHandler channelPoolhandler;
    private final List<K> nodes;
    private final int numberOfNodes;
    private final int retriesPerNode;
    private final Map<K, Bootstrap> bootstraps;
    private final Map<K, List<Channel>> channels;
    private final Map<K, Queue<Channel>> availableChannels;
    private final Map<K, Integer> counts;
    private final Map<K, Integer> failedCounts;
    private final Lock lock;
    private final AttributeKey<K> attributeKey;
    private PoolKeySelector<K> poolKeySelector;

    public BoundedChannelPool(Semaphore semaphore, HttpVersion httpVersion, List<K> nodes, Bootstrap bootstrap, ChannelPoolHandler channelPoolHandler, int retriesPerNode, Pool.PoolKeySelectorType poolKeySelectorType) {
        this.semaphore = semaphore;
        this.httpVersion = httpVersion;
        this.channelPoolhandler = channelPoolHandler;
        this.nodes = nodes;
        this.retriesPerNode = retriesPerNode;
        switch (poolKeySelectorType) {
            case RANDOM: {
                this.poolKeySelector = new RandomPoolKeySelector();
                break;
            }
            case ROUNDROBIN: {
                this.poolKeySelector = new RoundRobinKeySelector();
            }
        }
        this.lock = new ReentrantLock();
        this.attributeKey = AttributeKey.valueOf((String)"poolKey");
        if (nodes == null || nodes.isEmpty()) {
            throw new IllegalArgumentException("nodes must not be empty");
        }
        this.numberOfNodes = nodes.size();
        this.bootstraps = new HashMap<K, Bootstrap>(this.numberOfNodes);
        this.channels = new ConcurrentHashMap<K, List<Channel>>(this.numberOfNodes);
        this.availableChannels = new ConcurrentHashMap<K, Queue<Channel>>(this.numberOfNodes);
        this.counts = new ConcurrentHashMap<K, Integer>(this.numberOfNodes);
        this.failedCounts = new ConcurrentHashMap<K, Integer>(this.numberOfNodes);
        for (PoolKey node : nodes) {
            ChannelPoolInitializer initializer = new ChannelPoolInitializer(this, node, channelPoolHandler);
            this.bootstraps.put(node, (Bootstrap)bootstrap.clone().remoteAddress((SocketAddress)node.getInetSocketAddress()).handler((ChannelHandler)initializer));
            this.availableChannels.put(node, new ConcurrentLinkedQueue());
            this.counts.put(node, 0);
            this.failedCounts.put(node, 0);
        }
        logger.log(Level.FINE, "pool is up");
    }

    public HttpVersion getVersion() {
        return this.httpVersion;
    }

    public AttributeKey<K> getAttributeKey() {
        return this.attributeKey;
    }

    public void prepare(int channelCount) throws ConnectException {
        if (channelCount <= 0) {
            throw new IllegalArgumentException("channel count must be greater zero, but got " + channelCount);
        }
        for (int i = 0; i < channelCount; ++i) {
            Channel channel = this.newConnection();
            if (channel == null) {
                throw new ConnectException("failed to prepare channels");
            }
            PoolKey key = (PoolKey)channel.attr(this.attributeKey).get();
            if (channel.isActive()) {
                Queue<Channel> channelQueue = this.availableChannels.get(key);
                if (channelQueue == null) continue;
                channelQueue.add(channel);
                continue;
            }
            channel.close();
        }
        logger.log(Level.FINE, "prepared " + channelCount + " channels: " + this.availableChannels);
    }

    public Channel acquire() throws Exception {
        Channel channel = null;
        if (this.semaphore.tryAcquire()) {
            channel = this.poll();
            if (channel == null) {
                channel = this.newConnection();
            }
            if (channel == null) {
                this.semaphore.release();
                throw new ConnectException();
            }
            if (this.channelPoolhandler != null) {
                this.channelPoolhandler.channelAcquired(channel);
            }
        }
        return channel;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void release(Channel channel, boolean close) throws Exception {
        try {
            if (channel != null) {
                if (channel.isActive()) {
                    PoolKey key = (PoolKey)channel.attr(this.attributeKey).get();
                    Queue<Channel> channelQueue = this.availableChannels.get(key);
                    if (channelQueue != null) {
                        channelQueue.add(channel);
                    }
                } else if (channel.isOpen() && close) {
                    logger.log(Level.FINE, "closing channel " + channel);
                    channel.close();
                }
                if (this.channelPoolhandler != null) {
                    this.channelPoolhandler.channelReleased(channel);
                }
            }
        }
        finally {
            this.semaphore.release();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close() throws IOException {
        this.lock.lock();
        try {
            logger.log(Level.FINE, "closing pool");
            int count = 0;
            HashSet<Channel> channelSet = new HashSet<Channel>();
            for (Map.Entry<K, Queue<Channel>> entry : this.availableChannels.entrySet()) {
                channelSet.addAll((Collection)entry.getValue());
            }
            for (Map.Entry<K, Collection<Channel>> entry : this.channels.entrySet()) {
                channelSet.addAll(entry.getValue());
            }
            for (Channel channel : channelSet) {
                if (channel == null || !channel.isOpen()) continue;
                logger.log(Level.FINE, "trying to abort channel " + channel);
                if (this.httpVersion.majorVersion() == 2) {
                    DefaultHttp2GoAwayFrame goAwayFrame = new DefaultHttp2GoAwayFrame(0L);
                    ChannelPromise channelPromise = channel.newPromise();
                    channel.writeAndFlush((Object)goAwayFrame, channelPromise);
                    try {
                        channelPromise.get();
                        logger.log(Level.FINE, "goaway frame sent to " + channel);
                    }
                    catch (ExecutionException e) {
                        logger.log(Level.FINE, e.getMessage(), e);
                    }
                    catch (InterruptedException e) {
                        throw new IOException(e);
                    }
                }
                channel.close();
                ++count;
            }
            this.availableChannels.clear();
            this.channels.clear();
            this.bootstraps.clear();
            this.counts.clear();
            logger.log(Level.FINE, "closed pool (found " + count + " connections open)");
        }
        finally {
            this.lock.unlock();
        }
    }

    private Channel newConnection() throws ConnectException {
        Channel channel = null;
        Object key = null;
        Integer min = Integer.MAX_VALUE;
        for (int j = 0; j < this.numberOfNodes; ++j) {
            K nextKey = this.poolKeySelector.key();
            Integer next = this.counts.get(nextKey);
            if (next == null || next == 0) {
                key = nextKey;
                break;
            }
            if (next >= min) continue;
            min = next;
            key = nextKey;
        }
        if (key != null) {
            logger.log(Level.FINE, "trying connection to " + key);
            try {
                channel = this.connect(key);
            }
            catch (Exception e) {
                logger.log(Level.WARNING, "failed to create a new connection to " + key + ": " + e.toString());
                if (this.retriesPerNode > 0) {
                    int selectedNodeFailedConnAttemptsCount = this.failedCounts.get(key) + 1;
                    this.failedCounts.put(key, selectedNodeFailedConnAttemptsCount);
                    if (selectedNodeFailedConnAttemptsCount > this.retriesPerNode) {
                        logger.log(Level.WARNING, "failed to connect to the node " + key + " " + selectedNodeFailedConnAttemptsCount + " times, excluding the node from the connection pool");
                        this.counts.put(key, Integer.MAX_VALUE);
                        boolean allNodesExcluded = true;
                        for (PoolKey node2 : this.nodes) {
                            if (this.counts.get(node2) >= Integer.MAX_VALUE) continue;
                            allNodesExcluded = false;
                            break;
                        }
                        if (allNodesExcluded) {
                            logger.log(Level.SEVERE, "no nodes left in the connection pool");
                        }
                    }
                }
                if (e instanceof ConnectException) {
                    throw (ConnectException)e;
                }
                throw new ConnectException(e.getMessage());
            }
        }
        if (channel != null) {
            channel.closeFuture().addListener((GenericFutureListener)new CloseChannelListener(this, key, channel));
            channel.attr(this.attributeKey).set(key);
            this.channels.computeIfAbsent((PoolKey)key, (Function<PoolKey, List<Channel>>)((Function<PoolKey, List>)node -> new ArrayList())).add(channel);
            this.counts.put(key, this.counts.get(key) + 1);
            if (this.retriesPerNode > 0) {
                this.failedCounts.put(key, 0);
            }
        }
        return channel;
    }

    private Channel connect(K key) throws Exception {
        Bootstrap bootstrap = this.bootstraps.get(key);
        if (bootstrap != null) {
            return bootstrap.connect().sync().channel();
        }
        return null;
    }

    private Channel poll() {
        for (int j = 0; j < this.numberOfNodes; ++j) {
            K key = this.poolKeySelector.key();
            logger.log(Level.FINE, "poll: key = " + key);
            Queue<Channel> channelQueue = this.availableChannels.get(key);
            if (channelQueue != null) {
                Channel channel = channelQueue.poll();
                if (channel == null || !channel.isActive()) continue;
                return channel;
            }
            logger.log(Level.WARNING, "channel queue is null?");
        }
        return null;
    }

    static class ChannelPoolInitializer
    extends ChannelInitializer<SocketChannel> {
        private final K key;
        private final ChannelPoolHandler channelPoolHandler;
        final /* synthetic */ BoundedChannelPool this$0;

        ChannelPoolInitializer(K key, ChannelPoolHandler channelPoolHandler) {
            this.this$0 = this$0;
            this.key = key;
            this.channelPoolHandler = channelPoolHandler;
        }

        protected void initChannel(SocketChannel channel) throws Exception {
            if (!channel.eventLoop().inEventLoop()) {
                throw new IllegalStateException();
            }
            channel.attr(this.this$0.attributeKey).set(this.key);
            if (this.channelPoolHandler != null) {
                this.channelPoolHandler.channelCreated((Channel)channel);
            }
        }
    }

    private static class CloseChannelListener
    implements ChannelFutureListener {
        private final K key;
        private final Channel channel;
        final /* synthetic */ BoundedChannelPool this$0;

        private CloseChannelListener(K key, Channel channel) {
            this.this$0 = var1_1;
            this.key = key;
            this.channel = channel;
        }

        public void operationComplete(ChannelFuture future) {
            logger.log(Level.FINE, "connection to " + this.key + " closed");
            this.this$0.lock.lock();
            try {
                List<Channel> channels;
                if (this.this$0.counts.containsKey(this.key)) {
                    this.this$0.counts.put(this.key, this.this$0.counts.get(this.key) - 1);
                }
                if ((channels = this.this$0.channels.get(this.key)) != null) {
                    channels.remove(this.channel);
                }
                this.this$0.semaphore.release();
            }
            finally {
                this.this$0.lock.unlock();
            }
        }
    }

    private class RoundRobinKeySelector
    implements PoolKeySelector<K> {
        int r = 0;

        private RoundRobinKeySelector() {
        }

        @Override
        public K key() {
            return (PoolKey)BoundedChannelPool.this.nodes.get(this.r++ % BoundedChannelPool.this.numberOfNodes);
        }
    }

    private class RandomPoolKeySelector
    implements PoolKeySelector<K> {
        private RandomPoolKeySelector() {
        }

        @Override
        public K key() {
            int r = ThreadLocalRandom.current().nextInt(BoundedChannelPool.this.numberOfNodes);
            return (PoolKey)BoundedChannelPool.this.nodes.get(r % BoundedChannelPool.this.numberOfNodes);
        }
    }

    private static interface PoolKeySelector<K extends PoolKey> {
        public K key();
    }
}

