package org.xtext.gradle.builder;

import com.google.common.collect.Iterables;
import com.google.common.io.Files;
import com.google.inject.Inject;
import com.google.inject.Provider;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Collection;
import java.util.Map;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmMember;
import org.eclipse.xtext.generator.trace.AbstractTraceRegion;
import org.eclipse.xtext.generator.trace.ITraceToBytecodeInstaller;
import org.eclipse.xtext.generator.trace.SourceRelativeURI;
import org.eclipse.xtext.generator.trace.TraceAsPrimarySourceInstaller;
import org.eclipse.xtext.generator.trace.TraceAsSmapInstaller;
import org.eclipse.xtext.generator.trace.TraceFileNameProvider;
import org.eclipse.xtext.generator.trace.TraceRegionSerializer;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.gradle.api.file.FileCollection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xtext.gradle.builder.InstallDebugInfoRequest;

@SuppressWarnings("all")
public class DebugInfoInstaller {
  private final static Logger logger = LoggerFactory.getLogger(DebugInfoInstaller.class);
  
  @Inject
  private Provider<TraceAsPrimarySourceInstaller> traceAsPrimarySourceInstallerProvider;
  
  @Inject
  private Provider<TraceAsSmapInstaller> traceAsSmapInstaller;
  
  @Inject
  private TraceFileNameProvider traceFileNameProvider;
  
  @Inject
  private TraceRegionSerializer traceRegionSerializer;
  
  public void installDebugInfo(final InstallDebugInfoRequest request) {
    try {
      Collection<File> _generatedJavaFiles = request.getGeneratedJavaFiles();
      for (final File javaFile : _generatedJavaFiles) {
        {
          String _absolutePath = javaFile.getAbsolutePath();
          String _traceFromJava = this.traceFileNameProvider.getTraceFromJava(_absolutePath);
          final File traceFile = new File(_traceFromJava);
          this.installDebugInfo(request, javaFile, traceFile);
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private void installDebugInfo(final InstallDebugInfoRequest request, final File javaFile, final File traceFile) throws IOException {
    boolean _exists = traceFile.exists();
    boolean _not = (!_exists);
    if (_not) {
      return;
    }
    final AbstractTraceRegion trace = this.readTraceFile(traceFile);
    SourceRelativeURI _associatedSrcRelativePath = trace.getAssociatedSrcRelativePath();
    final ITraceToBytecodeInstaller installer = this.createTraceToBytecodeInstaller(request, _associatedSrcRelativePath);
    if ((installer == null)) {
      return;
    }
    String _absolutePath = javaFile.getAbsolutePath();
    final URI javaFileUri = URI.createFileURI(_absolutePath);
    final ResourceSet resourceSet = request.getResourceSet();
    final Resource javaResource = resourceSet.getResource(javaFileUri, true);
    EList<EObject> _contents = javaResource.getContents();
    Iterable<JvmGenericType> _filter = Iterables.<JvmGenericType>filter(_contents, JvmGenericType.class);
    for (final JvmGenericType type : _filter) {
      this.installDebugInfo(request, javaFile, type, trace);
    }
  }
  
  private void installDebugInfo(final InstallDebugInfoRequest request, final File javaFile, final JvmGenericType type, final AbstractTraceRegion trace) throws IOException {
    StringConcatenation _builder = new StringConcatenation();
    String _qualifiedName = type.getQualifiedName();
    String _replace = _qualifiedName.replace(".", File.separator);
    _builder.append(_replace, "");
    _builder.append(".class");
    final String relativePath = _builder.toString();
    FileCollection _classesDirs = request.getClassesDirs();
    for (final File classesDir : _classesDirs) {
      {
        final File classFile = new File(classesDir, relativePath);
        boolean _exists = classFile.exists();
        if (_exists) {
          this.installDebugInfo(request, javaFile, classFile, trace);
          EList<JvmMember> _members = type.getMembers();
          Iterable<JvmGenericType> _filter = Iterables.<JvmGenericType>filter(_members, JvmGenericType.class);
          for (final JvmGenericType member : _filter) {
            this.installDebugInfo(request, javaFile, member, trace);
          }
        }
      }
    }
  }
  
  private void installDebugInfo(final InstallDebugInfoRequest request, final File javaFile, final File classFile, final AbstractTraceRegion trace) throws IOException {
    SourceRelativeURI _associatedSrcRelativePath = trace.getAssociatedSrcRelativePath();
    final ITraceToBytecodeInstaller traceToBytecodeInstaller = this.createTraceToBytecodeInstaller(request, _associatedSrcRelativePath);
    String _name = javaFile.getName();
    traceToBytecodeInstaller.setTrace(_name, trace);
    final File outputFile = classFile;
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Installing Xtext debug information into ");
    _builder.append(classFile, "");
    _builder.append(" using ");
    Class<? extends ITraceToBytecodeInstaller> _class = traceToBytecodeInstaller.getClass();
    String _simpleName = _class.getSimpleName();
    _builder.append(_simpleName, "");
    DebugInfoInstaller.logger.info(_builder.toString());
    File _parentFile = outputFile.getParentFile();
    _parentFile.mkdirs();
    final byte[] classContent = Files.toByteArray(classFile);
    byte[] _elvis = null;
    byte[] _installTrace = traceToBytecodeInstaller.installTrace(classContent);
    if (_installTrace != null) {
      _elvis = _installTrace;
    } else {
      _elvis = classContent;
    }
    final byte[] newClassContent = _elvis;
    Files.write(newClassContent, outputFile);
  }
  
  private ITraceToBytecodeInstaller createTraceToBytecodeInstaller(final InstallDebugInfoRequest request, final SourceRelativeURI sourceFile) {
    Map<String, InstallDebugInfoRequest.SourceInstallerConfig> _sourceInstallerByFileExtension = request.getSourceInstallerByFileExtension();
    URI _uRI = sourceFile.getURI();
    String _fileExtension = _uRI.fileExtension();
    InstallDebugInfoRequest.SourceInstallerConfig debugInfoConfig = _sourceInstallerByFileExtension.get(_fileExtension);
    InstallDebugInfoRequest.SourceInstaller _sourceInstaller = debugInfoConfig.getSourceInstaller();
    if (_sourceInstaller != null) {
      switch (_sourceInstaller) {
        case PRIMARY:
          final TraceAsPrimarySourceInstaller installer = this.traceAsPrimarySourceInstallerProvider.get();
          boolean _isHideSyntheticVariables = debugInfoConfig.isHideSyntheticVariables();
          installer.setHideSyntheticVariables(_isHideSyntheticVariables);
          return installer;
        case SMAP:
          return this.traceAsSmapInstaller.get();
        default:
          return null;
      }
    } else {
      return null;
    }
  }
  
  private AbstractTraceRegion readTraceFile(final File traceFile) throws IOException {
    final FileInputStream in = new FileInputStream(traceFile);
    try {
      return this.traceRegionSerializer.readTraceRegionFrom(in);
    } finally {
      in.close();
    }
  }
}
