package org.xyou.xcommon.profiler;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;

import org.xyou.xcommon.app.XApp;
import org.xyou.xcommon.entity.XObj;
import org.xyou.xcommon.schedule.XSchedule;
import org.xyou.xcommon.tool.XStr;

import io.prometheus.client.Counter;
import io.prometheus.client.Gauge;
import io.prometheus.client.Summary;
import lombok.AccessLevel;
import lombok.Getter;

public final class Model {

    @Getter(AccessLevel.PACKAGE)
    private final String nameService;
    @Getter(AccessLevel.PACKAGE)
    private final Map<String, Counter> mapCounter;
    @Getter(AccessLevel.PACKAGE)
    private final Map<String, Gauge> mapGauge;
    @Getter(AccessLevel.PACKAGE)
    private final Map<String, Summary> mapSummary;
    @Getter(AccessLevel.PACKAGE)
    private final XSchedule scheduleGauge;
    @Getter(AccessLevel.PACKAGE)
    private final transient Map<String, Integer> mapNumObject;

    private Model() {
        nameService = XStr.toSnakeFromKebab(
            XApp.getName()
        );
        mapCounter = new HashMap<>();
        mapGauge = new HashMap<>();
        mapSummary = new HashMap<>();
        scheduleGauge = new XSchedule();
        mapNumObject = new HashMap<>();
    }

    private static class Holder {

        public static final Model INST = new Model();
    }

    public static Model getInst() {
        return Holder.INST;
    }

    private String buildNameMetric(String nameProfiler) {
        return nameService + ":" + nameProfiler;
    }

    public <V> V getMetric(String name, Map<String, V> mapMetric, Function<String, V> buildMetric) {
        String nameMetric = buildNameMetric(name);
        if (!mapMetric.containsKey(nameMetric)) {
            synchronized (mapMetric) {
                if (!mapMetric.containsKey(nameMetric)) {
                    V metric = buildMetric.apply(nameMetric);
                    mapMetric.put(nameMetric, metric);

                }
            }
        }
        return mapMetric.get(nameMetric);
    }

    Counter getCounter(String name) {
        return getMetric(name, mapCounter, k -> Counter.build().name(k).help("x").register());
    }

    Gauge getGauge(String name) {
        return getMetric(name, mapGauge, k -> Gauge.build().name(k).help("x").register());
    }

    Summary getSummary(String name) {
        return getMetric(name, mapSummary, k -> Summary.build().name(k).help("x").register());
    }

    XProfilerObj createObj(Object obj) {
        synchronized (mapNumObject) {
            String nameCls = obj.getClass().getSimpleName();
            String name = XObj.fromObject(obj).getStr("name");
            Integer numObject = mapNumObject.getOrDefault(name, 0);
            String idObject = numObject.toString();
            String nameClsSnakeCase = XStr.toSnakeFromPascal(nameCls);
            String nameRegister = String.join(":", nameClsSnakeCase, name, idObject);
            numObject += 1;
            mapNumObject.put(name, numObject);
            return new XProfilerObj(nameRegister);
        }
    }

}
