package org.xyou.xcommon.seq;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import org.xyou.xcommon.ex.XEx;

public final class XSeq {

    @SafeVarargs
    public static <V> V[] newArray(V... arr) {
        return arr;
    }

    @SafeVarargs
    public static <V> List<V> newArrayList(V... arrEle) {
        return Lists.newArrayList(arrEle);
    }

    @SafeVarargs
    public static <V> Set<V> newHashSet(V... arrEle) {
        return Sets.newHashSet(arrEle);
    }

    public static <V> Set<V> newConcurrentHashSet() {
        return Sets.newConcurrentHashSet();
    }

    public static <V> Set<V> newConcurrentHashSe(Iterable<? extends V> elements) {
        return Sets.newConcurrentHashSet(elements);
    }

    public static boolean isEmpty(Object seq) {
        if (seq == null) {
            return true;
        }
        if (seq instanceof Collection) {
            Collection<?> coll = (Collection<?>) seq;
            if (coll.isEmpty()) {
                return true;
            }
            return false;
        }
        if (seq instanceof Object[]) {
            Object[] arr = (Object[]) seq;
            if (arr.length == 0) {
                return true;
            }
            return false;
        }
        throw XEx.createClassInvalid(seq);
    }

    public static <V> boolean isEmpty(Collection<V> coll) {
        if (coll == null || coll.isEmpty()) {
            return true;
        }
        return false;
    }

    public static <V> List<V> subLs(List<V> ls, int idxStart, int count) {
        int sizeLs = ls.size();
        if (idxStart >= sizeLs || count < 0) {
            return new ArrayList<>();
        }
        count = count == 0 ? sizeLs : count;
        int startIndex = idxStart < 0 ? 0 : idxStart;
        int toIndex = (startIndex + count) < sizeLs ? (startIndex + count) : sizeLs;
        List<V> lsSub = ls.subList(startIndex, toIndex);

        return new ArrayList<>(lsSub);
    }

    public static <V> List<V> subLs(List<V> ls, int sizeMax) {
        if (sizeMax == 0) {
            return new ArrayList<>();
        }
        List<V> lsSub = ls.size() > sizeMax ? ls.subList(0, sizeMax) : ls;
        lsSub = new ArrayList<>(lsSub);
        return lsSub;
    }

    public static <V> void addNotNull(Collection<V> coll, V ele) {
        if (ele != null) {
            coll.add(ele);
        }
    }

    public static <V> List<V> shuffleLsTop(List<V> lsItem, int numTop) {
        List<V> lsData = new ArrayList<>();
        List<V> lsTop = subLs(lsItem, 0, numTop);
        List<V> lsRemain = subLs(lsItem, numTop, lsItem.size());
        Collections.shuffle(lsTop);
        lsData.addAll(lsTop);
        lsData.addAll(lsRemain);

        return lsData;
    }

    public static <V> List<V> shuffleLsLsByLsWeight(
        List<List<V>> lsLs,
        List<Double> lsWeight,
        int sizePartition
    ) {
        if (isEmpty(lsLs)) {
            throw new RuntimeException("lsLs must have data.");
        }
        if (isEmpty(lsWeight)) {
            throw new RuntimeException("lsWeight must have data.");
        }
        if (lsLs.size() != lsWeight.size()) {
            throw new RuntimeException("lsWeight must be the same size as lsWeight.");
        }
        double weightTotal = sum(lsWeight);

        if (Math.abs(weightTotal - 1.0) > 1e-2) {
            throw new RuntimeException("The sum of lsWeight must be 1.");
        }
        if (sizePartition < 1) {
            throw new RuntimeException("Size partition must at least 1.");
        }

        List<Integer> lsIndex = lsLs.stream()
            .map(ele -> 0)
            .collect(Collectors.toList());

        List<Integer> lsStep = lsWeight.stream()
            .map(weight -> (int) Math.ceil(weight * sizePartition))
            .collect(Collectors.toList());

        List<V> lsResult = new ArrayList<>();
        List<V> lsShuffled = new ArrayList<>();

        while (true) {
            boolean isEmptyAllLs = true;
            lsShuffled.clear();

            for (int idx = 0; idx < lsLs.size(); ++idx) {
                List<V> lsData = lsLs.get(idx);
                int idxFrom = lsIndex.get(idx);
                if (isEmpty(lsData)) {
                    continue;
                }
                if (idxFrom >= lsData.size()) {
                    continue;
                }
                isEmptyAllLs = false;
                int step = lsStep.get(idx);
                int idxTo = idxFrom + step;
                idxTo = Math.min(idxTo, lsData.size());

                lsShuffled.addAll(lsData.subList(idxFrom, idxTo));
                lsIndex.set(idx, idxTo);
            }

            Collections.shuffle(lsShuffled);
            lsResult.addAll(lsShuffled);

            if (isEmptyAllLs) {
                break;
            }
        }
        return lsResult;
    }

    public static <V extends Number> double sum(Collection<V> coll) {
        if (coll == null || coll.size() == 0) {
            return 0;
        }
        double result = 0d;
        for (V ele : coll) {
            result += ele.doubleValue();
        }
        return result;
    }

    public static <V> List<V> distinct(Collection<V> coll) {
        Set<V> set = new HashSet<>();
        List<V> lsDistinct = new ArrayList<>();
        coll.forEach(ele -> {
            if (set.contains(ele)) {
                return;
            }
            lsDistinct.add(ele);
            set.add(ele);
        });
        return lsDistinct;
    }

    public static <V> Collection<V> addEleToCollectionIfNotNull(Collection<V> coll, V ele) {
        if (ele != null) {
            coll.add(ele);
        }
        return coll;
    }

    public static <V, T> List<T> map(Collection<V> coll, Function<V, T> func) {
        return coll.stream()
            .map(func)
            .collect(Collectors.toList());
    }

    public static <V> V reduce(Collection<V> coll, BiFunction<V, V, V> func) {
        Iterator<V> iterator = coll.iterator();
        V data = iterator.next();
        while (iterator.hasNext()) {
            V ele = iterator.next();
            data = func.apply(data, ele);
        }
        return data;
    }

}
