package org.xyou.xcommon.thread;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.xyou.xcommon.base.XBaseObject;
import org.xyou.xcommon.config.XConfig;
import org.xyou.xcommon.profiler.XProfiler;
import org.xyou.xcommon.profiler.XProfilerObj;
import org.xyou.xcommon.tool.XTime;

import lombok.AccessLevel;
import lombok.Getter;

public final class XThreadPool extends XBaseObject {

    private static final long serialVersionUID = 1L;

    @Getter(AccessLevel.PUBLIC)
    private String name;
    @Getter(AccessLevel.PUBLIC)
    private Integer numThread;
    @Getter(AccessLevel.PUBLIC)
    private Integer sizeQueueMax;
    @Getter(AccessLevel.PUBLIC)
    private Boolean isProfiler;
    private transient ThreadPoolExecutor pool;
    private transient XProfilerObj objProfiler;

    public XThreadPool(Integer numThread, Integer sizeQueueMax) {
        name = null;
        this.numThread = numThread;
        this.sizeQueueMax = sizeQueueMax;
        isProfiler = false;
        pool = initPoolThread(numThread, sizeQueueMax);
        objProfiler = null;
    }

    public XThreadPool(String name) {
        XConfig config = new XConfig(name);
        this.name = name;
        numThread = config.getInt("numThread");
        sizeQueueMax = config.getInt("sizeQueueMax");
        isProfiler = config.getBool("isProfiler", false);
        pool = initPoolThread(numThread, sizeQueueMax);
        if (isProfiler) {
            objProfiler = XProfiler.createObj(this);
            objProfiler.scheduleGauge("size", this::sizeQueue);
        }
    }

    private ThreadPoolExecutor initPoolThread(int numThread, int sizeQueueMax) {
        BlockingQueue<Runnable> queue = new LinkedBlockingDeque<Runnable>(sizeQueueMax);
        return new ThreadPoolExecutor(
            numThread,
            numThread,
            XTime.MS_MIN,
            TimeUnit.MILLISECONDS,
            queue
        );
    }

    public int sizeQueue() {
        return pool.getQueue().size();
    }

    public boolean submit(Runnable func) {
        pool.submit(func::run);
        return true;
    }

    public boolean shutdown() {
        return shutdown(Long.MAX_VALUE);
    }

    public boolean shutdown(long msWait) {
        try {
            pool.shutdown();
            return pool.awaitTermination(msWait, TimeUnit.MILLISECONDS);
        } catch (Throwable ex) {
        }
        return false;
    }
}
