package org.xyou.xcommon.yaml;

import java.io.File;
import java.io.IOException;
import java.sql.Timestamp;
import java.util.LinkedHashMap;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonAutoDetect.Visibility;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.std.StdSerializer;
import com.fasterxml.jackson.databind.type.MapType;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;

import org.xyou.xcommon.entity.XObj;
import org.xyou.xcommon.logger.XLogger;
import org.xyou.xcommon.tool.XFile;
import org.xyou.xcommon.tool.XTime;

final class Model {

    private final transient XLogger logger;
    private final transient ObjectMapper mapperJson;
    private final transient ObjectMapper mapperYaml;

    private static class Holder {

        public static final Model INST = new Model();
    }

    static Model getInst() {
        return Holder.INST;
    }

    private void initMapper(ObjectMapper mapper) {
        mapper.setSerializationInclusion(Include.NON_NULL);
        mapper.setVisibility(PropertyAccessor.ALL, Visibility.NONE);
        mapper.setVisibility(PropertyAccessor.FIELD, Visibility.ANY);
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        SimpleModule module = new SimpleModule();
        module.addSerializer(Timestamp.class, new StdSerializer<Timestamp>(Timestamp.class) {

            private static final long serialVersionUID = 1L;

            @Override
            public void serialize(Timestamp value, JsonGenerator gen, SerializerProvider provider) throws IOException {
                gen.writeNumber(value.getTime() / XTime.MS_SEC);
            }

        });
        module.addDeserializer(Timestamp.class, new StdDeserializer<Timestamp>(Timestamp.class) {

            private static final long serialVersionUID = 1L;

            @Override
            public Timestamp deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JsonProcessingException {
                return new Timestamp(p.getValueAsLong() * XTime.MS_SEC);
            }

        });
        module.addSerializer(XObj.class, new StdSerializer<XObj>(XObj.class) {

            private static final long serialVersionUID = 1L;

            @Override
            public void serialize(XObj value, JsonGenerator gen, SerializerProvider provider) throws IOException {
                Map<Object, Object> map = value.getMap();
                gen.writeObject(map);
            }
        });
        module.addDeserializer(XObj.class, new StdDeserializer<XObj>(XObj.class) {

            private static final long serialVersionUID = 1L;

            @Override
            public XObj deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JsonProcessingException {
                @SuppressWarnings("unchecked")
                Map<Object, Object> map = p.readValueAs(Map.class);
                return new XObj().putAll(map);
            }
        });
        mapper.registerModule(module);
    }

    private Model() {
        logger = new XLogger();
        mapperJson = new ObjectMapper();
        mapperYaml = new ObjectMapper(new YAMLFactory());
        try {
            initMapper(mapperJson);
            initMapper(mapperYaml);
        } catch (Throwable ex) {
            logger.error(ex);
        }
    }

    public static String toStr(Object obj, ObjectMapper mapper) {
        try {
            if (obj == null ) {
                throw new RuntimeException("'obj null");
            }
            return mapper.writeValueAsString(obj);
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    public static <V> V fromStr(String data, Class<V> cls, ObjectMapper mapper) {
        try {
            return mapper.readValue(data, cls);
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    public static <V> Map<String, V> fromStrToMap(String data, Class<V> cls, ObjectMapper mapper) {
        try {
            MapType mapType = mapper.getTypeFactory().constructMapType(
                LinkedHashMap.class,
                String.class,
                cls
            );
            return mapper.readValue(data, mapType);
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    public static boolean toFile(String path, Object obj, ObjectMapper mapper) {
        try {
            XFile.mkdir(XFile.dirname(path));
            mapper.writeValue(new File(path), obj);
            return true;
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    public static <V> V fromFile(String path, Class<V> cls, ObjectMapper mapper) {
        try {
            return mapper.readValue(new File(path), cls);
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    public static <V> Map<String, V> fromFileToMap(String path, Class<V> cls, ObjectMapper mapper) {
        try {
            MapType mapType = mapper.getTypeFactory().constructMapType(
                LinkedHashMap.class,
                String.class,
                cls
            );
            return mapper.readValue(new File(path), mapType);
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    static ObjectMapper getMapperJson() {
        return getInst().mapperJson;
    }

    static ObjectMapper getMapperYaml() {
        return getInst().mapperYaml;
    }

}
