package org.xyou.xcommon.cache;

import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import org.xyou.xcommon.base.XBaseObject;
import org.xyou.xcommon.config.XConfig;
import org.xyou.xcommon.cvt.XCvt;
import org.xyou.xcommon.lock.XLock;
import org.xyou.xcommon.profiler.XProfiler;
import org.xyou.xcommon.profiler.XProfilerObj;
import org.xyou.xcommon.time.XTime;

import lombok.AccessLevel;
import lombok.Getter;
import lombok.NonNull;

public final class XCache<K, V> extends XBaseObject {

    private static final long serialVersionUID = 1L;

    @Getter
    private String name;
    @Getter
    private Integer sizeMax;
    @Getter
    private Integer setExpire;
    @Getter
    private Integer sizeMaxNull;
    @Getter
    private Integer secExpireNull;
    @Getter
    private Boolean isProfiler;
    private transient XLock poolLock;
    private transient Cache<K, V> cache;
    private transient Cache<K, Integer> cacheNull;
    @Getter(AccessLevel.PACKAGE)
    private transient XProfilerObj objProfiler;

    public XCache(@NonNull String name) {
        XConfig config = new XConfig(name);
        this.name = name;
        sizeMax = config.getInt("sizeMax");
        setExpire = config.getInt("secExpire");
        sizeMaxNull = config.getInt("sizeMaxNull", null);
        secExpireNull = config.getInt("secExpireNull", null);
        isProfiler = config.getBool("isProfiler", false);
        poolLock = new XLock();
        cache = CacheBuilder
            .newBuilder()
            .maximumSize(sizeMax)
            .expireAfterWrite(setExpire, TimeUnit.SECONDS)
            .build();
        if (sizeMaxNull != null && secExpireNull != null) {
            cacheNull = CacheBuilder
                .newBuilder()
                .maximumSize(sizeMaxNull)
                .expireAfterWrite(secExpireNull, TimeUnit.SECONDS)
                .build();
        }
        if (isProfiler) {
            objProfiler = XProfiler.createObj(this);
            objProfiler.scheduleGauge("size", this::size);
            objProfiler.scheduleGauge("sizeNull", this::sizeNull);
        }
    }

    public V get(@NonNull K key, @NonNull Supplier<V> func) {
        return get(key, func, false);
    }

    public V get(@NonNull K key, @NonNull Supplier<V> func, @NonNull Boolean isForceLoad) {
        if (isProfiler) {
            objProfiler.incCounter("count_request");
        }
        V value = null;
        if (!isForceLoad) {
            value = cache.getIfPresent(key);
            if (value != null) {
                if (isProfiler) {
                    objProfiler.incCounter("count_hit");
                }
                return value;
            }
            if (cacheNull != null) {
                if (cacheNull.getIfPresent(key) != null) {
                    if (isProfiler) {
                        objProfiler.incCounter("count_hit_null");
                    }
                    return value;
                }
            }
        }
        synchronized (poolLock.get(key)) {
            if (!isForceLoad) {
                value = cache.getIfPresent(key);
                if (value != null) {
                    if (isProfiler) {
                        objProfiler.incCounter("count_hit");
                    }
                    return value;
                }
            }
            if (isProfiler) {
                objProfiler.incCounter("count_load");
            }
            value = func.get();
            if (value != null) {
                if (isProfiler) {
                    objProfiler.incCounter("count_load_success");
                }
                put(key, value);
                return value;
            }
            if (cacheNull != null) {
                cacheNull.put(key, XCvt.toInt(XTime.getCurSec()));
            }
        }
        return value;
    }

    public void put(@NonNull K key, @NonNull V value) {
        cache.put(key, value);
    }

    public Long size() {
        return cache.size();
    }

    public Long sizeNull() {
        return cacheNull.size();
    }

}
