package org.xyou.xcommon.file;

import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import org.xyou.xcommon.ex.XEx;
import org.xyou.xcommon.seq.XSeq;
import org.xyou.xcommon.txt.XTxtReader;

import lombok.NonNull;

public final class XFile {

    public static void touch(@NonNull String path) {
        try {
            mkdir(dirname(path));
            new File(path).createNewFile();
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    public static boolean isfile(@NonNull String path) {
        return new File(path).isFile();
    }

    public static boolean isdir(@NonNull String path) {
        return new File(path).isDirectory();
    }

    public static boolean exist(@NonNull String path) {
        return new File(path).exists();
    }

    public static void rm(@NonNull String path) {
        File file = new File(path);
        File[] arrFile = file.listFiles();
        if (!XSeq.isEmpty(arrFile)) {
            for (File fileInner : arrFile) {
                try {
                    rm(fileInner.getPath());
                } catch (Throwable ex) {
                    throw new RuntimeException(ex);
                }
            }
        }
        file.delete();
    }

    public static String normpath(@NonNull String path) {
        return Paths.get("/", path).normalize().toString();
    }

    public static String realpath(@NonNull String path) {
        File file = new File(path);
        return file.getAbsoluteFile().getPath();
    }

    public static String dirname(@NonNull String path) {
        File file = new File(path);
        return file.getAbsoluteFile().getParent();
    }

    public static String basename(@NonNull String path) {
        File file = new File(path);
        return file.getName();
    }

    public static void mkdir(@NonNull String pathDir) {
        if (exist(pathDir)) {
            return;
        }
        if (!new File(pathDir).mkdirs()) {
            throw new RuntimeException("Create directory fail " + pathDir);
        }
    }

    public static boolean empty(@NonNull String path) {
        if (isfile(path)) {
            return XTxtReader.readAll(path).equals("");
        }
        if (isdir(path)) {
            return ls(path).size() == 0;
        }
        throw new RuntimeException("Not exist " + path);
    }

    public static List<String> ls(@NonNull String dir) {
        if (!isdir(dir)) {
            throw new RuntimeException("No directory " + dir);
        }
        return XSeq.newArrayList(new File(dir).listFiles()).stream()
            .map(File::getName)
            .collect(Collectors.toList());
    }

    public static List<String> walk(@NonNull String dir) {
        List<String> lsPathAll = new ArrayList<>();
        ls(dir).forEach(name -> {
            lsPathAll.add(name);
            String path = dir + "/" + name;
            if (!isdir(path)) {
                return;
            }
            walk(path).forEach(nameInner -> {
                String pathInner = name + "/" + nameInner;
                lsPathAll.add(pathInner);
            });
        });
        return lsPathAll;
    }

    public static String mime(@NonNull Object obj) {
        try {
            if (obj instanceof String) {
                return Model.getInst().getTika().detect((String) obj);
            }
            if (obj instanceof File) {
                return Model.getInst().getTika().detect((File) obj);
            }
            if (obj instanceof InputStream) {
                return Model.getInst().getTika().detect((InputStream) obj);
            }
            if (obj instanceof Path) {
                return Model.getInst().getTika().detect((Path) obj);
            }
            if (obj instanceof URL) {
                return Model.getInst().getTika().detect((URL) obj);
            }
            if (obj instanceof byte[]) {
                return Model.getInst().getTika().detect((byte[]) obj);
            }
            throw XEx.createClassInvalid(obj);
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    public static void cp(@NonNull Object from, @NonNull Object to) {
        try {
            if (to instanceof OutputStream) {
                OutputStream streamTo = (OutputStream) to;
                Files.copy(Paths.get((String) from), streamTo);
                streamTo.flush();
                return;
            }
            if (to instanceof String) {
                if (from instanceof InputStream) {
                    String pathTo = (String) to;
                    Files.copy((InputStream) from, Paths.get(pathTo), StandardCopyOption.REPLACE_EXISTING);
                    return;
                }
                String pathFrom = (String) from;
                String pathTo = (String) to;
                mkdir(dirname(pathTo));
                if (isdir(pathFrom)) {
                    ls(pathFrom).forEach(name -> cp(pathFrom + "/" + name, pathTo + "/" + name));
                    return;
                }
                Files.copy(Paths.get(pathFrom), Paths.get(pathTo), StandardCopyOption.REPLACE_EXISTING);
                return;
            }
            throw XEx.createClassInvalid(to);
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    public static void mv(@NonNull String from, @NonNull String to) {
        try {
            rm(to);
            mkdir(dirname(to));
            Files.move(Paths.get(from), Paths.get(to));
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

}
