package org.xyou.xcommon.schedule;

import java.time.Duration;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.xyou.xcommon.config.XConfig;
import org.xyou.xcommon.ex.XEx;
import org.xyou.xcommon.time.XTime;

import lombok.NonNull;

public final class XSchedule {

    private static final String ID_ZONE_VN_HCM = "Asia/Ho_Chi_Minh";

    private transient ScheduledExecutorService service;

    public XSchedule(@NonNull String name) {
        XConfig config = new XConfig(name);
        Integer numThread = config.getInt("numThread");
        init(numThread);
    }

    public XSchedule() {
        init(1);
    }

    public XSchedule(@NonNull Integer numThread) {
        init(numThread);
    }

    public void init(@NonNull Integer numThread) {
        service = Executors.newScheduledThreadPool(numThread);
    }

    public int sizeQueue() {
        ScheduledThreadPoolExecutor poolThread = (ScheduledThreadPoolExecutor) service;
        return poolThread.getQueue().size();
    }

    public boolean shutdown() {
        return shutdown(Long.MAX_VALUE);
    }

    public boolean shutdown(@NonNull Long msWait) {
        try {
            service.shutdown();
            return service.awaitTermination(msWait, TimeUnit.MILLISECONDS);
        } catch (Throwable ex) {
        }
        return false;
    }

    public XScheduleFuture scheduleDaily(@NonNull Integer hr, @NonNull Runnable func) {
        return scheduleDaily(
            XScheduleParam.builder()
                .hr(hr)
                .func(func)
                .build()
        );
    }

    public XScheduleFuture scheduleDaily(@NonNull XScheduleParam param) {
        XEx.checkNotNull(param, "func", "hr");
        ZonedDateTime now = ZonedDateTime.now(ZoneId.of(ID_ZONE_VN_HCM));
        Integer hr = param.getHr();
        Integer min = param.getMin();
        if (min == null) {
            min = 0;
        }
        Integer sec = param.getMin();
        if (sec == null) {
            sec = 0;
        }
        ZonedDateTime nextRun = now.withHour(hr).withMinute(min).withSecond(sec);
        if (now.compareTo(nextRun) > 0) {
            nextRun = nextRun.plusDays(1);
        }
        Duration duration = Duration.between(now, nextRun);
        long initalDelay = duration.getSeconds();
        Runnable func = param.getFunc();
        return scheduleAtFixedRate(
            XScheduleParam.builder()
                .msPeriod(XTime.MS_DAY)
                .msDelayInit(initalDelay * XTime.MS_SEC)
                .func(func)
                .build()
        );
    }

    public XScheduleFuture scheduleHourly(@NonNull Integer min, Runnable func) {
        return scheduleHourly(
            XScheduleParam.builder()
                .min(min)
                .func(func)
                .build()
        );
    }

    public XScheduleFuture scheduleHourly(@NonNull XScheduleParam param) {
        XEx.checkNotNull(param, "func", "min");
        ZonedDateTime now = ZonedDateTime.now(ZoneId.of(ID_ZONE_VN_HCM));
        Integer min = param.getMin();
        Integer sec = param.getMin();
        if (sec == null) {
            sec = 0;
        }
        ZonedDateTime nextRun = now.withMinute(min).withSecond(sec);
        if (now.compareTo(nextRun) > 0) {
            nextRun = nextRun.plusHours(1);
        }
        Duration duration = Duration.between(now, nextRun);
        long initalDelay = duration.getSeconds();
        Runnable func = param.getFunc();
        return scheduleAtFixedRate(
            XScheduleParam.builder()
                .msPeriod(XTime.MS_HR)
                .msDelayInit(initalDelay * XTime.MS_SEC)
                .func(func)
                .build()
        );
    }

    public XScheduleFuture scheduleAtFixedRate(@NonNull Long msPeriod, @NonNull Runnable func) {
        return scheduleAtFixedRate(
            XScheduleParam.builder()
                .msPeriod(msPeriod)
                .func(func)
                .build()
        );
    }

    public XScheduleFuture scheduleAtFixedRate(@NonNull XScheduleParam param) {
        XEx.checkNotNull(param, "func", "msPeriod");
        Long msPeriod = param.getMsPeriod();
        Long msDelayInit = param.getMsDelayInit();
        if (msDelayInit == null) {
            msDelayInit = 0L;
        }
        Runnable func = param.getFunc();
        return new XScheduleFuture(service.scheduleAtFixedRate(func::run, msDelayInit, msPeriod, TimeUnit.MILLISECONDS));
    }

    public XScheduleFuture scheduleWithFixedDelay(@NonNull Long msDelay, @NonNull Runnable func) {
        return scheduleWithFixedDelay(
            XScheduleParam.builder()
                .msDelay(msDelay)
                .func(func)
                .build()
        );
    }

    public XScheduleFuture scheduleWithFixedDelay(@NonNull XScheduleParam param) {
        XEx.checkNotNull(param, "func", "msDelay");
        Runnable func = param.getFunc();
        Long msDelay = param.getMsDelay();
        Long msDelayInit = param.getMsDelayInit();
        if (msDelayInit == null) {
            msDelayInit = 0L;
        }
        return new XScheduleFuture(service.scheduleWithFixedDelay(func::run, msDelayInit, msDelay, TimeUnit.MILLISECONDS));
    }

}
