package org.xyou.xcommon.thread;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import org.xyou.xcommon.base.XBaseObject;
import org.xyou.xcommon.config.XConfig;
import org.xyou.xcommon.profiler.XProfiler;
import org.xyou.xcommon.profiler.XProfilerObj;
import org.xyou.xcommon.time.XTime;

import lombok.Getter;
import lombok.NonNull;

public final class XThreadPool extends XBaseObject {

    private static final long serialVersionUID = 1L;

    @Getter
    private String name;
    @Getter
    private Integer numThread;
    @Getter
    private Integer sizeQueueMax;
    @Getter
    private Boolean isProfiler;
    private transient ThreadPoolExecutor pool;
    private transient XProfilerObj objProfiler;

    public XThreadPool(@NonNull Integer numThread, @NonNull Integer sizeQueueMax) {
        name = null;
        this.numThread = numThread;
        this.sizeQueueMax = sizeQueueMax;
        isProfiler = false;
        init(numThread, sizeQueueMax);
    }

    public XThreadPool(@NonNull String name) {
        XConfig config = new XConfig(name);
        this.name = name;
        numThread = config.getInt("numThread");
        sizeQueueMax = config.getInt("sizeQueueMax");
        isProfiler = config.getBool("isProfiler", false);
        init(numThread, sizeQueueMax);
        if (isProfiler) {
            objProfiler = XProfiler.createObj(this);
            objProfiler.scheduleGauge("size", this::sizeQueue);
        }
    }

    private void init(@NonNull Integer numThread, @NonNull Integer sizeQueueMax) {
        BlockingQueue<Runnable> queue = new LinkedBlockingDeque<Runnable>(sizeQueueMax);
        pool = new ThreadPoolExecutor(
            numThread,
            numThread,
            XTime.MS_MIN,
            TimeUnit.MILLISECONDS,
            queue
        );
    }

    public int sizeQueue() {
        return pool.getQueue().size();
    }

    public XThreadFuture submit(@NonNull Runnable func) {
        return new XThreadFuture(pool.submit(func::run));
    }

    public XThreadFuture submit(@NonNull Supplier<?> func) {
        return new XThreadFuture(pool.submit(func::get));
    }

    public boolean shutdown() {
        return shutdown(Long.MAX_VALUE);
    }

    public boolean shutdown(@NonNull Long msWait) {
        try {
            pool.shutdown();
            return pool.awaitTermination(msWait, TimeUnit.MILLISECONDS);
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }
}
