package org.xyou.xcommon.yaml;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.sql.Timestamp;
import java.util.LinkedHashMap;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonAutoDetect.Visibility;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.std.StdSerializer;
import com.fasterxml.jackson.databind.type.MapType;

import org.xyou.xcommon.entity.XObj;
import org.xyou.xcommon.ex.XEx;
import org.xyou.xcommon.file.XFile;
import org.xyou.xcommon.time.XTime;

import lombok.AccessLevel;
import lombok.Getter;
import lombok.NonNull;

final class Model {

    @Getter(AccessLevel.PACKAGE)
    private final transient ObjectMapper mapper;

    Model(ObjectMapper mapper) {
        this.mapper = mapper;
        mapper.setSerializationInclusion(Include.NON_NULL);
        mapper.setVisibility(PropertyAccessor.ALL, Visibility.NONE);
        mapper.setVisibility(PropertyAccessor.FIELD, Visibility.ANY);
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        mapper.configure(DeserializationFeature.FAIL_ON_IGNORED_PROPERTIES, false);
        SimpleModule module = new SimpleModule();
        module.addSerializer(Timestamp.class, new StdSerializer<Timestamp>(Timestamp.class) {

            private static final long serialVersionUID = 1L;

            @Override
            public void serialize(Timestamp value, JsonGenerator gen, SerializerProvider provider) throws IOException {
                gen.writeNumber(value.getTime() / XTime.MS_SEC);
            }

        });
        module.addDeserializer(Timestamp.class, new StdDeserializer<Timestamp>(Timestamp.class) {

            private static final long serialVersionUID = 1L;

            @Override
            public Timestamp deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JsonProcessingException {
                return new Timestamp(p.getValueAsLong() * XTime.MS_SEC);
            }

        });
        module.addSerializer(XObj.class, new StdSerializer<XObj>(XObj.class) {

            private static final long serialVersionUID = 1L;

            @Override
            public void serialize(XObj value, JsonGenerator gen, SerializerProvider provider) throws IOException {
                Map<Object, Object> map = value.getMap();
                gen.writeObject(map);
            }
        });
        module.addDeserializer(XObj.class, new StdDeserializer<XObj>(XObj.class) {

            private static final long serialVersionUID = 1L;

            @Override
            public XObj deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JsonProcessingException {
                @SuppressWarnings("unchecked")
                Map<Object, Object> map = p.readValueAs(Map.class);
                return new XObj().putAll(map);
            }
        });
        mapper.registerModule(module);
    }

    String toStr(@NonNull Object obj) {
        try {
            return mapper.writeValueAsString(obj);
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    <V> V fromStr(String data, Class<V> cls) {
        try {
            return mapper.readValue(data, cls);
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    <V> Map<String, V> fromStrToMap(@NonNull String data, @NonNull Class<V> cls) {
        try {
            MapType mapType = mapper.getTypeFactory().constructMapType(
                LinkedHashMap.class,
                String.class,
                cls
            );
            return mapper.readValue(data, mapType);
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    boolean toFile(@NonNull String path, @NonNull Object obj) {
        try {
            XFile.mkdir(XFile.dirname(path));
            mapper.writeValue(new File(path), obj);
            return true;
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    <V> V fromFile(@NonNull Object obj, @NonNull Class<V> cls) {
        try {
            if (obj instanceof String) {
                return mapper.readValue(new File((String) obj), cls);
            }
            if (obj instanceof File) {
                return mapper.readValue((File) obj, cls);
            }
            if (obj instanceof InputStream) {
                return mapper.readValue((InputStream) obj, cls);
            }
            if (obj instanceof URL) {
                return mapper.readValue(((URL) obj), cls);
            }
            throw XEx.createClassInvalid(obj);
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    <V> Map<String, V> fromFileToMap(@NonNull Object obj, @NonNull Class<V> cls) {
        try {
            MapType mapType = mapper.getTypeFactory().constructMapType(
                LinkedHashMap.class,
                String.class,
                cls
            );
            if (obj instanceof String) {
                return mapper.readValue(new File((String) obj), mapType);
            }
            if (obj instanceof File) {
                return mapper.readValue((File) obj, mapType);
            }
            if (obj instanceof InputStream) {
                return mapper.readValue((InputStream) obj, mapType);
            }
            if (obj instanceof URL) {
                return mapper.readValue((URL) obj, mapType);
            }
            throw XEx.createClassInvalid(obj);
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

}
