package org.yunchen.gb.plugin.springsecurity

import org.yunchen.gb.core.config.ServerConfigurationProperties
import org.yunchen.gb.plugin.springsecurity.config.SpringSecurityConfigurationProperties
import org.yunchen.gb.plugin.springsecurity.crypto.password.CustomPasswordEncoder
import org.yunchen.gb.plugin.springsecurity.crypto.password.GbDelegatingPasswordEncoder
import org.yunchen.gb.plugin.springsecurity.crypto.password.Sm3PasswordEncoder
import org.yunchen.gb.plugin.springsecurity.crypto.password.Sm4PasswordEncoder
import org.yunchen.gb.plugin.springsecurity.userdetails.DefaultPostAuthenticationChecks
import org.yunchen.gb.plugin.springsecurity.userdetails.DefaultPreAuthenticationChecks
import org.yunchen.gb.plugin.springsecurity.userdetails.GbUserDetailsService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.boot.web.servlet.ServletListenerRegistrationBean
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Configuration
import org.springframework.core.annotation.Order
import org.springframework.expression.ExpressionParser
import org.springframework.expression.spel.standard.SpelExpressionParser
import org.springframework.security.access.AccessDecisionManager
import org.springframework.security.access.AccessDecisionVoter
import org.springframework.security.access.PermissionEvaluator
import org.springframework.security.access.expression.DenyAllPermissionEvaluator
import org.springframework.security.access.expression.SecurityExpressionHandler
import org.springframework.security.access.hierarchicalroles.RoleHierarchy
import org.springframework.security.access.hierarchicalroles.RoleHierarchyAuthoritiesMapper
import org.springframework.security.access.hierarchicalroles.RoleHierarchyImpl
import org.springframework.security.access.vote.AuthenticatedVoter
import org.springframework.security.access.vote.RoleHierarchyVoter
import org.springframework.security.authentication.AnonymousAuthenticationToken
import org.springframework.security.authentication.AuthenticationDetailsSource
import org.springframework.security.authentication.AuthenticationTrustResolver
import org.springframework.security.authentication.AuthenticationTrustResolverImpl
import org.springframework.security.authentication.RememberMeAuthenticationToken
import org.springframework.security.authentication.dao.DaoAuthenticationProvider

import org.springframework.security.core.session.SessionRegistry
import org.springframework.security.core.session.SessionRegistryImpl
import org.springframework.security.core.userdetails.AuthenticationUserDetailsService
import org.springframework.security.core.userdetails.UserCache
import org.springframework.security.core.userdetails.UserDetailsByNameServiceWrapper
import org.springframework.security.core.userdetails.UserDetailsChecker
import org.springframework.security.core.userdetails.UserDetailsService
import org.springframework.security.core.userdetails.cache.NullUserCache
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder
import org.springframework.security.crypto.password.PasswordEncoder
import org.springframework.security.crypto.password.Pbkdf2PasswordEncoder
import org.springframework.security.crypto.scrypt.SCryptPasswordEncoder
import org.springframework.security.web.PortMapper
import org.springframework.security.web.PortMapperImpl
import org.springframework.security.web.PortResolver
import org.springframework.security.web.PortResolverImpl
import org.springframework.security.web.RedirectStrategy
import org.springframework.security.web.access.expression.DefaultWebSecurityExpressionHandler
import org.springframework.security.web.access.intercept.FilterInvocationSecurityMetadataSource
import org.springframework.security.web.authentication.WebAuthenticationDetailsSource
import org.springframework.security.web.savedrequest.HttpSessionRequestCache
import org.springframework.security.web.savedrequest.RequestCache
import org.springframework.security.web.session.HttpSessionEventPublisher
import org.springframework.security.web.util.ThrowableAnalyzer
import org.springframework.security.web.util.matcher.AnyRequestMatcher
import org.springframework.security.web.util.matcher.RequestMatcher
import org.springframework.stereotype.Component
import org.yunchen.gb.plugin.springsecurity.access.vote.AuthenticatedVetoableDecisionManager
import org.yunchen.gb.plugin.springsecurity.dao.provider.GbDaoAuthenticationProvider
import org.yunchen.gb.plugin.springsecurity.web.GbRedirectStrategy
import org.yunchen.gb.plugin.springsecurity.web.access.DefaultThrowableAnalyzer
import org.yunchen.gb.plugin.springsecurity.web.access.expression.WebExpressionVoter
import org.yunchen.gb.plugin.springsecurity.web.access.intercept.RequestmapFilterInvocationDefinition

/**
 * Created by @Author:xiaopeng on 2017/6/27.
 *  * 2019-08-08 在 3.6.0 版本中增加GlobalAuthority类处理用户及角色数据,
 *  * 在初始化passwordEncoder这个bean的时候初始化GlobalAuthority
 */
@Order(21)
@Configuration
@EnableConfigurationProperties([ServerConfigurationProperties.class,SpringSecurityConfigurationProperties.class])
@Component
class GbSpringSecurityBean {
    //用于保存全部支持的passwordEncoder
    public Map<String, PasswordEncoder> encoders;
    @Autowired
    private ServerConfigurationProperties serverproperties;
    @Autowired
    private SpringSecurityConfigurationProperties securityproperties;
    //session注册类
    @Bean("sessionRegistry")
    SessionRegistry getSessionRegistry(){
        return new SessionRegistryImpl();
    }
    // Register HttpSessionEventPublisher
    // httpsession的时间发布器，当session创建和销毁时通知sessionRegistry
    @Bean('httpSessionEventPublisher')
    public ServletListenerRegistrationBean getHttpSessionEventPublisher() {
        return new ServletListenerRegistrationBean(new HttpSessionEventPublisher());
    }
    public Map getSupportPasswordEncoder(){
        Map<String, PasswordEncoder> encoders = new HashMap<>();
        encoders.put("bcrypt", new BCryptPasswordEncoder());
        encoders.put("ldap", new org.springframework.security.crypto.password.LdapShaPasswordEncoder());
        encoders.put("MD4", new org.springframework.security.crypto.password.Md4PasswordEncoder());
        encoders.put("MD5", new org.springframework.security.crypto.password.MessageDigestPasswordEncoder("MD5"));
        encoders.put("noop", org.springframework.security.crypto.password.NoOpPasswordEncoder.getInstance());
        encoders.put("pbkdf2", new Pbkdf2PasswordEncoder());
        encoders.put("scrypt", new SCryptPasswordEncoder());
        encoders.put("SHA-1", new org.springframework.security.crypto.password.MessageDigestPasswordEncoder("SHA-1"));
        encoders.put("SHA-256", new org.springframework.security.crypto.password.MessageDigestPasswordEncoder("SHA-256"));
        encoders.put("SHA-512", new org.springframework.security.crypto.password.MessageDigestPasswordEncoder("SHA-512"));
        encoders.put("SHA-384", new org.springframework.security.crypto.password.MessageDigestPasswordEncoder("SHA-384"));
        encoders.put("SHA-224", new org.springframework.security.crypto.password.MessageDigestPasswordEncoder("SHA-224"));
        encoders.put("sha256", new org.springframework.security.crypto.password.StandardPasswordEncoder());
        encoders.put("SM3", new Sm3PasswordEncoder());
        encoders.put("SM4", new Sm4PasswordEncoder(securityproperties.password.sm4Key));
        encoders.put("custom", new CustomPasswordEncoder());
        this.encoders=encoders;
        return encoders;
    }
    //加密算法
    @Bean("passwordEncoder")
    public PasswordEncoder  getPasswordEncoder(){
        String algorithm=securityproperties.getPassword().getAlgorithm();
        //return PasswordEncoderFactories.createDelegatingPasswordEncoder();
        String encodingId = "bcrypt";
        Map<String, PasswordEncoder> encoders = getSupportPasswordEncoder();
        if(encoders.containsKey(algorithm)){
            encodingId=algorithm;
        }
        PasswordEncoder passwordEncoder=new GbDelegatingPasswordEncoder(encodingId, encoders);
        if(securityproperties.authorityWithoutDB){
            securityproperties.authorityMaps.each{ SpringSecurityConfigurationProperties.AuthorityUser one->
                Map user=[:]
                user.id=one.id
                user.username=one.username
                user.password=passwordEncoder.encode(one.password)
                user.enabled=one.enabled
                user.accountExpired = false
                user.accountLocked = false
                user.passwordExpired = false
                user.authorities = one.authorities.collect {[id:it,authority:it,description:it]}
                GlobalAuthority.users << user
                GlobalAuthority.userMaps[user.username]=user
                user.authorities.each{Map role->
                    if(!GlobalAuthority.roles.contains(role)){
                        GlobalAuthority.roles << role
                    }
                    GlobalAuthority.userRoles << [user:(user.username),role:role.id]
                }
            }
        }else{
/*            GlobalAuthority.users=[]
            GlobalAuthority.roles=[]
            GlobalAuthority.userRoles=[]
            GlobalAuthority.userMaps=[:]*/
        }
        return passwordEncoder
    }



    //自定义的userDetailsService
    @Bean("userDetailsService")
    public UserDetailsService  getUserDetailsService(){

        return   new GbUserDetailsService();
    }

    @Bean("authenticationUserDetailsService")
    public AuthenticationUserDetailsService  getAuthenticationUserDetailsService(){
        return   new UserDetailsByNameServiceWrapper(getUserDetailsService());
    }
    @Bean("authenticationDetailsSource")
    public AuthenticationDetailsSource  getAuthenticationDetailsSource(){
        return   new WebAuthenticationDetailsSource();
    }
    @Bean("portMapper")
    public PortMapper  getPortMapper(){
        PortMapper portMapper= new PortMapperImpl();
        //portMapper.portMappings=[8080:8443];
        String port=serverproperties.getPort()?:'8080';
        if(port=='-1'){
            port="8080";
        }
        String httpsPort=serverproperties.getHttpsport()?:'8443';
        Map<String,String> map=new HashMap<String,String>();
        map.put(port,httpsPort);
        portMapper.setPortMappings(map);
        return portMapper;
    }
    @Bean("portResolver")
    public PortResolver  getPortResolver(){
        return   new PortResolverImpl(portMapper:getPortMapper());
    }
    //跳转策略
    @Bean("redirectStrategy")
    public RedirectStrategy  getRedirectStrategy(){
        return   new GbRedirectStrategy(useHeaderCheckChannelSecurity:false,portResolver:getPortResolver());
    }

    @Bean("requestMatcher")
    public RequestMatcher  getRequestMatcher(){
        return   new AnyRequestMatcher();
    }
    @Bean("requestCache")
    public RequestCache  getRequestCache(){
        return   new HttpSessionRequestCache(portResolver: getPortResolver(),createSessionAllowed: true,requestMatcher:getRequestMatcher());
    }
    @Bean("securityMetadataSource")
    public FilterInvocationSecurityMetadataSource  getSecurityMetadataSource(){
        FilterInvocationSecurityMetadataSource obj=new RequestmapFilterInvocationDefinition(rejectIfNoRule: true);
        //obj.reset();
        return obj;
    }

/*



    @Bean("anonymousAuthenticationProvider")
    public AuthenticationProvider  getAnonymousAuthenticationProvider(){
        return   new GbAnonymousAuthenticationProvider();
    }*/
    //add by xiaopeng to support reauthenticate method check password

    @Bean("preAuthenticationChecks")
    public UserDetailsChecker  getPreAuthenticationChecks(){
        return   new DefaultPreAuthenticationChecks();
    }
    @Bean("postAuthenticationChecks")
    public UserDetailsChecker  getPostAuthenticationChecks(){
        return   new DefaultPostAuthenticationChecks();
    }
    @Bean("authoritiesMapper")
    public RoleHierarchyAuthoritiesMapper  getAuthoritiesMapper(){
        return   new RoleHierarchyAuthoritiesMapper(getRoleHierarchy());
    }
    @Bean("userCache")
    public UserCache  getUserCache(){
        return   new NullUserCache();
    }

    @Bean("daoAuthenticationProvider")
    public DaoAuthenticationProvider  getDaoAuthenticationProvider(){
        DaoAuthenticationProvider daoAuthenticationProvider=new GbDaoAuthenticationProvider();
        daoAuthenticationProvider.userDetailsService=getUserDetailsService();
        daoAuthenticationProvider.passwordEncoder=getPasswordEncoder();
        daoAuthenticationProvider.userCache=getUserCache();
        daoAuthenticationProvider.preAuthenticationChecks=getPreAuthenticationChecks();
        daoAuthenticationProvider.postAuthenticationChecks=getPostAuthenticationChecks();
        daoAuthenticationProvider.authoritiesMapper=getAuthoritiesMapper();
        daoAuthenticationProvider.hideUserNotFoundExceptions=true;
        return daoAuthenticationProvider;
    }

    @Bean("expressionParser")
    public ExpressionParser  getExpressionParser(){
        return new SpelExpressionParser();
    }
    @Bean("permissionEvaluator")
    public PermissionEvaluator  getPermissionEvaluator(){
        return new DenyAllPermissionEvaluator();
    }

    @Bean("throwableAnalyzer")
    public ThrowableAnalyzer  getThrowableAnalyzer(){
        return new DefaultThrowableAnalyzer();
    }

/*    @Bean("exceptionTranslationFilter")
    public ExceptionTranslationFilter  getExceptionTranslationFilter(){
        UpdateRequestContextHolderExceptionTranslationFilter updateRequestContextHolderExceptionTranslationFilter=new UpdateRequestContextHolderExceptionTranslationFilter(getAuthenticationEntryPoint(),getRequestCache());
        updateRequestContextHolderExceptionTranslationFilter.accessDeniedHandler=getAccessDeniedHandler();
        updateRequestContextHolderExceptionTranslationFilter.authenticationTrustResolver=getAuthenticationTrustResolver();
        updateRequestContextHolderExceptionTranslationFilter.throwableAnalyzer=getThrowableAnalyzer();
        return updateRequestContextHolderExceptionTranslationFilter;
    }*/


    /*
    //@todo 不允许配置在bean中AuthenticationEntryPoint
    //认证登录点
    //@Bean("authenticationEntryPoint")
    public AuthenticationEntryPoint  getAuthenticationEntryPoint(){
        AjaxAwareAuthenticationEntryPoint aaaep= new AjaxAwareAuthenticationEntryPoint(securityproperties.getAuth().getLoginFormUrl());
        aaaep.ajaxLoginFormUrl=securityproperties.getAuth().getAjaxLoginFormUrl();
        aaaep.forceHttps=false;
        aaaep.useForward=securityproperties.getAuth().getUseForward();
        aaaep.portMapper=getPortMapper();
        aaaep.portResolver=getPortResolver();
        aaaep.redirectStrategy=getRedirectStrategy();
        return aaaep;
    }*/
    @Bean("authenticatedVoter")
    public AccessDecisionVoter  getAuthenticatedVoter(){
        return  new AuthenticatedVoter();
    }
    @Bean("roleHierarchy")
    public RoleHierarchy  getRoleHierarchy(){
        return new RoleHierarchyImpl();
    }
    @Bean("roleVoter")
    public AccessDecisionVoter  getRoleHierarchyVoter(){
        return  new RoleHierarchyVoter(getRoleHierarchy());
    }

    //鉴别权限管理器
    @Bean("accessDecisionManager")
    public AccessDecisionManager  getAccessDecisionManager(){
        List<AccessDecisionVoter> list= new ArrayList<AccessDecisionVoter>();
        //@todo 无法获取配置信息，初始化早于environment
        String voterNames//=environment.getProperty("gb.springsecurity.voterNames");
        if(!voterNames || !voterNames?.trim()){
            voterNames="authenticatedVoter,roleVoter,webExpressionVoter"//,closureVoter"; //
        }
/*        voterNames.tokenize(",").each{name->
            list<< applicationContext.getBean(name);
        }*/
        list<<getAuthenticatedVoter();
        list<< getRoleHierarchyVoter();
        list << getWebExpressionVoter();
        AuthenticatedVetoableDecisionManager authenticatedVetoableDecisionManager=new AuthenticatedVetoableDecisionManager(list);
        authenticatedVetoableDecisionManager.allowIfAllAbstainDecisions=false;
        return authenticatedVetoableDecisionManager;
    }
/*    @Bean("accessDecisionManager")
    public AccessDecisionManager  getAccessDecisionManager(){
        return new GbAccessDecisionManager();
    }*/

    @Bean("authenticationTrustResolver")
    public AuthenticationTrustResolver  getAuthenticationTrustResolver(){
        return new AuthenticationTrustResolverImpl(anonymousClass: AnonymousAuthenticationToken,rememberMeClass: RememberMeAuthenticationToken);
    }
    //不用自定义匿名角色
/*    @Bean("anonymousAuthenticationProvider")
    public AuthenticationProvider  getAnonymousAuthenticationProvider(){
        return new GbAnonymousAuthenticationProvider();
    }*/

    @Bean("webExpressionVoter")
    public WebExpressionVoter  getWebExpressionVoter(){
        return new WebExpressionVoter(expressionHandler:getWebExpressionHandler());
    }
    @Bean("webExpressionHandler")
    public SecurityExpressionHandler  getWebExpressionHandler(){
        RoleHierarchyImpl roleHierarchy=getRoleHierarchy();
        roleHierarchy.setHierarchy('');
        SecurityExpressionHandler obj=new DefaultWebSecurityExpressionHandler(expressionParser:getExpressionParser(),
                permissionEvaluator:getPermissionEvaluator(),roleHierarchy:roleHierarchy,trustResolver:getAuthenticationTrustResolver());

        return obj;
    }
}
