package org.yunchen.gb.plugin.springsecurity

import org.springframework.beans.factory.annotation.Autowired
import org.yunchen.gb.core.GbSpringUtils
import org.yunchen.gb.plugin.springsecurity.crypto.password.GbDelegatingPasswordEncoder
import org.yunchen.gb.plugin.springsecurity.userdetails.CoreUser
import org.springframework.security.authentication.AuthenticationTrustResolver

import org.springframework.security.core.Authentication
import org.springframework.security.crypto.password.PasswordEncoder
import org.springframework.security.web.access.intercept.FilterInvocationSecurityMetadataSource
import org.springframework.stereotype.Service
import org.springframework.security.core.context.SecurityContextHolder as SCH
import javax.servlet.http.HttpServletRequest

/**
 * Created by @Author:xiaopeng on 2017/6/22.
 */
@Service
class GbSpringSecurityService {

    protected static final List<String> NO_SALT = ['bcrypt', 'pbkdf2']

    /**
     * Get the currently logged in user's principal. If not authenticated and the
     * AnonymousAuthenticationFilter is active (true by default) then the anonymous
     * user's name will be returned ('anonymousUser' unless overridden).
     *
     * @return the principal
     */
    def getPrincipal() { getAuthentication()?.principal }

    /**
     * Get the currently logged in user's <code>Authentication</code>. If not authenticated
     * and the AnonymousAuthenticationFilter is active (true by default) then the anonymous
     * user's auth will be returned (AnonymousAuthenticationToken with username 'anonymousUser'
     * unless overridden).
     *
     * @return the authentication
     */
    Authentication getAuthentication() { SCH.context?.authentication }

    /**
     * Get the domain class instance associated with the current authentication.
     * @return the user
     */
    def getCurrentUser() {
        if (!isLoggedIn()) {
            return null
        }

        def User = getClassForName(GbSpringUtils.getConfiginfo("gb.springsecurity.userLookup.userDomainClassName"))

        if (principal instanceof CoreUser) {
            User.get principal.id
        }
        else {
            String usernamePropertyName = "username"

            User.createCriteria().get {
/*                if(securityConfig.userLookup.usernameIgnoreCase) {
                    eq(usernamePropertyName, principal[usernamePropertyName], [ignoreCase: true])
                } else {
                    eq(usernamePropertyName, principal[usernamePropertyName])
                }*/
                eq(usernamePropertyName, principal[usernamePropertyName])
                cache true
            }
        }
    }

    protected Class<?> getClassForName(String name) {
        GbSpringUtils.getDomain(name);
    }

    protected ConfigObject getSecurityConfig() { GbSpringSecurityUtils.securityConfig }

    protected boolean useRequestmaps() { GbSpringSecurityUtils.securityConfigType == 'Requestmap' }

    def getCurrentUserId() {
        def principal = getPrincipal()
        principal instanceof CoreUser ? principal.id : null
    }

    /**
     * Get a proxy for the domain class instance associated with the current authentication. Use this when you
     * want the user only for its id, e.g. as a proxy for the foreign key in queries like "CreditCard.findAllByUser(user)"
     *
     * @return the proxy
     */
    def loadCurrentUser() {
        if (!isLoggedIn()) {
            return null
        }

        // load() requires an id, so this only works if there's an id property in the principal
        assert principal instanceof CoreUser

        getClassForName(GbSpringUtils.getConfiginfo("gb.springsecurity.userLookup.userDomainClassName")).load(currentUserId)
    }

    /**
     * Encode the password using the configured PasswordEncoder.
     */
    String encodePassword(String password, salt = null) {
        if (GbSpringUtils.getConfiginfo("gb.springsecurity.password.algorithm") in NO_SALT) {
            salt = null
        }
        PasswordEncoder passwordEncoder=GbSpringUtils.getBean("passwordEncoder");
        //passwordEncoder.encodePassword password, salt
        passwordEncoder.encode(password)
    }

    /**
     * Quick check to see if the current user is logged in.
     * @return <code>true</code> if the user is authenticated and not anonymous
     */
    boolean isLoggedIn() {
        def authentication = SCH.context.authentication
        AuthenticationTrustResolver authenticationTrustResolver=GbSpringUtils.getBean("authenticationTrustResolver");
        authentication && !authenticationTrustResolver.isAnonymous(authentication)
    }

    /**
     * Call when editing, creating, or deleting a Requestmap to flush the cached
     * configuration and rebuild using the most recent data.
     */
    void clearCachedRequestmaps() {
        FilterInvocationSecurityMetadataSource securityMetadataSource=GbSpringUtils.getBean("securityMetadataSource");
        securityMetadataSource?.reset();
        //log.trace 'Cleared cached requestmaps'
    }

    /**
     * Call for reloading the role hierarchy configuration from the database.
     * @author fpape
     */
/*    void reloadDBRoleHierarchy() {
        //@TODO目前不支持
        Class roleHierarchyEntryClass = Class.forName(securityConfig.roleHierarchyEntryClassName)
        roleHierarchyEntryClass.withTransaction {
            String hierarchy = roleHierarchyEntryClass.list()*.entry.join('\n')
            //log.trace 'Loaded persistent role hierarchy {}', hierarchy
            grailsApplication.mainContext.roleHierarchy.hierarchy = hierarchy
        }
    }*/

    /**
     * Delete a role, and if Requestmap class is used to store roles, remove the role
     * from all Requestmap definitions. If a Requestmap's config attribute is this role,
     * it will be deleted.
     *
     * @param role the role to delete
     */

    void deleteRole(role) {
        //def conf = securityConfig
        String configAttributePropertyName = "configAttribute"
        String authorityFieldName = "authority"

        if (useRequestmaps()) {
            String roleName = role."$authorityFieldName"
            def requestmaps = findRequestmapsByRole(roleName, [:])
            for (rm in requestmaps) {
                String configAttribute = rm."$configAttributePropertyName"
                if (configAttribute == roleName) {
                    rm.delete()
                }
                else {
                    List parts = configAttribute.split(',')*.trim()
                    parts.remove roleName
                    rm."$configAttributePropertyName" = parts.join(',')
                }
            }
            clearCachedRequestmaps()
        }

        // remove the role grant from all users
        getClassForName("gb.springsecurity.userLookup.authorityJoinClassName").removeAll role

        role.delete()

        //log.trace 'Deleted role {}', role
    }

    /**
     * Update a role, and if Requestmap class is used to store roles, replace the new role
     * name in all Requestmap definitions that use it if the name was changed.
     *
     * @param role the role to update
     * @param newProperties the new role attributes ('params' from the calling controller)
     */
        //@todo 因为不再支持role.properties = newProperties方法，取消此函数
/*    boolean updateRole(role, newProperties) {
        def conf = securityConfig
        String authorityFieldName = "authority"

        String oldRoleName = role."$authorityFieldName"
        role.properties = newProperties

        if (!role.save()) {
            return false
        }

        if (!useRequestmaps()) {
            return true
        }

        String newRoleName = role."$authorityFieldName"
        if (newRoleName == oldRoleName) {
            return true
        }

        //String configAttributePropertyName = conf.requestMap.configAttributeField
        String configAttributePropertyName = "configAttribute"
        for (rm in findRequestmapsByRole(oldRoleName, conf)) {
            rm."$configAttributePropertyName" = rm."$configAttributePropertyName".replace(oldRoleName, newRoleName)
        }
        clearCachedRequestmaps()

        true
    }*/

    /**
     * Rebuild an Authentication for the given username and register it in the security context.
     * Typically used after updating a user's authorities or other auth-cached info.
     * <p/>
     * Also removes the user from the user cache to force a refresh at next login.
     *
     * @param username the user's login name
     * @param password optional
     */
    void reauthenticate(String username, String password = null) {
        GbSpringSecurityUtils.reauthenticate username, password
    }

    /**
     * Check if the request was triggered by an Ajax call.
     * @param request the request
     * @return <code>true</code> if Ajax
     */
    boolean isAjax(HttpServletRequest request) {
        GbSpringSecurityUtils.isAjax request
    }

    /**
     * Create multiple requestmap instances in a transaction.
     * @param data
     *           a list of maps where each map contains the data for one instance
     *           (configAttribute and url are required, httpMethod is optional)
     */
    void createRequestMaps(List<Map<String, Object>> data) {
        //def requestmapClass = grailsApplication.getClassForName(conf.requestMap.className)
        def requestmapClass = GbSpringUtils.getDomain(GbSpringUtils.getConfiginfo("gb.springsecurity.requestMap.className"));
        for (Map<String, Object> instanceData in data) {
            requestmapClass.newInstance(instanceData).save(failOnError: true)
        }
    }

    /**
     * Create multiple requestmap instances in a transaction that all share the same <code>configAttribute</code>.
     * @param urls a list of url patterns
     */
    void createRequestMaps(List<String> urls, String configAttribute) {
        //def requestmapClass = grailsApplication.getClassForName(conf.requestMap.className)
        def requestmapClass = GbSpringUtils.getDomain(GbSpringUtils.getConfiginfo("gb.springsecurity.requestMap.className"));
        String configAttributePropertyName = "configAttribute"
        String urlPropertyName = "url"
        for (String url in urls) {
            requestmapClass.newInstance((urlPropertyName): url, (configAttributePropertyName): configAttribute).save(failOnError: true)
        }
    }

    protected List findRequestmapsByRole(String roleName, conf) {
        getClassForName(GbSpringUtils.getConfiginfo("gb.springsecurity.requestMap.className")).withCriteria {
            like "configAttribute", "%$roleName%"
        }
    }
    @Autowired GbSpringSecurityBean gbSpringSecurityBean

    // 获取指定算法的 PasswordEncoder,目前仅支持bcrypt,pbkdf2,ldap,SHA-256,SHA-1,SHA-512,SHA-384,SHA-224,MD4,MD5,MD2,noop,scrypt,sha256,SM3,SM4,custom
    public PasswordEncoder findPasswordEncoder(String algorithm){
        Map<String, PasswordEncoder> encoders = gbSpringSecurityBean.getSupportPasswordEncoder();
        if(encoders.containsKey(algorithm)){
            return new GbDelegatingPasswordEncoder(algorithm, encoders);
        }else{
            return null;
        }
    }
}
