package org.yunchen.gb.plugin.springsecurity.web

import javax.servlet.http.HttpServletRequest
import javax.servlet.http.HttpServletResponse

import groovy.transform.CompileStatic

/**
 * Uses a {@link ThreadLocal} to store the current request and response.
 *
 * @author <a href='mailto:burt@burtbeckwith.com'>Burt Beckwith</a>
 */
@CompileStatic
final class SecurityRequestHolder {

    private static final ThreadLocal<HttpServletRequest> REQUEST_HOLDER = new ThreadLocal<HttpServletRequest>()
    private static final ThreadLocal<HttpServletResponse> RESPONSE_HOLDER = new ThreadLocal<HttpServletResponse>()

    private SecurityRequestHolder() {
        // static only
    }

    /**
     * Clear the saved request.
     */
    static void reset() {
        REQUEST_HOLDER.remove()
        RESPONSE_HOLDER.remove()
    }

    /**
     * Set the current request and response.
     * @param request the request
     * @param response the response
     */
    static void set(HttpServletRequest request, HttpServletResponse response) {
        REQUEST_HOLDER.set request
        RESPONSE_HOLDER.set response
    }

    /**
     * Get the current request.
     * @return the request
     */
    static HttpServletRequest getRequest() {
        REQUEST_HOLDER.get()
    }

    /**
     * Get the current response.
     * @return the response
     */
    static HttpServletResponse getResponse() {
        RESPONSE_HOLDER.get()
    }
}
