/* Copyright 2006-2016 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.yunchen.gb.plugin.springsecurity.web.access.intercept

import org.yunchen.gb.plugin.springsecurity.InterceptedUrl
import org.yunchen.gb.plugin.springsecurity.ReflectionUtils
import org.yunchen.gb.plugin.springsecurity.config.SpringSecurityConfigurationProperties
import groovy.transform.CompileStatic
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.context.annotation.Configuration
import org.springframework.http.HttpMethod

/**
 * @author <a href='mailto:burt@burtbeckwith.com'>Burt Beckwith</a>
 */
@Configuration
@EnableConfigurationProperties([SpringSecurityConfigurationProperties.class])
@CompileStatic
class RequestmapFilterInvocationDefinition extends AbstractFilterInvocationDefinition {
	@Autowired
	private SpringSecurityConfigurationProperties securityproperties;

	@Override
	protected void initialize() {
		if (initialized) {
			return
		}

		try {
			reset()
			initialized = true
		}
		catch (RuntimeException e) {
			log.warn("Exception initializing; this is ok if it's at startup and due " +
			         'to GORM not being initialized yet since the first web request will ' +
			         're-initialize. Error message is: {}', e.message)
		}
	}

	/**
	 * Call at startup or when <code>Requestmap</code> instances have been added, removed, or changed.
	 */
	@Override
	synchronized void reset() {
		resetConfigs()

		loadRequestmaps().each { InterceptedUrl iu -> compileAndStoreMapping(iu) }

		log.trace 'configs: {}', configAttributeMap
	}

	protected List<InterceptedUrl> loadRequestmaps() {
		boolean supportsHttpMethod = ReflectionUtils.requestmapClassSupportsHttpMethod()
		List requestmaps
		if(securityproperties.requestMapWithoutDB){
			requestmaps=securityproperties.interceptUrlMaps?.collect{[url:it.url,configAttribute:it.configAttribute,httpMethod:it.httpMethod]}
		}else{
			requestmaps=ReflectionUtils.loadAllRequestmaps()
		}
		requestmaps.collect { requestmap ->
			String urlPattern = ReflectionUtils.getRequestmapUrl(requestmap)
			String configAttribute = ReflectionUtils.getRequestmapConfigAttribute(requestmap)
			HttpMethod method = supportsHttpMethod ? ReflectionUtils.getRequestmapHttpMethod(requestmap) : null
			new InterceptedUrl(urlPattern, split(configAttribute), method)
		}

	}
}
