@file:Suppress("PARAMETER_NAME_CHANGED_ON_OVERRIDE")

package pt.lightweightform.lfkotlin.schemas

import pt.lightweightform.lfkotlin.AllowedValues
import pt.lightweightform.lfkotlin.Bound
import pt.lightweightform.lfkotlin.ComputedValue
import pt.lightweightform.lfkotlin.Context
import pt.lightweightform.lfkotlin.InitialValue
import pt.lightweightform.lfkotlin.IsRequired
import pt.lightweightform.lfkotlin.LfDate
import pt.lightweightform.lfkotlin.LfDateRange
import pt.lightweightform.lfkotlin.Schema
import pt.lightweightform.lfkotlin.StateProperty
import pt.lightweightform.lfkotlin.SyncBound
import pt.lightweightform.lfkotlin.SyncStateProperty
import pt.lightweightform.lfkotlin.Validation

@Suppress("UNCHECKED_CAST")
internal fun dateRangeSchemaImpl(
    isNullable: Boolean,
    initialValue: LfDateRange?,
    computedInitialValue: InitialValue<LfDateRange?>?,
    computedValue: ComputedValue<LfDateRange?>?,
    isClientOnly: Boolean?,
    isRequired: Boolean?,
    computedIsRequired: IsRequired?,
    allowedValues: List<LfDateRange>?,
    computedAllowedValues: AllowedValues<LfDateRange>?,
    minDate: LfDate?,
    computedMinDate: Bound<LfDate>?,
    maxDate: LfDate?,
    computedMaxDate: Bound<LfDate>?,
    validations: List<Validation<LfDateRange>>?,
    initialState: Map<String, Any?>?,
    extra: Map<String, Any?>?
): Schema<LfDateRange?> {
    val state = HashMap(initialState ?: mapOf())
    val startDateSchema: Schema<LfDate>
    val endDateSchema: Schema<LfDate>

    // When the date-range isn't computed, we apply `minDate` and `maxDate` to the inner tuple
    // elements for automatic validation
    if (computedValue == null) {
        if (minDate != null) {
            state["minDate"] = minDate
        } else if (computedMinDate != null) {
            state["minDate"] = statePropertyFromDateBound(computedMinDate)
        }
        if (maxDate != null) {
            state["maxDate"] = maxDate
        } else if (computedMaxDate != null) {
            state["maxDate"] = statePropertyFromDateBound(computedMaxDate)
        }

        startDateSchema = dateSchema(
            minDate = minDate,
            computedMinDate = computedMinDate?.let {
                dateBoundFromStateProperty(computedMinDate, "minDate")
            }
        )
        endDateSchema = dateSchema(
            computedMinDate = EndDateMinDate,
            maxDate = maxDate,
            computedMaxDate = computedMaxDate?.let {
                dateBoundFromStateProperty(computedMaxDate, "maxDate")
            })
    } else {
        startDateSchema = dateSchema()
        endDateSchema = dateSchema()
    }

    return tupleSchemaImpl(
        listOf(startDateSchema, endDateSchema) as List<Schema<Any?>>,
        isNullable,
        initialValue?.also {
            if (it.size != 2) {
                throw IllegalArgumentException("Initial value must be an array with 2 elements.")
            }
        } as Array<Any?>?,
        computedInitialValue as InitialValue<Array<Any?>?>?,
        computedValue as ComputedValue<Array<Any?>?>?,
        isClientOnly,
        isRequired,
        computedIsRequired,
        allowedValues as List<Array<Any?>>?,
        computedAllowedValues as AllowedValues<Array<Any?>>?,
        validations as List<Validation<Array<Any?>>>?,
        state,
        extra
    ) as Schema<LfDateRange?>
}

/**
 * Builds a date state property from a date bound. This state property is used by LF to query the
 * date bound of an LF date-range component and is used by the date bound validations.
 */
private fun statePropertyFromDateBound(
    computedDate: Bound<LfDate>
): StateProperty<LfDate?> = when (computedDate) {
    is SyncBound<LfDate> -> object : SyncStateProperty<LfDate?> {
        override fun Context.property(): LfDate? = computedDate.run {
            relativeContext("..").bound()
        }
    }
    else -> error("Unsupported `Bound` implementation")
}

/**
 * Builds a date bound by querying the date state property with name `dateProp`.
 */
private fun dateBoundFromStateProperty(
    computedDate: Bound<LfDate>,
    dateProp: String
): Bound<LfDate> = when (computedDate) {
    is SyncBound<LfDate> -> object : SyncBound<LfDate> {
        override fun Context.bound(): LfDate? = getStateProperty(".", dateProp)
    }
    else -> error("Unsupported `Bound` implementation")
}

/**
 * Validates that the end date of a date-range cannot be before its start date.
 */
private object EndDateMinDate : SyncBound<LfDate> {
    override fun Context.bound(): LfDate = get("0")
}

/**
 * Creates a date-range schema. Maps to a schema of type "tuple" with two inner schemas of type
 * "date" in LF.
 */
@Suppress("UNCHECKED_CAST")
public fun dateRangeSchema(
    initialValue: LfDateRange? = null,
    computedInitialValue: InitialValue<LfDateRange>? = null,
    computedValue: ComputedValue<LfDateRange>? = null,
    isClientOnly: Boolean? = null,
    allowedValues: List<LfDateRange>? = null,
    computedAllowedValues: AllowedValues<LfDateRange>? = null,
    minDate: LfDate? = null,
    computedMinDate: Bound<LfDate>? = null,
    maxDate: LfDate? = null,
    computedMaxDate: Bound<LfDate>? = null,
    validations: List<Validation<LfDateRange>>? = null,
    initialState: Map<String, Any?>? = null,
    extra: Map<String, Any?>? = null
): Schema<LfDateRange> = dateRangeSchemaImpl(
    false,
    initialValue,
    computedInitialValue,
    computedValue as ComputedValue<LfDateRange?>?,
    isClientOnly,
    null,
    null,
    allowedValues,
    computedAllowedValues,
    minDate,
    computedMinDate,
    maxDate,
    computedMaxDate,
    validations,
    initialState,
    extra
) as Schema<LfDateRange>

/**
 * Creates a nullable date-range schema. Maps to a schema of type "tuple" with `isNullable` set to
 * `true` and two inner schemas of type "date" in LF.
 */
@Suppress("UNCHECKED_CAST")
public fun nullableDateRangeSchema(
    initialValue: LfDateRange? = null,
    computedInitialValue: InitialValue<LfDateRange?>? = null,
    computedValue: ComputedValue<LfDateRange?>? = null,
    isClientOnly: Boolean? = null,
    isRequired: Boolean? = null,
    computedIsRequired: IsRequired? = null,
    allowedValues: List<LfDateRange>? = null,
    computedAllowedValues: AllowedValues<LfDateRange>? = null,
    minDate: LfDate? = null,
    computedMinDate: Bound<LfDate>? = null,
    maxDate: LfDate? = null,
    computedMaxDate: Bound<LfDate>? = null,
    validations: List<Validation<LfDateRange>>? = null,
    initialState: Map<String, Any?>? = null,
    extra: Map<String, Any?>? = null
): Schema<LfDateRange?> = dateRangeSchemaImpl(
    true,
    initialValue,
    computedInitialValue,
    computedValue,
    isClientOnly,
    isRequired,
    computedIsRequired,
    allowedValues,
    computedAllowedValues,
    minDate,
    computedMinDate,
    maxDate,
    computedMaxDate,
    validations,
    initialState,
    extra
)
