@file:OptIn(ExperimentalStdlibApi::class)

package pt.lightweightform.lfkotlin.validations

import pt.lightweightform.lfkotlin.Context
import pt.lightweightform.lfkotlin.Issue
import pt.lightweightform.lfkotlin.REPEATED_ELEMENTS_CODE
import pt.lightweightform.lfkotlin.SyncValidation
import pt.lightweightform.lfkotlin.objectOf

/**
 * Validation that ensures that an array doesn't contain duplicated elements. Returns data about the
 * first occurrence of a repeated element.
 */
public open class Unique<T> : SyncValidation<Array<T>> {
    override fun Context.validate(value: Array<T>): Iterable<Issue> = sequence {
        val map = mutableMapOf<T, Int>()
        for ((i, el) in value.withIndex()) {
            val conflict = map[el]
            if (conflict != null) {
                yield(
                    Issue(
                        REPEATED_ELEMENTS_CODE,
                        data = objectOf("indices" to listOf(conflict, i))
                    )
                )
                break
            } else {
                map[el] = i
            }
        }
    }.asIterable()
}

/**
 * Validation that ensures that an array doesn't contain duplicated elements, where the uniqueness
 * of an element is represented by its key as returned by [selector]. Returns data about the first
 * occurrence of a repeated element.
 */
public open class UniqueBy<T, K>(private val selector: (T) -> K) : SyncValidation<Array<T>> {
    override fun Context.validate(value: Array<T>): Iterable<Issue> = sequence {
        val map = mutableMapOf<K, Int>()
        for ((i, el) in value.withIndex()) {
            val key = selector(el)
            val conflict = map[key]
            if (conflict != null) {
                yield(
                    Issue(
                        REPEATED_ELEMENTS_CODE,
                        data = objectOf("indices" to listOf(conflict, i))
                    )
                )
                break
            } else {
                map[key] = i
            }
        }
    }.asIterable()
}
