package pt.lightweightform.lfkotlin

import pt.lightweightform.lfkotlin.computedvalues.SumBy
import pt.lightweightform.lfkotlin.schemas.ClassSchema
import pt.lightweightform.lfkotlin.schemas.arraySchema
import pt.lightweightform.lfkotlin.schemas.booleanSchema
import pt.lightweightform.lfkotlin.schemas.byteSchema
import pt.lightweightform.lfkotlin.schemas.classSchema
import pt.lightweightform.lfkotlin.schemas.dateRangeSchema
import pt.lightweightform.lfkotlin.schemas.dateSchema
import pt.lightweightform.lfkotlin.schemas.doubleSchema
import pt.lightweightform.lfkotlin.schemas.enumNameSchema
import pt.lightweightform.lfkotlin.schemas.enumOrdinalSchema
import pt.lightweightform.lfkotlin.schemas.floatSchema
import pt.lightweightform.lfkotlin.schemas.intSchema
import pt.lightweightform.lfkotlin.schemas.longSchema
import pt.lightweightform.lfkotlin.schemas.nullableArraySchema
import pt.lightweightform.lfkotlin.schemas.nullableBooleanSchema
import pt.lightweightform.lfkotlin.schemas.nullableByteSchema
import pt.lightweightform.lfkotlin.schemas.nullableDateRangeSchema
import pt.lightweightform.lfkotlin.schemas.nullableDateSchema
import pt.lightweightform.lfkotlin.schemas.nullableDoubleSchema
import pt.lightweightform.lfkotlin.schemas.nullableEnumNameSchema
import pt.lightweightform.lfkotlin.schemas.nullableEnumOrdinalSchema
import pt.lightweightform.lfkotlin.schemas.nullableFloatSchema
import pt.lightweightform.lfkotlin.schemas.nullableIntSchema
import pt.lightweightform.lfkotlin.schemas.nullableLongSchema
import pt.lightweightform.lfkotlin.schemas.nullableShortSchema
import pt.lightweightform.lfkotlin.schemas.nullableStringSchema
import pt.lightweightform.lfkotlin.schemas.nullableTableSchema
import pt.lightweightform.lfkotlin.schemas.nullableTupleSchema
import pt.lightweightform.lfkotlin.schemas.shortSchema
import pt.lightweightform.lfkotlin.schemas.stringSchema
import pt.lightweightform.lfkotlin.schemas.tableSchema
import pt.lightweightform.lfkotlin.schemas.tupleSchema
import pt.lightweightform.lfkotlin.validations.UniqueBy

// Data ============================================================================================

data class ExampleValue(
    var arrayOfStrings: Array<String> = arrayOf(),
    var nullableArrayOfStrings: Array<String>? = null,
    var boolean: Boolean = false,
    var nullableBoolean: Boolean? = null,
    var byte: Byte = 0,
    var nullableByte: Byte? = null,
    var dateRange: LfDateRange = arrayOf(0L.toLfDate(), 100L.toLfDate()),
    var nullableDateRange: LfDateRange? = null,
    var date: LfDate = 0L.toLfDate(),
    var nullableDate: LfDate? = null,
    var double: Double = 0.0,
    var nullableDouble: Double? = null,
    var enumName: String = "V1",
    var nullableEnumName: String? = null,
    var enumOrdinal: Int = 0,
    var nullableEnumOrdinal: Int? = null,
    var float: Float = 0f,
    var nullableFloat: Float? = null,
    var int: Int = 0,
    var nullableInt: Int? = null,
    var long: LfLong = 0.toLfLong(),
    var nullableLong: LfLong? = null,
    var short: Short = 0,
    var nullableShort: Short? = null,
    var string: String = "",
    var nullableString: String? = null,
    var table: Array<ExampleTableRow> = arrayOf(),
    var tableSum: Double = 0.0,
    var nullableTable: Array<ExampleTableRow>? = null,
    var tupleStringInt: Array<Any?> = arrayOf("", 0),
    var nullableTupleStringInt: Array<Any?>? = null
)

enum class ExampleEnum { V1, V2, V3 }

data class ExampleTableRow(
    var stringCell: String = "",
    var intCell: Int = 0
)

// Schemas =========================================================================================

@Suppress("UNCHECKED_CAST")
object ExampleValueSchema : ClassSchema<ExampleValue> by classSchema({
    childSchema(ExampleValue::arrayOfStrings, arraySchema(stringSchema(), minSize = 1, maxSize = 3))
    childSchema(
        ExampleValue::nullableArrayOfStrings,
        nullableArraySchema(stringSchema(), isRequired = true)
    )
    childSchema(ExampleValue::boolean, booleanSchema(allowedValues = listOf(true)))
    childSchema(ExampleValue::nullableBoolean, nullableBooleanSchema())
    childSchema(ExampleValue::byte, byteSchema(min = 10, max = 44))
    childSchema(ExampleValue::nullableByte, nullableByteSchema())
    childSchema(ExampleValue::dateRange, dateRangeSchema())
    childSchema(
        ExampleValue::nullableDateRange,
        nullableDateRangeSchema(minDate = 10L.toLfDate())
    )
    childSchema(
        ExampleValue::date,
        dateSchema(maxDate = 15L.toLfDate())
    )
    childSchema(ExampleValue::nullableDate, nullableDateSchema())
    childSchema(ExampleValue::double, doubleSchema(max = 99999.0))
    childSchema(ExampleValue::nullableDouble, nullableDoubleSchema())
    childSchema(ExampleValue::enumName, enumNameSchema<ExampleEnum>())
    childSchema(ExampleValue::nullableEnumName, nullableEnumNameSchema<ExampleEnum>())
    childSchema(ExampleValue::enumOrdinal, enumOrdinalSchema<ExampleEnum>())
    childSchema(ExampleValue::nullableEnumOrdinal, nullableEnumOrdinalSchema<ExampleEnum>())
    childSchema(
        ExampleValue::float,
        floatSchema(validations = listOf(MustBeOdd as Validation<Float>))
    )
    childSchema(ExampleValue::nullableFloat, nullableFloatSchema())
    childSchema(ExampleValue::int, intSchema(computedMin = Bound10WhenBooleanIsTrue))
    childSchema(
        ExampleValue::nullableInt,
        nullableIntSchema(computedMax = Bound10WhenBooleanIsTrue)
    )
    childSchema(
        ExampleValue::long,
        longSchema(validations = listOf(MustBeOdd as Validation<LfLong>))
    )
    childSchema(ExampleValue::nullableLong, nullableLongSchema())
    childSchema(
        ExampleValue::short,
        shortSchema(validations = listOf(ShouldBeEvenWhenNullableBooleanIsFalse))
    )
    childSchema(ExampleValue::nullableShort, nullableShortSchema())
    childSchema(ExampleValue::string, stringSchema(minLength = 3))
    childSchema(ExampleValue::nullableString, nullableStringSchema())
    childSchema(
        ExampleValue::table, tableSchema(
            ExampleTableRowSchema,
            maxSize = 1,
            validations = listOf(UniqueBy { row -> row.stringCell })
        )
    )
    childSchema(
        ExampleValue::tableSum,
        doubleSchema(
            computedValue = SumBy("table") { el: ExampleTableRow -> el.intCell },
            isClientOnly = false
        )
    )
    childSchema(ExampleValue::nullableTable, nullableTableSchema(ExampleTableRowSchema))
    childSchema(
        ExampleValue::tupleStringInt,
        tupleSchema(
            listOf(
                stringSchema(maxLength = 18),
                intSchema(validations = listOf(ShouldBeEvenWhenNullableBooleanIsFalse))
            ) as List<Schema<Any?>>
        )
    )
    childSchema(
        ExampleValue::nullableTupleStringInt,
        nullableTupleSchema(listOf(stringSchema(), intSchema()) as List<Schema<Any?>>)
    )
})

object ExampleTableRowSchema : ClassSchema<ExampleTableRow> by classSchema({
    childSchema(ExampleTableRow::stringCell, stringSchema())
    childSchema(ExampleTableRow::intCell, intSchema())
})

// Validations =====================================================================================

object Bound10WhenBooleanIsTrue : SyncBound<Int> {
    override fun Context.bound(): Int? {
        val boolean: Boolean = get("/boolean")
        return if (boolean) 10 else null
    }
}

object MustBeOdd : SyncValidation<Number> {
    override fun Context.validate(value: Number): Iterable<Issue> = sequence {
        if (value.toDouble() % 2 == 0.0) {
            yield(Issue("NUMBER_MUST_BE_ODD"))
        }
    }.asIterable()
}

object ShouldBeEvenWhenNullableBooleanIsFalse : SyncValidation<Number> {
    override fun Context.validate(value: Number): Iterable<Issue> = sequence {
        val nullableBoolean: Boolean? = get("/nullableBoolean")

        if (nullableBoolean == false && value.toDouble() % 2 != 0.0) {
            yield(Issue("NUMBER_MUST_BE_EVEN", isWarning = true))
        }
    }.asIterable()
}
