package pt.lightweightform.lfkotlin.internal

import pt.lightweightform.lfkotlin.AllowedValues
import pt.lightweightform.lfkotlin.Bound
import pt.lightweightform.lfkotlin.ComputedValue
import pt.lightweightform.lfkotlin.Context
import pt.lightweightform.lfkotlin.IS_REJECTED_CODE
import pt.lightweightform.lfkotlin.IsRequired
import pt.lightweightform.lfkotlin.Issue
import pt.lightweightform.lfkotlin.Path
import pt.lightweightform.lfkotlin.Schema
import pt.lightweightform.lfkotlin.SyncAllowedValues
import pt.lightweightform.lfkotlin.SyncBound
import pt.lightweightform.lfkotlin.SyncIsRequired
import pt.lightweightform.lfkotlin.SyncValidation
import pt.lightweightform.lfkotlin.Validation
import pt.lightweightform.lfkotlin.objectOf
import pt.lightweightform.lfkotlin.resolvePath

/**
 * Emits a rejected issue.
 */
internal suspend fun SequenceScope<Pair<Path, Issue>>.addRejectedIssue(path: Path, ex: Throwable) {
    yield(path to Issue(IS_REJECTED_CODE, data = objectOf("reason" to ex.toString())))
}

/**
 * Runs a computed value and emits a rejected issue in case of an error. `Unit` is returned in case
 * of an error.
 */
internal suspend fun <C> SequenceScope<Pair<Path, Issue>>.runComputedValue(
    rootSchema: Schema<*>,
    rootValue: Any?,
    state: MutableMap<Path, MutableMap<String, Any?>>,
    path: Path,
    externalContext: C,
    computedValue: ComputedValue<*>
): Any? {
    val ctx = Context(rootSchema, rootValue, state, resolvePath(path, ".."), externalContext)
    return try {
        return computedValue.run { ctx.compute() }
    } catch (ex: Throwable) {
        addRejectedIssue(path, ex)
    }
}

/**
 * Runs an "is required" validation and emits a rejected issue in case of an error.
 */
internal suspend fun <C> SequenceScope<Pair<Path, Issue>>.runIsRequired(
    rootSchema: Schema<*>,
    rootValue: Any?,
    state: MutableMap<Path, MutableMap<String, Any?>>,
    path: Path,
    externalContext: C,
    isRequired: IsRequired
): Boolean {
    val ctx = Context(rootSchema, rootValue, state, resolvePath(path, ".."), externalContext)
    return try {
        when (isRequired) {
            is SyncIsRequired -> isRequired.run { ctx.isRequired() }
            else -> error("Unsupported `IsRequired` implementation")
        }
    } catch (ex: Throwable) {
        addRejectedIssue(path, ex)
        false
    }
}

/**
 * Runs an "allowed values" validation and emits a rejected issue in case of an error.
 */
internal suspend fun <C> SequenceScope<Pair<Path, Issue>>.runAllowedValues(
    rootSchema: Schema<*>,
    rootValue: Any?,
    state: MutableMap<Path, MutableMap<String, Any?>>,
    path: Path,
    externalContext: C,
    allowedValues: AllowedValues<*>
): List<Any?>? {
    val ctx = Context(rootSchema, rootValue, state, resolvePath(path, ".."), externalContext)
    return try {
        when (allowedValues) {
            is SyncAllowedValues<*> ->
                (allowedValues as SyncAllowedValues<Any?>).run { ctx.allowedValues() }
            else -> error("Unsupported `AllowedValues` implementation")
        }
    } catch (ex: Throwable) {
        addRejectedIssue(path, ex)
        null
    }
}

/**
 * Runs a "bound" validation and emits a rejected issue in case of an error.
 */
internal suspend fun <T, C> SequenceScope<Pair<Path, Issue>>.runBound(
    rootSchema: Schema<*>,
    rootValue: Any?,
    state: MutableMap<Path, MutableMap<String, Any?>>,
    path: Path,
    externalContext: C,
    bound: Bound<T>
): T? {
    val ctx = Context(rootSchema, rootValue, state, resolvePath(path, ".."), externalContext)
    return try {
        when (bound) {
            is SyncBound<*> -> (bound as SyncBound<T>).run { ctx.bound() }
            else -> error("Unsupported `Bound` implementation")
        }
    } catch (ex: Throwable) {
        addRejectedIssue(path, ex)
        null
    }
}

/**
 * Runs a "regular" validation and emits a rejected issue in case of an error.
 */
@OptIn(ExperimentalStdlibApi::class)
internal suspend fun SequenceScope<Pair<Path, Issue>>.runValidation(
    rootSchema: Schema<*>,
    rootValue: Any?,
    state: MutableMap<Path, MutableMap<String, Any?>>,
    path: Path,
    externalContext: Any?,
    validation: Validation<*>
): Iterable<Issue>? {
    val ctx = Context(rootSchema, rootValue, state, path, externalContext)
    return try {
        @Suppress("UNCHECKED_CAST")
        when (validation) {
            is SyncValidation<*> -> (validation as SyncValidation<Any?>).run {
                ctx.validate(ctx.get("."))
            }
            else -> error("Unsupported `Validation` implementation")
        }
    } catch (ex: Throwable) {
        addRejectedIssue(path, ex)
        null
    }
}
