@file:OptIn(ExperimentalStdlibApi::class)

package pt.lightweightform.lfkotlin.internal

import pt.lightweightform.lfkotlin.COMPUTED_VALUE_DOES_NOT_MATCH_CODE
import pt.lightweightform.lfkotlin.DATE_OUT_OF_BOUNDS_CODE
import pt.lightweightform.lfkotlin.DISALLOWED_VALUE_CODE
import pt.lightweightform.lfkotlin.IS_REQUIRED_CODE
import pt.lightweightform.lfkotlin.Issue
import pt.lightweightform.lfkotlin.LENGTH_OUT_OF_BOUNDS_CODE
import pt.lightweightform.lfkotlin.LfDate
import pt.lightweightform.lfkotlin.NUMBER_OUT_OF_BOUNDS_CODE
import pt.lightweightform.lfkotlin.Path
import pt.lightweightform.lfkotlin.SIZE_OUT_OF_BOUNDS_CODE
import pt.lightweightform.lfkotlin.Schema
import pt.lightweightform.lfkotlin.appendToPath
import pt.lightweightform.lfkotlin.objectOf
import pt.lightweightform.lfkotlin.schemas.ArraySchema
import pt.lightweightform.lfkotlin.schemas.BooleanSchema
import pt.lightweightform.lfkotlin.schemas.ClassSchema
import pt.lightweightform.lfkotlin.schemas.CollectionSchema
import pt.lightweightform.lfkotlin.schemas.DateSchema
import pt.lightweightform.lfkotlin.schemas.NumberSchema
import pt.lightweightform.lfkotlin.schemas.StringSchema
import pt.lightweightform.lfkotlin.schemas.TupleSchema

/**
 * Validates [value] according to [schema] by returning a flow over all found issues.s
 */
internal fun <C> validate(
    rootSchema: Schema<*>,
    rootValue: Any?,
    state: MutableMap<Path, MutableMap<String, Any?>>,
    schema: Schema<*> = rootSchema,
    value: Any? = rootValue,
    path: Path = "/",
    externalContext: C
): Sequence<Pair<String, Issue>> = sequence {
    // Children of computed schemas shouldn't be validated
    val isComputed = schema.computedValue != null

    // Validate that the computed value matches the value in the object when `isClientOnly` is
    // `false` (the default for computed values is `true`, so `null` in fact means `true`)
    if (schema.computedValue != null && schema.isClientOnly == false) {
        val computed = runComputedValue(
            rootSchema,
            rootValue,
            state,
            path,
            externalContext,
            schema.computedValue!!
        )
        // `Unit` is returned in case of an error
        @Suppress("UNCHECKED_CAST")
        if (computed != Unit && !(schema as Schema<Any?>).valuesAreEqual(value, computed)) {
            yield(
                path to Issue(
                    COMPUTED_VALUE_DOES_NOT_MATCH_CODE,
                    data = objectOf("expected" to computed)
                )
            )
        }
    }

    // Validate `null` values
    if (schema.isNullable && value == null) {
        val isRequired = when {
            schema.isRequired != null -> schema.isRequired!!
            schema.computedIsRequired != null -> runIsRequired(
                rootSchema,
                rootValue,
                state,
                path,
                externalContext,
                schema.computedIsRequired!!
            )
            else -> false
        }
        if (isRequired) {
            yield(path to Issue(IS_REQUIRED_CODE))
        }
    } else {
        val allowedValues = when {
            schema.allowedValues != null -> schema.allowedValues
            schema.computedAllowedValues != null -> runAllowedValues(
                rootSchema,
                rootValue,
                state,
                path,
                externalContext,
                schema.computedAllowedValues!!
            )
            else -> null
        }
        // Validate "allowed values"
        if (allowedValues != null) {
            val isAllowed = allowedValues.find { allowed ->
                @Suppress("UNCHECKED_CAST")
                (schema as Schema<Any?>).valuesAreEqual(value, allowed)
            } != null

            if (!isAllowed) {
                yield(
                    path to Issue(
                        DISALLOWED_VALUE_CODE,
                        data = objectOf("value" to value, "allowedValues" to allowedValues)
                    )
                )
            }
        }

        @Suppress("UNCHECKED_CAST")
        when (schema) {
            // Validate collections
            is CollectionSchema<*> -> {
                schema as CollectionSchema<Any?>
                val size = schema.size(value)
                val minSize = when {
                    schema.minSize != null -> schema.minSize
                    schema.computedMinSize != null -> runBound(
                        rootSchema,
                        rootValue,
                        state,
                        path,
                        externalContext,
                        schema.computedMinSize
                    )
                    else -> null
                }
                val maxSize = when {
                    schema.maxSize != null -> schema.maxSize
                    schema.computedMaxSize != null -> runBound(
                        rootSchema,
                        rootValue,
                        state,
                        path,
                        externalContext,
                        schema.computedMaxSize
                    )
                    else -> null
                }
                if (
                    (minSize != null && size < minSize) ||
                    (maxSize != null && size > maxSize)
                ) {
                    yield(path to Issue(SIZE_OUT_OF_BOUNDS_CODE, data = buildMap<String, Any?> {
                        put("size", size)
                        if (minSize != null) put("minSize", minSize)
                        if (maxSize != null) put("maxSize", maxSize)
                    }))
                }

                // Validate collection children
                if (!isComputed) {
                    when (schema) {
                        is ArraySchema<*> -> {
                            value as Array<*>
                            for ((i, el) in value.withIndex()) {
                                yieldAll(
                                    validate(
                                        rootSchema,
                                        rootValue,
                                        state,
                                        schema.elementsSchema,
                                        el,
                                        appendToPath(path, "$i"),
                                        externalContext
                                    )
                                )
                            }
                        }
                        else -> error("Invalid schema")
                    }
                }
            }
            is DateSchema -> {
                value as LfDate
                val minDate = when {
                    schema.minDate != null -> schema.minDate
                    schema.computedMinDate != null -> runBound(
                        rootSchema,
                        rootValue,
                        state,
                        path,
                        externalContext,
                        schema.computedMinDate
                    )
                    else -> null
                }
                val maxDate = when {
                    schema.maxDate != null -> schema.maxDate
                    schema.computedMaxDate != null -> runBound(
                        rootSchema,
                        rootValue,
                        state,
                        path,
                        externalContext,
                        schema.computedMaxDate
                    )
                    else -> null
                }
                if (
                    (minDate != null && value < minDate) ||
                    (maxDate != null && value > maxDate)
                ) {
                    yield(path to Issue(DATE_OUT_OF_BOUNDS_CODE, data = buildMap<String, Any?> {
                        put("value", value.toString())
                        if (minDate != null) put("minDate", minDate.toString())
                        if (maxDate != null) put("maxDate", maxDate.toString())
                    }))
                }
            }
            is ClassSchema<*> -> {
                // Validate class children
                if (!isComputed) {
                    schema as ClassSchema<Any>
                    for ((prop, childSchema) in schema.childrenSchemas) {
                        yieldAll(
                            validate(
                                rootSchema,
                                rootValue,
                                state,
                                childSchema,
                                prop.get(value as Any),
                                appendToPath(path, prop.name),
                                externalContext
                            )
                        )
                    }
                }
            }
            is NumberSchema<*> -> {
                value as Comparable<Number>
                val min = when {
                    schema.min != null -> schema.min
                    schema.computedMin != null -> runBound(
                        rootSchema,
                        rootValue,
                        state,
                        path,
                        externalContext,
                        schema.computedMin
                    )
                    else -> null
                }
                val max = when {
                    schema.max != null -> schema.max
                    schema.computedMax != null -> runBound(
                        rootSchema,
                        rootValue,
                        state,
                        path,
                        externalContext,
                        schema.computedMax
                    )
                    else -> null
                }
                if (
                    (min != null && value < min) ||
                    (max != null && value > max)
                ) {
                    yield(path to Issue(NUMBER_OUT_OF_BOUNDS_CODE, data = buildMap<String, Any?> {
                        put("value", value as Number)
                        if (min != null) put("min", min)
                        if (max != null) put("max", max)
                    }))
                }
            }
            is StringSchema -> {
                value as String
                val length = value.length
                val minLength = when {
                    schema.minLength != null -> schema.minLength
                    schema.computedMinLength != null -> runBound(
                        rootSchema,
                        rootValue,
                        state,
                        path,
                        externalContext,
                        schema.computedMinLength
                    )
                    else -> null
                }
                val maxLength = when {
                    schema.maxLength != null -> schema.maxLength
                    schema.computedMaxLength != null -> runBound(
                        rootSchema,
                        rootValue,
                        state,
                        path,
                        externalContext,
                        schema.computedMaxLength
                    )
                    else -> null
                }
                if (
                    (minLength != null && length < minLength) ||
                    (maxLength != null && length > maxLength)
                ) {
                    yield(path to Issue(LENGTH_OUT_OF_BOUNDS_CODE, data = buildMap<String, Any?> {
                        put("length", length)
                        if (minLength != null) put("minLength", minLength)
                        if (maxLength != null) put("maxLength", maxLength)
                    }))
                }
            }
            is TupleSchema -> {
                // Validate tuple children
                if (!isComputed) {
                    value as Array<*>
                    for ((i, childSchema) in schema.elementsSchemas.withIndex()) {
                        yieldAll(
                            validate(
                                rootSchema,
                                rootValue,
                                state,
                                childSchema,
                                value[i],
                                appendToPath(path, "$i"),
                                externalContext
                            )
                        )
                    }
                }
            }
            !is BooleanSchema -> error("Invalid schema")
        }

        if (schema.validations != null) {
            for (validation in schema.validations!!) {
                runValidation(
                    rootSchema,
                    rootValue,
                    state,
                    path,
                    externalContext,
                    validation
                )?.forEach { issue -> yield(path to issue) }
            }
        }
    }
}
