package ru.playa.keycloak.kafka;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.util.concurrent.ExecutionException;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.jboss.logging.Logger;
import org.keycloak.events.Event;
import org.keycloak.events.EventListenerProvider;
import org.keycloak.events.admin.AdminEvent;

/**
 * Kafka event dispatch provider.
 *
 * @author Aleksandr Buldaev
 * @author Anatoliy Pokhresnyi
 */
public class KeycloakEventListener implements EventListenerProvider {

    /**
     * Logger.
     */
    private static final Logger LOGGER = Logger.getLogger(KeycloakEventListener.class);

    @Override
    public void onEvent(Event event) {
        LOGGER.info("Send event " + event.toString());

        publish(
                Configuration.getLoginEventTopic(),
                Configuration.getKafkaLoginEventProducer(),
                KeycloakEvent.of(event)
        );
    }

    @Override
    public void onEvent(AdminEvent adminEvent, boolean includeRepresentation) {
        LOGGER.info("Send event " + adminEvent.toString());

        publish(
                Configuration.getAdminEventTopic(),
                Configuration.getKafkaAdminEventProducer(),
                KeycloakAdminEvent.of(adminEvent)
        );
    }

    @Override
    public void close() {
    }

    /**
     * Sending a message to kafka.
     *
     * @param topic Topic.
     * @param producer Producer.
     * @param event Event
     * @param <T> Type of event.
     */
    private <T> void publish(String topic, KafkaProducer<String, byte[]> producer, T event) {
        try {
            byte[] message = new ObjectMapper().writeValueAsString(event).getBytes();
            ProducerRecord<String, byte[]> record = new ProducerRecord<>(topic, message);

            if (Configuration.isSync()) {
                producer.send(record).get();
            } else {
                producer.send(record);
            }
            producer.flush();
        } catch (InterruptedException | ExecutionException | JsonProcessingException e) {
            LOGGER.error(e.getMessage(), e);
        }
    }
}
