package site.howaric.encryption;

import site.howaric.encryption.exception.RSACryptogramException;

import javax.crypto.Cipher;
import java.security.*;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;

public class RSAUtil {

    /**
     * DH default is 1024
     * 64*n，between 512 and 65536
     */
    private static final int KEY_SIZE = 512;
    private static final String ALGORITHM = "RSA";

    private static final String PUBLIC_KEY = "RSAPublicKey";
    private static final String PRIVATE_KEY = "RSAPrivateKey";

    /**
     * generate key pair map
     *
     * @return key pair
     */
    public static Map<String, Object> getKeyPairMap() {
        KeyPairGenerator keyPairGenerator = null;
        try {
            keyPairGenerator = KeyPairGenerator.getInstance(ALGORITHM);
        } catch (NoSuchAlgorithmException e) {
            throw new RSACryptogramException("Get key pair failed", e);
        }
        keyPairGenerator.initialize(KEY_SIZE);
        KeyPair keyPair = keyPairGenerator.generateKeyPair();
        RSAPublicKey publicKey = (RSAPublicKey) keyPair.getPublic();
        RSAPrivateKey privateKey = (RSAPrivateKey) keyPair.getPrivate();
        Map<String, Object> keyMap = new HashMap<String, Object>();
        keyMap.put(PUBLIC_KEY, publicKey);
        keyMap.put(PRIVATE_KEY, privateKey);
        return keyMap;
    }

    /**
     * get privateKey from key pair map
     *
     * @param keyPairMap key pair map
     * @return privateKey
     */
    public static String getPrivateKey(Map<String, Object> keyPairMap) {
        Key key = (Key) keyPairMap.get(PRIVATE_KEY);
        return Base64.getEncoder().encodeToString(key.getEncoded());
    }

    /**
     * get publicKey from key pair map
     *
     * @param keyPairMap key pair map
     * @return publicKey
     */
    public static String getPublicKey(Map<String, Object> keyPairMap) {
        Key key = (Key) keyPairMap.get(PUBLIC_KEY);
        return Base64.getEncoder().encodeToString(key.getEncoded());
    }

    /**
     * encrypt data by privateKey
     *
     * @param data       target text
     * @param privateKey key pair map
     * @return encrypted data
     */
    public static String encryptByPrivateKey(String data, String privateKey) {
        PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(Base64.getDecoder().decode(privateKey));
        try {
            KeyFactory keyFactory = KeyFactory.getInstance(ALGORITHM);
            PrivateKey pKey = keyFactory.generatePrivate(pkcs8KeySpec);
            Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
            cipher.init(Cipher.ENCRYPT_MODE, pKey);
            byte[] doFinal = cipher.doFinal(data.getBytes());
            return Base64.getEncoder().encodeToString(doFinal);
        } catch (Exception e) {
            throw new RSACryptogramException("Encrypt by privateKey failed", e);
        }
    }

    /**
     * encrypt data by publicKey
     *
     * @param data      target text
     * @param publicKey key pair map
     * @return encrypted data
     */
    public static String encryptByPublicKey(String data, String publicKey) {
        try {
            KeyFactory keyFactory = KeyFactory.getInstance(ALGORITHM);
            X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(Base64.getDecoder().decode(publicKey));
            PublicKey pubKey = keyFactory.generatePublic(x509KeySpec);
            Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
            cipher.init(Cipher.ENCRYPT_MODE, pubKey);
            return Base64.getEncoder().encodeToString(cipher.doFinal(data.getBytes()));
        } catch (Exception e) {
            throw new RSACryptogramException("Encrypt by publicKey failed", e);
        }
    }

    /**
     * decrypt data by privateKey
     *
     * @param data       target text
     * @param privateKey key pair map
     * @return encrypted data
     */
    public static String decryptByPrivateKey(String data, String privateKey) {
        PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(Base64.getDecoder().decode(privateKey));
        try {
            KeyFactory keyFactory = KeyFactory.getInstance(ALGORITHM);
            PrivateKey priKey = keyFactory.generatePrivate(pkcs8KeySpec);
            Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
            cipher.init(Cipher.DECRYPT_MODE, priKey);
            return new String(cipher.doFinal(Base64.getDecoder().decode(data)));
        } catch (Exception e) {
            throw new RSACryptogramException("Decrypt by privateKey failed", e);
        }
    }

    /**
     * decrypt data by publicKey
     *
     * @param data      target text
     * @param publicKey key pair map
     * @return encrypted data
     */
    public static String decryptByPublicKey(String data, String publicKey) {
        try {
            KeyFactory keyFactory = KeyFactory.getInstance(ALGORITHM);
            X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(Base64.getDecoder().decode(publicKey));
            PublicKey pubKey = keyFactory.generatePublic(x509KeySpec);
            Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
            cipher.init(Cipher.DECRYPT_MODE, pubKey);
            return new String(cipher.doFinal(Base64.getDecoder().decode(data)));
        } catch (Exception e) {
            throw new RSACryptogramException("Decrypt by publicKey failed", e);
        }
    }

}
