package site.howaric.encryption;

import site.howaric.encryption.exception.SignatureCheckFailedException;

import java.lang.reflect.Field;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;

public class SignedData implements Signed {

    protected String nonce;
    protected String timestamp;
    protected String sign;

    private void prepare() {
        this.nonce = String.valueOf(Math.random());
        this.timestamp = String.valueOf(System.currentTimeMillis());
        this.sign = "";
    }

    @Override
    public void sign(String secretKey, SecretKeyType keyType) {
        String _sign = sign();
        switch (keyType) {
        case PUBLIC_KEY:
            this.sign = RSAUtil.encryptByPublicKey(_sign, secretKey);
            break;
        case PRIVATE_KEY:
            this.sign = RSAUtil.encryptByPrivateKey(_sign, secretKey);
            break;
        }
    }

    private void fillSignField(Field field, Map<String, String> params) {
        field.setAccessible(true);
        try {
            Object value = field.get(this);
            if (value == null || "".equals(value.toString())) {
                return;
            }
            params.put(field.getName(), value.toString());
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        }
    }

    private String sign() {
        if (sign == null) {
            prepare();
        }
        Map<String, String> params = new HashMap<>();
        Field[] parentFields = this.getClass().getSuperclass().getDeclaredFields();
        for (Field parentField : parentFields) {
            fillSignField(parentField, params);
        }
        Field[] declaredFields = this.getClass().getDeclaredFields();
        for (Field field : declaredFields) {
            fillSignField(field, params);
        }
        params.remove("sign");
        LinkedList<String> keyList = new LinkedList<>(params.keySet());
        Collections.sort(keyList);
        StringBuilder signLink = new StringBuilder();
        for (int i = 0; i < keyList.size(); i++) {
            signLink.append(keyList.get(i));
            signLink.append("=");
            signLink.append(params.get(keyList.get(i)));
        }

        return MD5Util.MD5(signLink.toString());
    }

    @Override
    public void verify(String secretKey, SecretKeyType keyType) {
        if (sign == null || "".equals(sign)) {
            throw new SignatureCheckFailedException("Sign Not Found");
        }
        String originSign = null;
        switch (keyType) {
        case PUBLIC_KEY:
            originSign = RSAUtil.decryptByPublicKey(sign, secretKey);
            break;
        case PRIVATE_KEY:
            originSign = RSAUtil.decryptByPrivateKey(sign, secretKey);
            break;
        }
        if (!originSign.equals(sign())) {
            throw new SignatureCheckFailedException("Illegal Signature");
        }
    }

    @Override
    public String getNonce() {
        return nonce;
    }

    @Override
    public String getTimestamp() {
        return timestamp;
    }

    @Override
    public String getSign() {
        return sign;
    }

}
