package com.yim.manager

import com.google.gson.Gson
import com.google.gson.reflect.TypeToken
import com.yim.core._YIMCore
import com.yim.model.YIMAttachment
import com.yim.model.YIMAttachmentExt_Audio
import com.yim.model.YIMAttachmentExt_File
import com.yim.model.YIMAttachmentExt_Image
import com.yim.model.YIMAttachmentExt_Video
import com.yim.model.YIMEnum_AttachmentType
import com.yim.model.YIMEnum_ConversationType
import com.yim.model.YIMEnum_MessageState
import com.yim.model.YIMEnum_MessageType
import com.yim.model.YIMEnum_QueryDirection
import com.yim.model.YIMMessage
import com.yim.model._YIMEnum_ConversatypeUnRead
import com.yim.model._YIMSDK
import com.yim.utils._YIMTable_Message
import com.yim.utils._YIMTable_Message_
import de.huxhorn.sulky.ulid.ULID
import io.objectbox.kotlin.notEqual
import java.io.File
import java.util.Date

class YIMMessageManager private constructor() {
    init {
        if (_YIMSDK.option == null || _YIMSDK.context == null) _YIMSDK.utils.yimLog("${javaClass.simpleName}:Please init First!")
    }

    companion object {
        internal val instance by lazy(mode = LazyThreadSafetyMode.SYNCHRONIZED) { YIMMessageManager() }
    }

    fun clearLocalMessage(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
    ) {
        if (!_YIMCore.userManager.isLogin()) return
        val isSuccess_delete = _YIMSDK.db.delete<_YIMTable_Message>(
            when (conversationType) {
                YIMEnum_ConversationType.P2P -> {
                    (_YIMTable_Message_.from.equal(_YIMSDK.currentUser).and(_YIMTable_Message_.to.equal(conversationID))).or(_YIMTable_Message_.to.equal(_YIMSDK.currentUser).and(_YIMTable_Message_.from.equal(conversationID))).and(_YIMTable_Message_.conversationType.equal(conversationType.rawValue))
                }

                YIMEnum_ConversationType.Team -> {
                    _YIMTable_Message_.to.equal(conversationID).and(_YIMTable_Message_.conversationType.equal(conversationType.rawValue))
                }

                else -> _YIMTable_Message_.id.isNull
            }
        )
        if (!isSuccess_delete) return
        _YIMCore.conversationManager.delete(conversationType, conversationID)
    }

    fun clearServerMessages(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        callback: ((isSuccess: Boolean) -> Unit)? = null,
    ) {
        if (!_YIMCore.userManager.isLogin()) {
            _YIMSDK.utils.runOnUIThread {
                callback?.invoke(false)
            }
            return
        }
        _YIMSDK.server.emit(
            "clearServerHistory", mapOf(
                "conversationType" to conversationType.rawValue,
                "conversationID" to conversationID,
            )
        ) { isSuccess, results, code, msg ->
            if (!isSuccess) {
                _YIMSDK.utils.runOnUIThread {
                    callback?.invoke(false)
                }
                return@emit
            }
            _YIMSDK.utils.runOnUIThread {
                callback?.invoke(true)
            }
        }
    }

    fun sendTextMessage(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        content: String,
        callback_start: ((message: YIMMessage) -> Unit)? = null,
        callback_finish: ((isSuccess: Boolean, message: YIMMessage, code: Int, msg: String) -> Unit)? = null,
    ) {
        _sendMessage(
            conversationType,
            conversationID,
            YIMEnum_MessageType.Text,
            content,
            callback_start = callback_start,
            callback_finish = callback_finish,
        )
    }

    fun sendAudioMessage(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        filePath: String,
        duration: Long,
        fileSize: Long? = null, //仅filePath为网络url时需要
        callback_start: ((message: YIMMessage) -> Unit)? = null,
        callback_upload: ((message: YIMMessage, count: Long, total: Long) -> Unit)? = null,
        callback_finish: ((isSuccess: Boolean, message: YIMMessage, code: Int, msg: String) -> Unit)? = null,
    ) {
        val ext = YIMAttachmentExt_Audio()
        ext.duration = duration
        val attachment = YIMAttachment()
        attachment.type = YIMEnum_AttachmentType.Audio
        attachment.url = filePath
        attachment.size = fileSize ?: File(filePath).length()
        attachment.format = filePath.split(".").last()
        attachment.ext = ext
        _sendMessage(
            conversationType,
            conversationID,
            YIMEnum_MessageType.Audio,
            attachment = attachment,
            callback_start = callback_start,
            callback_upload = callback_upload,
            callback_finish = callback_finish,
        )
    }

    fun sendImageMessage(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        filePath: String,
        compress: Boolean = true,
        fileSize: Long? = null, //仅filePath为网络url时需要
        callback_start: ((message: YIMMessage) -> Unit)? = null,
        callback_upload: ((message: YIMMessage, count: Long, total: Long) -> Unit)? = null,
        callback_finish: ((isSuccess: Boolean, message: YIMMessage, code: Int, msg: String) -> Unit)? = null,
    ) {
        val ext = YIMAttachmentExt_Image()
        ext.compress = compress
        val attachment = YIMAttachment()
        attachment.type = YIMEnum_AttachmentType.Image
        attachment.url = filePath
        attachment.size = fileSize ?: File(filePath).length()
        attachment.format = filePath.split(".").last()
        attachment.ext = ext
        _sendMessage(
            conversationType,
            conversationID,
            YIMEnum_MessageType.Image,
            attachment = attachment,
            callback_start = callback_start,
            callback_upload = callback_upload,
            callback_finish = callback_finish,
        )
    }

    fun sendVideoMessage(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        filePath: String,
        compress: Boolean = true,
        duration: Long,
        fileSize: Long? = null, //仅filePath为网络url时需要
        callback_start: ((message: YIMMessage) -> Unit)? = null,
        callback_upload: ((message: YIMMessage, count: Long, total: Long) -> Unit)? = null,
        callback_finish: ((isSuccess: Boolean, message: YIMMessage, code: Int, msg: String) -> Unit)? = null,
    ) {
        val ext = YIMAttachmentExt_Video()
        ext.compress = compress
        ext.image = _YIMSDK.compress.getVideoFirstFrame(filePath)
        ext.duration = duration
        val attachment = YIMAttachment()
        attachment.type = YIMEnum_AttachmentType.Video
        attachment.url = filePath
        attachment.size = fileSize ?: File(filePath).length()
        attachment.format = filePath.split(".").last()
        attachment.ext = ext
        _sendMessage(
            conversationType,
            conversationID,
            YIMEnum_MessageType.Video,
            attachment = attachment,
            callback_start = callback_start,
            callback_upload = callback_upload,
            callback_finish = callback_finish,
        )
    }

    fun sendFileMessage(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        filePath: String,
        fileName: String,
        fileSize: Long? = null, //仅filePath为网络url时需要
        callback_start: ((message: YIMMessage) -> Unit)? = null,
        callback_upload: ((message: YIMMessage, count: Long, total: Long) -> Unit)? = null,
        callback_finish: ((isSuccess: Boolean, message: YIMMessage, code: Int, msg: String) -> Unit)? = null,
    ) {
        val ext = YIMAttachmentExt_File()
        ext.name = fileName
        val attachment = YIMAttachment()
        attachment.type = YIMEnum_AttachmentType.File
        attachment.url = filePath
        attachment.size = fileSize ?: File(filePath).length()
        attachment.format = filePath.split(".").last()
        attachment.ext = ext
        _sendMessage(
            conversationType,
            conversationID,
            YIMEnum_MessageType.File,
            attachment = attachment,
            callback_start = callback_start,
            callback_upload = callback_upload,
            callback_finish = callback_finish,
        )
    }

    fun sendCustomMessage(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        content: String,
        callback_start: ((message: YIMMessage) -> Unit)? = null,
        callback_finish: ((isSuccess: Boolean, message: YIMMessage, code: Int, msg: String) -> Unit)? = null,
    ) {
        _sendMessage(
            conversationType,
            conversationID,
            YIMEnum_MessageType.Custom,
            content,
            callback_start = callback_start,
            callback_finish = callback_finish,
        )
    }

    private fun _sendMessage(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        messageType: YIMEnum_MessageType,
        content: String? = null,
        attachment: YIMAttachment? = null,
        callback_start: ((message: YIMMessage) -> Unit)? = null,
        callback_upload: ((message: YIMMessage, count: Long, total: Long) -> Unit)? = null,
        callback_finish: ((isSuccess: Boolean, message: YIMMessage, code: Int, msg: String) -> Unit)? = null,
    ) {
        val message = YIMMessage()
        message.conversationType = conversationType
        message.messageType = messageType
        message.content = content ?: ""
        message.attachment = attachment
        message.messageState = YIMEnum_MessageState.Sending
        message.from = _YIMSDK.currentUser!!
        message.to = conversationID
        message.time = Date().time
        message.id = when (conversationType) {
            YIMEnum_ConversationType.P2P -> "message_p2p_${ULID().nextULID()}"
            YIMEnum_ConversationType.Team -> "message_team_${ULID().nextULID()}"
            else -> ""
        }
        _YIMSDK.utils.runOnUIThread {
            callback_start?.invoke(message)
        }
        if (!_YIMCore.userManager.isLogin()) {
            message.messageState = YIMEnum_MessageState.Fail
            _YIMSDK.utils.runOnUIThread {
                callback_finish?.invoke(false, message, 0, "Please login first")
            }
            return
        }
        val isSuccess_insert = _YIMSDK.db.insert(_YIMTable_Message(message))
        if (!isSuccess_insert) {
            message.messageState = YIMEnum_MessageState.Fail
            _YIMSDK.utils.runOnUIThread {
                callback_finish?.invoke(false, message, 0, "insert msg error")
            }
            return
        }
        // 创建或修改会话最后一条消息
        val isSuccess_updateLastMessage = _YIMCore.conversationManager._updateLastMessage(conversationType, conversationID)
        if (!isSuccess_updateLastMessage) {
            message.messageState = YIMEnum_MessageState.Fail
            updateMessage(message)
            _YIMSDK.utils.runOnUIThread {
                callback_finish?.invoke(false, message, 0, "update conversation1 error")
            }
            return
        }
        _YIMSDK.utils.runOnUIThread {
            _YIMCore.listenerManager.callback_recentConversation?.invoke()
        }
        if (content != null && content.length > 4096) {
            message.messageState = YIMEnum_MessageState.Fail
            updateMessage(message)
            _YIMSDK.utils.runOnUIThread {
                callback_finish?.invoke(false, message, 0, "content size over than 4096")
            }
            val isSuccess_updateLastMessage = _YIMCore.conversationManager._updateLastMessage(conversationType, conversationID)
            if (!isSuccess_updateLastMessage) {
                _YIMSDK.utils.runOnUIThread {
                    callback_finish?.invoke(false, message, 0, "update conversation2 error")
                }
                return
            }
            _YIMSDK.utils.runOnUIThread {
                _YIMCore.listenerManager.callback_recentConversation?.invoke()
            }
            return
        }

        fun doSend() {
            _YIMSDK.server.emit(
                "sendMessage", mapOf(
                    "message" to _YIMSDK.utils.toJson(_YIMTable_Message(message)),
                )
            ) { isSuccess, results, code, msg ->
                if (isSuccess) {
                    message.messageState = YIMEnum_MessageState.Success_UnRead
                    message.time = results!!["time"].toString().toDouble().toLong()
                } else {
                    message.messageState = YIMEnum_MessageState.Fail
                }
                // 更新新消息发送状态
                updateMessage(message)
                // 更新会话最后一条消息
                val isSuccess_updateLastMessage = _YIMCore.conversationManager._updateLastMessage(conversationType, conversationID)
                if (!isSuccess_updateLastMessage) {
                    message.messageState = YIMEnum_MessageState.Fail
                    updateMessage(message)
                    _YIMSDK.utils.runOnUIThread {
                        callback_finish?.invoke(false, message, 0, "update conversation3 error")
                    }
                    return@emit
                }
                _YIMSDK.utils.runOnUIThread {
                    _YIMCore.listenerManager.callback_recentConversation?.invoke()
                }
                _YIMSDK.utils.runOnUIThread {
                    callback_finish?.invoke(isSuccess, message, code, msg)
                }
            }
        }

        if (message.attachment?.url != null) {
            fun upload(onCompress: ((url: String, onSuccess: (path: String) -> Unit) -> Unit)? = null, onSuccess: () -> Unit) {
                if (message.attachment!!.url.startsWith("http://") || message.attachment!!.url.startsWith("https://")) {
                    onSuccess()
                    return
                }

                fun doUpload() {
                    _YIMSDK.oss.upload(
                        message.attachment!!.url,
                        "${_YIMSDK.option!!.appKey}/${_YIMSDK.currentUser}/${messageType.name}",
                        { count, total ->
                            callback_upload?.invoke(message, count, total)
                        },
                        { isSuccess, url, error ->
                            if (!isSuccess) {
                                message.messageState = YIMEnum_MessageState.Fail
                                updateMessage(message)
                                _YIMSDK.utils.runOnUIThread {
                                    callback_finish?.invoke(false, message, 0, "file upload error:$error")
                                }
                                return@upload
                            }
                            message.attachment!!.url = url!!
                            updateMessage(message)
                            onSuccess()
                        },
                    )
                }

                if (onCompress != null) {
                    onCompress(message.attachment!!.url) { path ->
                        message.attachment!!.url = path
                        message.attachment!!.size = File(path).length()
                        updateMessage(message)
                        doUpload()
                    }
                } else {
                    doUpload()
                }
            }

            when (messageType) {
                YIMEnum_MessageType.Audio -> {
                    upload(onSuccess = { doSend() })
                }

                YIMEnum_MessageType.Image -> {
                    val ext = message.attachment!!.ext as YIMAttachmentExt_Image
                    upload(
                        onCompress = if (ext.compress) { url, onSuccess ->
                            _YIMSDK.compress.compressImage(url) { onSuccess(it!!) }
                        } else null,
                        onSuccess = { doSend() }
                    )
                }

                YIMEnum_MessageType.Video -> {
                    val ext = message.attachment!!.ext as YIMAttachmentExt_Video
                    upload(
                        onCompress = if (ext.compress) { url, onSuccess ->
                            _YIMSDK.compress.compressVideo(url) { onSuccess(it!!) }
                        } else null,
                        onSuccess = {
                            ext.image = _YIMSDK.oss.ossVideo2Image(message.attachment!!.url)
                            updateMessage(message)
                            doSend()
                        }
                    )
                }

                YIMEnum_MessageType.File -> {
                    upload(onSuccess = { doSend() })
                }

                else -> {}
            }
        } else {
            doSend()
        }
    }

    fun deleteLocalMessage(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        messageID: String,
    ) {
        if (!_YIMCore.userManager.isLogin()) return
        deleteLocalMessages(conversationType, conversationID, listOf(messageID))
    }

    fun deleteLocalMessages(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        messageIDs: List<String>,
    ) {
        if (!_YIMCore.userManager.isLogin()) return
        val isSuccess_delete = _YIMSDK.db.delete<_YIMTable_Message>(
            when (conversationType) {
                YIMEnum_ConversationType.P2P -> {
                    ((_YIMTable_Message_.from.equal(_YIMSDK.currentUser).and(_YIMTable_Message_.to.equal(conversationID))).or(_YIMTable_Message_.to.equal(_YIMSDK.currentUser).and(_YIMTable_Message_.from.equal(conversationID)))).and(_YIMTable_Message_.id.oneOf(messageIDs.toTypedArray())).and(_YIMTable_Message_.conversationType.equal(conversationType.rawValue))
                }

                YIMEnum_ConversationType.Team -> {
                    _YIMTable_Message_.to.equal(conversationID).and(_YIMTable_Message_.id.oneOf(messageIDs.toTypedArray())).and(_YIMTable_Message_.conversationType.equal(conversationType.rawValue))
                }

                else -> _YIMTable_Message_.id.isNull
            }
        )
        if (!isSuccess_delete) return
        val count = _YIMSDK.db.count<_YIMTable_Message>(
            when (conversationType) {
                YIMEnum_ConversationType.P2P -> {
                    (_YIMTable_Message_.from.equal(_YIMSDK.currentUser).and(_YIMTable_Message_.to.equal(conversationID))).or(_YIMTable_Message_.to.equal(_YIMSDK.currentUser).and(_YIMTable_Message_.from.equal(conversationID))).and(_YIMTable_Message_.conversationType.equal(conversationType.rawValue))
                }

                YIMEnum_ConversationType.Team -> {
                    _YIMTable_Message_.to.equal(conversationID).and(_YIMTable_Message_.conversationType.equal(conversationType.rawValue))
                }

                else -> _YIMTable_Message_.id.isNull
            }
        ) ?: return
        if (count > 0) {
            val isSuccess_updateLastMessage = _YIMCore.conversationManager._updateLastMessage(conversationType, conversationID, _YIMEnum_ConversatypeUnRead.Zero)
            if (!isSuccess_updateLastMessage) return
            _YIMSDK.utils.runOnUIThread {
                _YIMCore.listenerManager.callback_recentConversation?.invoke()
            }
        } else {
            _YIMCore.conversationManager.delete(conversationType, conversationID)
        }
    }

    fun queryLocalMessages(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        time: Date? = null,
        onlyMine: Boolean = false,
        limit: Int = 20,
        queryDirection: YIMEnum_QueryDirection = YIMEnum_QueryDirection.Old,
        callback: ((messages: List<YIMMessage>?) -> Unit),
    ) {
        if (!_YIMCore.userManager.isLogin()) {
            _YIMSDK.utils.runOnUIThread {
                callback.invoke(null)
            }
            return
        }
        val limit = if (limit > 100) 100 else limit
        val results = _YIMSDK.db.query<_YIMTable_Message>(
            queryCondition = when (conversationType) {
                YIMEnum_ConversationType.P2P -> {
                    (if (time == null) {
                        _YIMTable_Message_.time.notNull()
                    } else {
                        if (queryDirection == YIMEnum_QueryDirection.Old) {
                            _YIMTable_Message_.time.less(time.time)
                        } else {
                            _YIMTable_Message_.time.greater(time.time)
                        }
                    }).and(
                        if (onlyMine) {
                            _YIMTable_Message_.from.equal(_YIMSDK.currentUser).and(_YIMTable_Message_.to.equal(conversationID))
                        } else {
                            (_YIMTable_Message_.from.equal(_YIMSDK.currentUser).and(_YIMTable_Message_.to.equal(conversationID))).or(_YIMTable_Message_.to.equal(_YIMSDK.currentUser).and(_YIMTable_Message_.from.equal(conversationID)))
                        }
                    ).and(_YIMTable_Message_.conversationType.equal(conversationType.rawValue))
                }

                YIMEnum_ConversationType.Team -> {
                    (if (time == null) {
                        _YIMTable_Message_.time.notNull()
                    } else {
                        if (queryDirection == YIMEnum_QueryDirection.Old) {
                            _YIMTable_Message_.time.less(time.time)
                        } else {
                            _YIMTable_Message_.time.greater(time.time)
                        }
                    }).and(
                        if (onlyMine) {
                            _YIMTable_Message_.to.equal(conversationID).and(_YIMTable_Message_.from.equal(_YIMSDK.currentUser))
                        } else {
                            _YIMTable_Message_.to.equal(conversationID)
                        }
                    ).and(_YIMTable_Message_.conversationType.equal(conversationType.rawValue))
                }

                else -> null
            },
            limit = limit,
            order = _YIMTable_Message_.time,
            asc = false,
        )
        if (results == null) {
            _YIMSDK.utils.runOnUIThread {
                callback.invoke(null)
            }
            return
        }
        _YIMSDK.utils.runOnUIThread {
            callback.invoke(results.reversed().map { YIMMessage(it) })
        }
    }

    fun queryServerMessages(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        time: Date? = null,
        limit: Int = 20,
        queryDirection: YIMEnum_QueryDirection = YIMEnum_QueryDirection.Old,
        callback: (isSuccess: Boolean, messages: List<YIMMessage>?) -> (Unit),
    ) {
        if (!_YIMCore.userManager.isLogin()) {
            _YIMSDK.utils.runOnUIThread {
                callback.invoke(false, null)
            }
            return
        }
        _YIMSDK.server.emit(
            "queryMessages", mapOf(
                "conversationType" to conversationType.rawValue,
                "conversationID" to conversationID,
                "time" to (time ?: Date()).time,
                "limit" to limit,
                "queryDirection" to queryDirection.rawValue,
            )
        ) { isSuccess, results, code, msg ->
            if (!isSuccess) {
                _YIMSDK.utils.runOnUIThread {
                    callback.invoke(false, null)
                }
                return@emit
            }
            _YIMSDK.utils.runOnUIThread {
                callback.invoke(true, Gson().fromJson(_YIMSDK.utils.toJson(results!!["messages"]!!), object : TypeToken<List<_YIMTable_Message>>() {}).map { YIMMessage(it) })
            }
        }
    }

    fun queryLocalMessageByID(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        messageID: String,
        callback: ((message: YIMMessage?) -> Unit),
    ) {
        if (!_YIMCore.userManager.isLogin()) {
            _YIMSDK.utils.runOnUIThread {
                callback.invoke(null)
            }
            return
        }
        queryLocalMessagesByIDs(conversationType, conversationID, listOf(messageID)) { messages ->
            _YIMSDK.utils.runOnUIThread {
                callback.invoke(messages?.firstOrNull())
            }
        }
    }

    fun queryLocalMessagesByIDs(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        messageIDs: List<String>,
        callback: ((messages: List<YIMMessage>?) -> Unit),
    ) {
        if (!_YIMCore.userManager.isLogin()) {
            _YIMSDK.utils.runOnUIThread {
                callback.invoke(null)
            }
            return
        }
        val results = _YIMSDK.db.query<_YIMTable_Message>(
            queryCondition = _YIMTable_Message_.id.oneOf(messageIDs.toTypedArray()),
            order = _YIMTable_Message_.time,
            asc = true,
        )
        if (results == null) {
            _YIMSDK.utils.runOnUIThread {
                callback.invoke(null)
            }
            return
        }
        _YIMSDK.utils.runOnUIThread {
            callback.invoke(results.map { YIMMessage(it) })
        }
    }

    fun updateMessage(
        message: YIMMessage,
    ) {
        if (!_YIMCore.userManager.isLogin()) return
        _YIMSDK.db.update<_YIMTable_Message>(
            queryCondition = _YIMTable_Message_.id.equal(message.id),
            onQueryFinish = {
                return@update listOf(_YIMTable_Message(message))
            },
        )
    }

    fun sendMessageReceipt(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        callback: ((isSuccess: Boolean) -> Unit)? = null,
    ) {
        if (!_YIMCore.userManager.isLogin()) {
            _YIMSDK.utils.runOnUIThread {
                callback?.invoke(false)
            }
            return
        }
        _YIMSDK.server.emit(
            "sendMessageReceipt", mapOf(
                "conversationType" to conversationType.rawValue,
                "conversationID" to conversationID,
            )
        ) { isSuccess, results, code, msg ->
            if (!isSuccess) {
                _YIMSDK.utils.runOnUIThread {
                    callback?.invoke(false)
                }
                return@emit
            }
            val isSuccess_markConversationRead = _markConversationRead(conversationType, conversationID, true)
            if (!isSuccess_markConversationRead) {
                _YIMSDK.utils.runOnUIThread {
                    callback?.invoke(false)
                }
                return@emit
            }
            _YIMSDK.utils.runOnUIThread {
                callback?.invoke(true)
            }
        }
    }

    fun sendMessageRevoke(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        messageID: String,
        callback: ((isSuccess: Boolean) -> Unit)? = null,
    ) {
        if (!_YIMCore.userManager.isLogin()) {
            _YIMSDK.utils.runOnUIThread {
                callback?.invoke(false)
            }
            return
        }
        _YIMSDK.server.emit(
            "sendMessageRevoke", mapOf(
                "conversationType" to conversationType.rawValue,
                "conversationID" to conversationID,
                "messageID" to messageID,
            )
        ) { isSuccess, results, code, msg ->
            if (!isSuccess) {
                _YIMSDK.utils.runOnUIThread {
                    callback?.invoke(false)
                }
                return@emit
            }
            val isSuccess_markConversationRevoke = _markConversationRevoke(conversationType, messageID)
            if (!isSuccess_markConversationRevoke) {
                _YIMSDK.utils.runOnUIThread {
                    callback?.invoke(false)
                }
                return@emit
            }
            val isSuccess_updateLastMessage = _YIMCore.conversationManager._updateLastMessage(conversationType, conversationID)
            if (!isSuccess_updateLastMessage) {
                _YIMSDK.utils.runOnUIThread {
                    callback?.invoke(false)
                }
                return@emit
            }
            _YIMSDK.utils.runOnUIThread {
                _YIMCore.listenerManager.callback_recentConversation?.invoke()
            }
            _YIMSDK.utils.runOnUIThread {
                callback?.invoke(true)
            }
        }
    }

    internal fun _markConversationRead(
        conversationType: YIMEnum_ConversationType,
        conversationID: String,
        onlyOther: Boolean,
    ): Boolean {
        when (conversationType) {
            YIMEnum_ConversationType.P2P -> {
                val isSuccess_update = _YIMSDK.db.update<_YIMTable_Message>(
                    queryCondition = (if (onlyOther) (_YIMTable_Message_.to.equal(_YIMSDK.currentUser).and(_YIMTable_Message_.from.equal(conversationID))).and(_YIMTable_Message_.conversationType.equal(conversationType.rawValue))
                    else (_YIMTable_Message_.from.equal(_YIMSDK.currentUser).and(_YIMTable_Message_.to.equal(conversationID))).or(_YIMTable_Message_.to.equal(_YIMSDK.currentUser).and(_YIMTable_Message_.from.equal(conversationID)))).and(_YIMTable_Message_.messageState.equal(YIMEnum_MessageState.Success_UnRead.rawValue)),
                    onQueryFinish = {
                        it?.forEach { it.messageState = YIMEnum_MessageState.Success_Read }
                        return@update it
                    },
                )
                if (!isSuccess_update) return false
            }

            YIMEnum_ConversationType.Team -> {
                val isSuccess_update = _YIMSDK.db.update<_YIMTable_Message>(
                    queryCondition = (if (onlyOther) (_YIMTable_Message_.to.equal(conversationID).and(_YIMTable_Message_.from.notEqual(_YIMSDK.currentUser))).and(_YIMTable_Message_.conversationType.equal(conversationType.rawValue))
                    else _YIMTable_Message_.to.equal(conversationID)).and(_YIMTable_Message_.messageState.equal(YIMEnum_MessageState.Success_UnRead.rawValue)),
                    onQueryFinish = {
                        it?.forEach { it.messageState = YIMEnum_MessageState.Success_Read }
                        return@update it
                    },
                )
                if (!isSuccess_update) return false
            }

            else -> {}
        }

        return true
    }

    internal fun _markConversationRevoke(
        conversationType: YIMEnum_ConversationType,
        messageID: String,
    ): Boolean {
        val isSuccess_update = _YIMSDK.db.update<_YIMTable_Message>(
            queryCondition = _YIMTable_Message_.id.equal(messageID),
            onQueryFinish = {
                it?.forEach { it.isRevoke = true }
                return@update it
            },
        )
        if (!isSuccess_update) return false
        return true
    }
}
