/**
 * <h1>AWS AppSync Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <blockquote>
 * <p>
 * All classes with the <code>Cfn</code> prefix in this module (<a href="https://docs.aws.amazon.com/cdk/latest/guide/constructs.html#constructs_lib">CFN Resources</a>) are always stable and safe to use.
 * <p>
 * </blockquote>
 * <p>
 * <img alt="cdk-constructs: Experimental" src="https://img.shields.io/badge/cdk--constructs-experimental-important.svg?style=for-the-badge">
 * <p>
 * <blockquote>
 * <p>
 * The APIs of higher level constructs in this module are experimental and under active development.
 * They are subject to non-backward compatible changes or removal in any future version. These are
 * not subject to the <a href="https://semver.org/">Semantic Versioning</a> model and breaking changes will be
 * announced in the release notes. This means that while you may use them, you may need to update
 * your source code when upgrading to a newer version of this package.
 * <p>
 * </blockquote>
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * The <code>&#64;aws-cdk/aws-appsync</code> package contains constructs for building flexible
 * APIs that use GraphQL.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.appsync.*;
 * </pre></blockquote>
 * <p>
 * <h2>Example</h2>
 * <p>
 * <h3>DynamoDB</h3>
 * <p>
 * Example of a GraphQL API with <code>AWS_IAM</code> <a href="#authorization">authorization</a> resolving into a DynamoDb
 * backend data source.
 * <p>
 * GraphQL schema file <code>schema.graphql</code>:
 * <p>
 * <blockquote><pre>
 * type demo {
 *   id: String!
 *   version: String!
 * }
 * type Query {
 *   getDemos: [ demo! ]
 * }
 * input DemoInput {
 *   version: String!
 * }
 * type Mutation {
 *   addDemo(input: DemoInput!): demo
 * }
 * </pre></blockquote>
 * <p>
 * CDK stack file <code>app-stack.ts</code>:
 * <p>
 * <blockquote><pre>
 * GraphqlApi api = GraphqlApi.Builder.create(this, "Api")
 *         .name("demo")
 *         .schema(Schema.fromAsset(join(__dirname, "schema.graphql")))
 *         .authorizationConfig(AuthorizationConfig.builder()
 *                 .defaultAuthorization(AuthorizationMode.builder()
 *                         .authorizationType(AuthorizationType.IAM)
 *                         .build())
 *                 .build())
 *         .xrayEnabled(true)
 *         .build();
 * 
 * Table demoTable = Table.Builder.create(this, "DemoTable")
 *         .partitionKey(Attribute.builder()
 *                 .name("id")
 *                 .type(AttributeType.STRING)
 *                 .build())
 *         .build();
 * 
 * DynamoDbDataSource demoDS = api.addDynamoDbDataSource("demoDataSource", demoTable);
 * 
 * // Resolver for the Query "getDemos" that scans the DynamoDb table and returns the entire list.
 * demoDS.createResolver(BaseResolverProps.builder()
 *         .typeName("Query")
 *         .fieldName("getDemos")
 *         .requestMappingTemplate(MappingTemplate.dynamoDbScanTable())
 *         .responseMappingTemplate(MappingTemplate.dynamoDbResultList())
 *         .build());
 * 
 * // Resolver for the Mutation "addDemo" that puts the item into the DynamoDb table.
 * demoDS.createResolver(BaseResolverProps.builder()
 *         .typeName("Mutation")
 *         .fieldName("addDemo")
 *         .requestMappingTemplate(MappingTemplate.dynamoDbPutItem(PrimaryKey.partition("id").auto(), Values.projecting("input")))
 *         .responseMappingTemplate(MappingTemplate.dynamoDbResultItem())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Aurora Serverless</h3>
 * <p>
 * AppSync provides a data source for executing SQL commands against Amazon Aurora
 * Serverless clusters. You can use AppSync resolvers to execute SQL statements
 * against the Data API with GraphQL queries, mutations, and subscriptions.
 * <p>
 * <blockquote><pre>
 * // Build a data source for AppSync to access the database.
 * GraphqlApi api;
 * // Create username and password secret for DB Cluster
 * DatabaseSecret secret = DatabaseSecret.Builder.create(this, "AuroraSecret")
 *         .username("clusteradmin")
 *         .build();
 * 
 * // The VPC to place the cluster in
 * Vpc vpc = new Vpc(this, "AuroraVpc");
 * 
 * // Create the serverless cluster, provide all values needed to customise the database.
 * ServerlessCluster cluster = ServerlessCluster.Builder.create(this, "AuroraCluster")
 *         .engine(DatabaseClusterEngine.AURORA_MYSQL)
 *         .vpc(vpc)
 *         .credentials(Map.of("username", "clusteradmin"))
 *         .clusterIdentifier("db-endpoint-test")
 *         .defaultDatabaseName("demos")
 *         .build();
 * RdsDataSource rdsDS = api.addRdsDataSource("rds", cluster, secret, "demos");
 * 
 * // Set up a resolver for an RDS query.
 * rdsDS.createResolver(BaseResolverProps.builder()
 *         .typeName("Query")
 *         .fieldName("getDemosRds")
 *         .requestMappingTemplate(MappingTemplate.fromString("\n  {\n    \"version\": \"2018-05-29\",\n    \"statements\": [\n      \"SELECT * FROM demos\"\n    ]\n  }\n  "))
 *         .responseMappingTemplate(MappingTemplate.fromString("\n    $utils.toJson($utils.rds.toJsonObject($ctx.result)[0])\n  "))
 *         .build());
 * 
 * // Set up a resolver for an RDS mutation.
 * rdsDS.createResolver(BaseResolverProps.builder()
 *         .typeName("Mutation")
 *         .fieldName("addDemoRds")
 *         .requestMappingTemplate(MappingTemplate.fromString("\n  {\n    \"version\": \"2018-05-29\",\n    \"statements\": [\n      \"INSERT INTO demos VALUES (:id, :version)\",\n      \"SELECT * WHERE id = :id\"\n    ],\n    \"variableMap\": {\n      \":id\": $util.toJson($util.autoId()),\n      \":version\": $util.toJson($ctx.args.version)\n    }\n  }\n  "))
 *         .responseMappingTemplate(MappingTemplate.fromString("\n    $utils.toJson($utils.rds.toJsonObject($ctx.result)[1][0])\n  "))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>HTTP Endpoints</h3>
 * <p>
 * GraphQL schema file <code>schema.graphql</code>:
 * <p>
 * <blockquote><pre>
 * type job {
 *   id: String!
 *   version: String!
 * }
 * 
 * input DemoInput {
 *   version: String!
 * }
 * 
 * type Mutation {
 *   callStepFunction(input: DemoInput!): job
 * }
 * </pre></blockquote>
 * <p>
 * GraphQL request mapping template <code>request.vtl</code>:
 * <p>
 * <blockquote><pre>
 * {
 *   "version": "2018-05-29",
 *   "method": "POST",
 *   "resourcePath": "/",
 *   "params": {
 *     "headers": {
 *       "content-type": "application/x-amz-json-1.0",
 *       "x-amz-target":"AWSStepFunctions.StartExecution"
 *     },
 *     "body": {
 *       "stateMachineArn": "&lt;your step functions arn&gt;",
 *       "input": "{ \"id\": \"$context.arguments.id\" }"
 *     }
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * GraphQL response mapping template <code>response.vtl</code>:
 * <p>
 * <blockquote><pre>
 * {
 *   "id": "${context.result.id}"
 * }
 * </pre></blockquote>
 * <p>
 * CDK stack file <code>app-stack.ts</code>:
 * <p>
 * <blockquote><pre>
 * GraphqlApi api = GraphqlApi.Builder.create(this, "api")
 *         .name("api")
 *         .schema(Schema.fromAsset(join(__dirname, "schema.graphql")))
 *         .build();
 * 
 * HttpDataSource httpDs = api.addHttpDataSource("ds", "https://states.amazonaws.com", HttpDataSourceOptions.builder()
 *         .name("httpDsWithStepF")
 *         .description("from appsync to StepFunctions Workflow")
 *         .authorizationConfig(AwsIamConfig.builder()
 *                 .signingRegion("us-east-1")
 *                 .signingServiceName("states")
 *                 .build())
 *         .build());
 * 
 * httpDs.createResolver(BaseResolverProps.builder()
 *         .typeName("Mutation")
 *         .fieldName("callStepFunction")
 *         .requestMappingTemplate(MappingTemplate.fromFile("request.vtl"))
 *         .responseMappingTemplate(MappingTemplate.fromFile("response.vtl"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Amazon OpenSearch Service</h3>
 * <p>
 * AppSync has builtin support for Amazon OpenSearch Service (successor to Amazon
 * Elasticsearch Service) from domains that are provisioned through your AWS account. You can
 * use AppSync resolvers to perform GraphQL operations such as queries, mutations, and
 * subscriptions.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.opensearchservice.*;
 * 
 * GraphqlApi api;
 * 
 * 
 * User user = new User(this, "User");
 * Domain domain = Domain.Builder.create(this, "Domain")
 *         .version(EngineVersion.OPENSEARCH_1_2)
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .fineGrainedAccessControl(AdvancedSecurityOptions.builder().masterUserArn(user.getUserArn()).build())
 *         .encryptionAtRest(EncryptionAtRestOptions.builder().enabled(true).build())
 *         .nodeToNodeEncryption(true)
 *         .enforceHttps(true)
 *         .build();
 * OpenSearchDataSource ds = api.addOpenSearchDataSource("ds", domain);
 * 
 * ds.createResolver(BaseResolverProps.builder()
 *         .typeName("Query")
 *         .fieldName("getTests")
 *         .requestMappingTemplate(MappingTemplate.fromString(JSON.stringify(Map.of(
 *                 "version", "2017-02-28",
 *                 "operation", "GET",
 *                 "path", "/id/post/_search",
 *                 "params", Map.of(
 *                         "headers", Map.of(),
 *                         "queryString", Map.of(),
 *                         "body", Map.of("from", 0, "size", 50))))))
 *         .responseMappingTemplate(MappingTemplate.fromString("[\n    #foreach($entry in $context.result.hits.hits)\n    #if( $velocityCount &gt; 1 ) , #end\n    $utils.toJson($entry.get(\"_source\"))\n    #end\n  ]"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Custom Domain Names</h2>
 * <p>
 * For many use cases you may want to associate a custom domain name with your
 * GraphQL API. This can be done during the API creation.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.certificatemanager.*;
 * import software.amazon.awscdk.services.route53.*;
 * 
 * // hosted zone and route53 features
 * String hostedZoneId;
 * String zoneName = "example.com";
 * 
 * 
 * String myDomainName = "api.example.com";
 * Certificate certificate = Certificate.Builder.create(this, "cert").domainName(myDomainName).build();
 * GraphqlApi api = GraphqlApi.Builder.create(this, "api")
 *         .name("myApi")
 *         .domainName(DomainOptions.builder()
 *                 .certificate(certificate)
 *                 .domainName(myDomainName)
 *                 .build())
 *         .build();
 * 
 * // hosted zone for adding appsync domain
 * IHostedZone zone = HostedZone.fromHostedZoneAttributes(this, "HostedZone", HostedZoneAttributes.builder()
 *         .hostedZoneId(hostedZoneId)
 *         .zoneName(zoneName)
 *         .build());
 * 
 * // create a cname to the appsync domain. will map to something like xxxx.cloudfront.net
 * // create a cname to the appsync domain. will map to something like xxxx.cloudfront.net
 * CnameRecord.Builder.create(this, "CnameApiRecord")
 *         .recordName("api")
 *         .zone(zone)
 *         .domainName(myDomainName)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Schema</h2>
 * <p>
 * Every GraphQL Api needs a schema to define the Api. CDK offers <code>appsync.Schema</code>
 * for static convenience methods for various types of schema declaration: code-first
 * or schema-first.
 * <p>
 * <h3>Code-First</h3>
 * <p>
 * When declaring your GraphQL Api, CDK defaults to a code-first approach if the
 * <code>schema</code> property is not configured.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api = GraphqlApi.Builder.create(this, "api").name("myApi").build();
 * </pre></blockquote>
 * <p>
 * CDK will declare a <code>Schema</code> class that will give your Api access functions to
 * define your schema code-first: <code>addType</code>, <code>addToSchema</code>, etc.
 * <p>
 * You can also declare your <code>Schema</code> class outside of your CDK stack, to define
 * your schema externally.
 * <p>
 * <blockquote><pre>
 * Schema schema = new Schema();
 * schema.addType(ObjectType.Builder.create("demo")
 *         .definition(Map.of("id", GraphqlType.id()))
 *         .build());
 * GraphqlApi api = GraphqlApi.Builder.create(this, "api")
 *         .name("myApi")
 *         .schema(schema)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See the <a href="#Code-First-Schema">code-first schema</a> section for more details.
 * <p>
 * <h3>Schema-First</h3>
 * <p>
 * You can define your GraphQL Schema from a file on disk. For convenience, use
 * the <code>appsync.Schema.fromAsset</code> to specify the file representing your schema.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api = GraphqlApi.Builder.create(this, "api")
 *         .name("myApi")
 *         .schema(Schema.fromAsset(join(__dirname, "schema.graphl")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Imports</h2>
 * <p>
 * Any GraphQL Api that has been created outside the stack can be imported from
 * another stack into your CDK app. Utilizing the <code>fromXxx</code> function, you have
 * the ability to add data sources and resolvers through a <code>IGraphqlApi</code> interface.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * Table table;
 * 
 * IGraphqlApi importedApi = GraphqlApi.fromGraphqlApiAttributes(this, "IApi", GraphqlApiAttributes.builder()
 *         .graphqlApiId(api.getApiId())
 *         .graphqlApiArn(api.getArn())
 *         .build());
 * importedApi.addDynamoDbDataSource("TableDataSource", table);
 * </pre></blockquote>
 * <p>
 * If you don't specify <code>graphqlArn</code> in <code>fromXxxAttributes</code>, CDK will autogenerate
 * the expected <code>arn</code> for the imported api, given the <code>apiId</code>. For creating data
 * sources and resolvers, an <code>apiId</code> is sufficient.
 * <p>
 * <h2>Authorization</h2>
 * <p>
 * There are multiple authorization types available for GraphQL API to cater to different
 * access use cases. They are:
 * <p>
 * <ul>
 * <li>API Keys (<code>AuthorizationType.API_KEY</code>)</li>
 * <li>Amazon Cognito User Pools (<code>AuthorizationType.USER_POOL</code>)</li>
 * <li>OpenID Connect (<code>AuthorizationType.OPENID_CONNECT</code>)</li>
 * <li>AWS Identity and Access Management (<code>AuthorizationType.AWS_IAM</code>)</li>
 * <li>AWS Lambda (<code>AuthorizationType.AWS_LAMBDA</code>)</li>
 * </ul>
 * <p>
 * These types can be used simultaneously in a single API, allowing different types of clients to
 * access data. When you specify an authorization type, you can also specify the corresponding
 * authorization mode to finish defining your authorization. For example, this is a GraphQL API
 * with AWS Lambda Authorization.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * Function authFunction;
 * 
 * 
 * GraphqlApi.Builder.create(this, "api")
 *         .name("api")
 *         .schema(Schema.fromAsset(join(__dirname, "appsync.test.graphql")))
 *         .authorizationConfig(AuthorizationConfig.builder()
 *                 .defaultAuthorization(AuthorizationMode.builder()
 *                         .authorizationType(AuthorizationType.LAMBDA)
 *                         .lambdaAuthorizerConfig(LambdaAuthorizerConfig.builder()
 *                                 .handler(authFunction)
 *                                 .build())
 *                         .build())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Permissions</h2>
 * <p>
 * When using <code>AWS_IAM</code> as the authorization type for GraphQL API, an IAM Role
 * with correct permissions must be used for access to API.
 * <p>
 * When configuring permissions, you can specify specific resources to only be
 * accessible by <code>IAM</code> authorization. For example, if you want to only allow mutability
 * for <code>IAM</code> authorized access you would configure the following.
 * <p>
 * In <code>schema.graphql</code>:
 * <p>
 * <blockquote><pre>
 * type Mutation {
 *   updateExample(...): ...
 *     &#64;aws_iam
 * }
 * </pre></blockquote>
 * <p>
 * In <code>IAM</code>:
 * <p>
 * <blockquote><pre>
 * {
 *   "Version": "2012-10-17",
 *   "Statement": [
 *     {
 *       "Effect": "Allow",
 *       "Action": [
 *         "appsync:GraphQL"
 *       ],
 *       "Resource": [
 *         "arn:aws:appsync:REGION:ACCOUNT_ID:apis/GRAPHQL_ID/types/Mutation/fields/updateExample"
 *       ]
 *     }
 *   ]
 * }
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/appsync/latest/devguide/security.html#aws-iam-authorization">documentation</a> for more details.
 * <p>
 * To make this easier, CDK provides <code>grant</code> API.
 * <p>
 * Use the <code>grant</code> function for more granular authorization.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com"))
 *         .build();
 * 
 * api.grant(role, IamResource.custom("types/Mutation/fields/updateExample"), "appsync:GraphQL");
 * </pre></blockquote>
 * <p>
 * <h3>IamResource</h3>
 * <p>
 * In order to use the <code>grant</code> functions, you need to use the class <code>IamResource</code>.
 * <p>
 * <ul>
 * <li><code>IamResource.custom(...arns)</code> permits custom ARNs and requires an argument.</li>
 * <li><code>IamResouce.ofType(type, ...fields)</code> permits ARNs for types and their fields.</li>
 * <li><code>IamResource.all()</code> permits ALL resources.</li>
 * </ul>
 * <p>
 * <h3>Generic Permissions</h3>
 * <p>
 * Alternatively, you can use more generic <code>grant</code> functions to accomplish the same usage.
 * <p>
 * These include:
 * <p>
 * <ul>
 * <li>grantMutation (use to grant access to Mutation fields)</li>
 * <li>grantQuery (use to grant access to Query fields)</li>
 * <li>grantSubscription (use to grant access to Subscription fields)</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * Role role;
 * 
 * 
 * // For generic types
 * api.grantMutation(role, "updateExample");
 * 
 * // For custom types and granular design
 * api.grant(role, IamResource.ofType("Mutation", "updateExample"), "appsync:GraphQL");
 * </pre></blockquote>
 * <p>
 * <h2>Pipeline Resolvers and AppSync Functions</h2>
 * <p>
 * AppSync Functions are local functions that perform certain operations onto a
 * backend data source. Developers can compose operations (Functions) and execute
 * them in sequence with Pipeline Resolvers.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * 
 * 
 * AppsyncFunction appsyncFunction = AppsyncFunction.Builder.create(this, "function")
 *         .name("appsync_function")
 *         .api(api)
 *         .dataSource(api.addNoneDataSource("none"))
 *         .requestMappingTemplate(MappingTemplate.fromFile("request.vtl"))
 *         .responseMappingTemplate(MappingTemplate.fromFile("response.vtl"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * AppSync Functions are used in tandem with pipeline resolvers to compose multiple
 * operations.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * AppsyncFunction appsyncFunction;
 * 
 * 
 * Resolver pipelineResolver = Resolver.Builder.create(this, "pipeline")
 *         .api(api)
 *         .dataSource(api.addNoneDataSource("none"))
 *         .typeName("typeName")
 *         .fieldName("fieldName")
 *         .requestMappingTemplate(MappingTemplate.fromFile("beforeRequest.vtl"))
 *         .pipelineConfig(List.of(appsyncFunction))
 *         .responseMappingTemplate(MappingTemplate.fromFile("afterResponse.vtl"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Learn more about Pipeline Resolvers and AppSync Functions <a href="https://docs.aws.amazon.com/appsync/latest/devguide/pipeline-resolvers.html">here</a>.
 * <p>
 * <h2>Code-First Schema</h2>
 * <p>
 * CDK offers the ability to generate your schema in a code-first approach.
 * A code-first approach offers a developer workflow with:
 * <p>
 * <ul>
 * <li><strong>modularity</strong>: organizing schema type definitions into different files</li>
 * <li><strong>reusability</strong>: simplifying down boilerplate/repetitive code</li>
 * <li><strong>consistency</strong>: resolvers and schema definition will always be synced</li>
 * </ul>
 * <p>
 * The code-first approach allows for <strong>dynamic</strong> schema generation. You can generate your schema based on variables and templates to reduce code duplication.
 * <p>
 * <h3>Code-First Example</h3>
 * <p>
 * To showcase the code-first approach. Let's try to model the following schema segment.
 * <p>
 * <blockquote><pre>
 * interface Node {
 *   id: String
 * }
 * 
 * type Query {
 *   allFilms(after: String, first: Int, before: String, last: Int): FilmConnection
 * }
 * 
 * type FilmNode implements Node {
 *   filmName: String
 * }
 * 
 * type FilmConnection {
 *   edges: [FilmEdge]
 *   films: [Film]
 *   totalCount: Int
 * }
 * 
 * type FilmEdge {
 *   node: Film
 *   cursor: String
 * }
 * </pre></blockquote>
 * <p>
 * Above we see a schema that allows for generating paginated responses. For example,
 * we can query <code>allFilms(first: 100)</code> since <code>FilmConnection</code> acts as an intermediary
 * for holding <code>FilmEdges</code> we can write a resolver to return the first 100 films.
 * <p>
 * In a separate file, we can declare our object types and related functions.
 * We will call this file <code>object-types.ts</code> and we will have created it in a way that
 * allows us to generate other <code>XxxConnection</code> and <code>XxxEdges</code> in the future.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.appsync.*;
 * Object pluralize = require("pluralize");
 * 
 * Map&lt;String, GraphqlType&gt; args = Map.of(
 *         "after", GraphqlType.string(),
 *         "first", GraphqlType.int(),
 *         "before", GraphqlType.string(),
 *         "last", GraphqlType.int());
 * 
 * InterfaceType Node = InterfaceType.Builder.create("Node")
 *         .definition(Map.of("id", GraphqlType.string()))
 *         .build();
 * ObjectType FilmNode = ObjectType.Builder.create("FilmNode")
 *         .interfaceTypes(List.of(Node))
 *         .definition(Map.of("filmName", GraphqlType.string()))
 *         .build();
 * 
 * public Map&lt;String, ObjectType&gt; generateEdgeAndConnection(ObjectType base) {
 *     ObjectType edge = ObjectType.Builder.create(String.format("%sEdge", base.getName()))
 *             .definition(Map.of("node", base.attribute(), "cursor", GraphqlType.string()))
 *             .build();
 *     ObjectType connection = ObjectType.Builder.create(String.format("%sConnection", base.getName()))
 *             .definition(Map.of(
 *                     "edges", edge.attribute(BaseTypeOptions.builder().isList(true).build()),
 *                     pluralize(base.getName()), base.attribute(BaseTypeOptions.builder().isList(true).build()),
 *                     "totalCount", GraphqlType.int()))
 *             .build();
 *     return Map.of("edge", edge, "connection", connection);
 * }
 * </pre></blockquote>
 * <p>
 * Finally, we will go to our <code>cdk-stack</code> and combine everything together
 * to generate our schema.
 * <p>
 * <blockquote><pre>
 * MappingTemplate dummyRequest;
 * MappingTemplate dummyResponse;
 * 
 * 
 * GraphqlApi api = GraphqlApi.Builder.create(this, "Api")
 *         .name("demo")
 *         .build();
 * 
 * InterfaceType[] objectTypes = List.of(Node, FilmNode);
 * 
 * Map&lt;String, ObjectType&gt; filmConnections = generateEdgeAndConnection(FilmNode);
 * 
 * api.addQuery("allFilms", ResolvableField.Builder.create()
 *         .returnType(filmConnections.connection.attribute())
 *         .args(args)
 *         .dataSource(api.addNoneDataSource("none"))
 *         .requestMappingTemplate(dummyRequest)
 *         .responseMappingTemplate(dummyResponse)
 *         .build());
 * 
 * api.addType(Node);
 * api.addType(FilmNode);
 * api.addType(filmConnections.getEdge());
 * api.addType(filmConnections.getConnection());
 * </pre></blockquote>
 * <p>
 * Notice how we can utilize the <code>generateEdgeAndConnection</code> function to generate
 * Object Types. In the future, if we wanted to create more Object Types, we can simply
 * create the base Object Type (i.e. Film) and from there we can generate its respective
 * <code>Connections</code> and <code>Edges</code>.
 * <p>
 * Check out a more in-depth example <a href="https://github.com/BryanPan342/starwars-code-first">here</a>.
 * <p>
 * <h2>GraphQL Types</h2>
 * <p>
 * One of the benefits of GraphQL is its strongly typed nature. We define the
 * types within an object, query, mutation, interface, etc. as <strong>GraphQL Types</strong>.
 * <p>
 * GraphQL Types are the building blocks of types, whether they are scalar, objects,
 * interfaces, etc. GraphQL Types can be:
 * <p>
 * <ul>
 * <li><a href="https://docs.aws.amazon.com/appsync/latest/devguide/scalars.html"><strong>Scalar Types</strong></a>: Id, Int, String, AWSDate, etc.</li>
 * <li><a href="#Object-Types"><strong>Object Types</strong></a>: types that you generate (i.e. <code>demo</code> from the example above)</li>
 * <li><a href="#Interface-Types"><strong>Interface Types</strong></a>: abstract types that define the base implementation of other
 * Intermediate Types</li>
 * </ul>
 * <p>
 * More concretely, GraphQL Types are simply the types appended to variables.
 * Referencing the object type <code>Demo</code> in the previous example, the GraphQL Types
 * is <code>String!</code> and is applied to both the names <code>id</code> and <code>version</code>.
 * <p>
 * <h3>Directives</h3>
 * <p>
 * <code>Directives</code> are attached to a field or type and affect the execution of queries,
 * mutations, and types. With AppSync, we use <code>Directives</code> to configure authorization.
 * CDK provides static functions to add directives to your Schema.
 * <p>
 * <ul>
 * <li><code>Directive.iam()</code> sets a type or field's authorization to be validated through <code>Iam</code></li>
 * <li><code>Directive.apiKey()</code> sets a type or field's authorization to be validated through a <code>Api Key</code></li>
 * <li><code>Directive.oidc()</code> sets a type or field's authorization to be validated through <code>OpenID Connect</code></li>
 * <li><code>Directive.cognito(...groups: string[])</code> sets a type or field's authorization to be validated
 * through <code>Cognito User Pools</code>
 * <p>
 * <ul>
 * <li><code>groups</code> the name of the cognito groups to give access</li>
 * </ul></li>
 * </ul>
 * <p>
 * To learn more about authorization and directives, read these docs <a href="https://docs.aws.amazon.com/appsync/latest/devguide/security.html">here</a>.
 * <p>
 * <h3>Field and Resolvable Fields</h3>
 * <p>
 * While <code>GraphqlType</code> is a base implementation for GraphQL fields, we have abstractions
 * on top of <code>GraphqlType</code> that provide finer grain support.
 * <p>
 * <h3>Field</h3>
 * <p>
 * <code>Field</code> extends <code>GraphqlType</code> and will allow you to define arguments. <a href="#Interface-Types"><strong>Interface Types</strong></a> are not resolvable and this class will allow you to define arguments,
 * but not its resolvers.
 * <p>
 * For example, if we want to create the following type:
 * <p>
 * <blockquote><pre>
 * type Node {
 *   test(argument: string): String
 * }
 * </pre></blockquote>
 * <p>
 * The CDK code required would be:
 * <p>
 * <blockquote><pre>
 * Field field = Field.Builder.create()
 *         .returnType(GraphqlType.string())
 *         .args(Map.of(
 *                 "argument", GraphqlType.string()))
 *         .build();
 * InterfaceType type = InterfaceType.Builder.create("Node")
 *         .definition(Map.of("test", field))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Resolvable Fields</h3>
 * <p>
 * <code>ResolvableField</code> extends <code>Field</code> and will allow you to define arguments and its resolvers.
 * <a href="#Object-Types"><strong>Object Types</strong></a> can have fields that resolve and perform operations on
 * your backend.
 * <p>
 * You can also create resolvable fields for object types.
 * <p>
 * <blockquote><pre>
 * type Info {
 *   node(id: String): String
 * }
 * </pre></blockquote>
 * <p>
 * The CDK code required would be:
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * MappingTemplate dummyRequest;
 * MappingTemplate dummyResponse;
 * 
 * ObjectType info = ObjectType.Builder.create("Info")
 *         .definition(Map.of(
 *                 "node", ResolvableField.Builder.create()
 *                         .returnType(GraphqlType.string())
 *                         .args(Map.of(
 *                                 "id", GraphqlType.string()))
 *                         .dataSource(api.addNoneDataSource("none"))
 *                         .requestMappingTemplate(dummyRequest)
 *                         .responseMappingTemplate(dummyResponse)
 *                         .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To nest resolvers, we can also create top level query types that call upon
 * other types. Building off the previous example, if we want the following graphql
 * type definition:
 * <p>
 * <blockquote><pre>
 * type Query {
 *   get(argument: string): Info
 * }
 * </pre></blockquote>
 * <p>
 * The CDK code required would be:
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * MappingTemplate dummyRequest;
 * MappingTemplate dummyResponse;
 * 
 * ObjectType query = ObjectType.Builder.create("Query")
 *         .definition(Map.of(
 *                 "get", ResolvableField.Builder.create()
 *                         .returnType(GraphqlType.string())
 *                         .args(Map.of(
 *                                 "argument", GraphqlType.string()))
 *                         .dataSource(api.addNoneDataSource("none"))
 *                         .requestMappingTemplate(dummyRequest)
 *                         .responseMappingTemplate(dummyResponse)
 *                         .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Learn more about fields and resolvers <a href="https://docs.aws.amazon.com/appsync/latest/devguide/resolver-mapping-template-reference-overview.html">here</a>.
 * <p>
 * <h3>Intermediate Types</h3>
 * <p>
 * Intermediate Types are defined by Graphql Types and Fields. They have a set of defined
 * fields, where each field corresponds to another type in the system. Intermediate
 * Types will be the meat of your GraphQL Schema as they are the types defined by you.
 * <p>
 * Intermediate Types include:
 * <p>
 * <ul>
 * <li><a href="#Interface-Types"><strong>Interface Types</strong></a></li>
 * <li><a href="#Object-Types"><strong>Object Types</strong></a></li>
 * <li><a href="#Enum-Types"><strong>Enum Types</strong></a></li>
 * <li><a href="#Input-Types"><strong>Input Types</strong></a></li>
 * <li><a href="#Union-Types"><strong>Union Types</strong></a></li>
 * </ul>
 * <p>
 * <h4>Interface Types</h4>
 * <p>
 * <strong>Interface Types</strong> are abstract types that define the implementation of other
 * intermediate types. They are useful for eliminating duplication and can be used
 * to generate Object Types with less work.
 * <p>
 * You can create Interface Types <em><strong>externally</strong></em>.
 * <p>
 * <blockquote><pre>
 * InterfaceType node = InterfaceType.Builder.create("Node")
 *         .definition(Map.of(
 *                 "id", GraphqlType.string(BaseTypeOptions.builder().isRequired(true).build())))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To learn more about <strong>Interface Types</strong>, read the docs <a href="https://graphql.org/learn/schema/#interfaces">here</a>.
 * <p>
 * <h4>Object Types</h4>
 * <p>
 * <strong>Object Types</strong> are types that you declare. For example, in the <a href="#code-first-example">code-first example</a>
 * the <code>demo</code> variable is an <strong>Object Type</strong>. <strong>Object Types</strong> are defined by
 * GraphQL Types and are only usable when linked to a GraphQL Api.
 * <p>
 * You can create Object Types in two ways:
 * <p>
 * <ol>
 * <li>Object Types can be created <em><strong>externally</strong></em>.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api = GraphqlApi.Builder.create(this, "Api")
 *         .name("demo")
 *         .build();
 * ObjectType demo = ObjectType.Builder.create("Demo")
 *         .definition(Map.of(
 *                 "id", GraphqlType.string(BaseTypeOptions.builder().isRequired(true).build()),
 *                 "version", GraphqlType.string(BaseTypeOptions.builder().isRequired(true).build())))
 *         .build();
 * 
 * api.addType(demo);
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * This method allows for reusability and modularity, ideal for larger projects.
 * For example, imagine moving all Object Type definition outside the stack.
 * <p>
 * </blockquote>
 * <p>
 * <code>object-types.ts</code> - a file for object type definitions
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.appsync.*;
 * ObjectType demo = ObjectType.Builder.create("Demo")
 *         .definition(Map.of(
 *                 "id", GraphqlType.string(BaseTypeOptions.builder().isRequired(true).build()),
 *                 "version", GraphqlType.string(BaseTypeOptions.builder().isRequired(true).build())))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <code>cdk-stack.ts</code> - a file containing our cdk stack
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * 
 * api.addType(demo);
 * </pre></blockquote></li>
 * <li>Object Types can be created <em><strong>externally</strong></em> from an Interface Type.
 * <p>
 * <blockquote><pre>
 * InterfaceType node = InterfaceType.Builder.create("Node")
 *         .definition(Map.of(
 *                 "id", GraphqlType.string(BaseTypeOptions.builder().isRequired(true).build())))
 *         .build();
 * ObjectType demo = ObjectType.Builder.create("Demo")
 *         .interfaceTypes(List.of(node))
 *         .definition(Map.of(
 *                 "version", GraphqlType.string(BaseTypeOptions.builder().isRequired(true).build())))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * This method allows for reusability and modularity, ideal for reducing code duplication.
 * <p>
 * </blockquote></li>
 * </ol>
 * <p>
 * To learn more about <strong>Object Types</strong>, read the docs <a href="https://graphql.org/learn/schema/#object-types-and-fields">here</a>.
 * <p>
 * <h4>Enum Types</h4>
 * <p>
 * <strong>Enum Types</strong> are a special type of Intermediate Type. They restrict a particular
 * set of allowed values for other Intermediate Types.
 * <p>
 * <blockquote><pre>
 * enum Episode {
 *   NEWHOPE
 *   EMPIRE
 *   JEDI
 * }
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * This means that wherever we use the type Episode in our schema, we expect it to
 * be exactly one of NEWHOPE, EMPIRE, or JEDI.
 * <p>
 * </blockquote>
 * <p>
 * The above GraphQL Enumeration Type can be expressed in CDK as the following:
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * 
 * EnumType episode = EnumType.Builder.create("Episode")
 *         .definition(List.of("NEWHOPE", "EMPIRE", "JEDI"))
 *         .build();
 * api.addType(episode);
 * </pre></blockquote>
 * <p>
 * To learn more about <strong>Enum Types</strong>, read the docs <a href="https://graphql.org/learn/schema/#enumeration-types">here</a>.
 * <p>
 * <h4>Input Types</h4>
 * <p>
 * <strong>Input Types</strong> are special types of Intermediate Types. They give users an
 * easy way to pass complex objects for top level Mutation and Queries.
 * <p>
 * <blockquote><pre>
 * input Review {
 *   stars: Int!
 *   commentary: String
 * }
 * </pre></blockquote>
 * <p>
 * The above GraphQL Input Type can be expressed in CDK as the following:
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * 
 * InputType review = InputType.Builder.create("Review")
 *         .definition(Map.of(
 *                 "stars", GraphqlType.int(BaseTypeOptions.builder().isRequired(true).build()),
 *                 "commentary", GraphqlType.string()))
 *         .build();
 * api.addType(review);
 * </pre></blockquote>
 * <p>
 * To learn more about <strong>Input Types</strong>, read the docs <a href="https://graphql.org/learn/schema/#input-types">here</a>.
 * <p>
 * <h4>Union Types</h4>
 * <p>
 * <strong>Union Types</strong> are a special type of Intermediate Type. They are similar to
 * Interface Types, but they cannot specify any common fields between types.
 * <p>
 * <strong>Note:</strong> the fields of a union type need to be <code>Object Types</code>. In other words, you
 * can't create a union type out of interfaces, other unions, or inputs.
 * <p>
 * <blockquote><pre>
 * union Search = Human | Droid | Starship
 * </pre></blockquote>
 * <p>
 * The above GraphQL Union Type encompasses the Object Types of Human, Droid and Starship. It
 * can be expressed in CDK as the following:
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * 
 * GraphqlType string = GraphqlType.string();
 * ObjectType human = ObjectType.Builder.create("Human").definition(Map.of("name", string)).build();
 * ObjectType droid = ObjectType.Builder.create("Droid").definition(Map.of("name", string)).build();
 * ObjectType starship = ObjectType.Builder.create("Starship").definition(Map.of("name", string)).build();
 * UnionType search = UnionType.Builder.create("Search")
 *         .definition(List.of(human, droid, starship))
 *         .build();
 * api.addType(search);
 * </pre></blockquote>
 * <p>
 * To learn more about <strong>Union Types</strong>, read the docs <a href="https://graphql.org/learn/schema/#union-types">here</a>.
 * <p>
 * <h3>Query</h3>
 * <p>
 * Every schema requires a top level Query type. By default, the schema will look
 * for the <code>Object Type</code> named <code>Query</code>. The top level <code>Query</code> is the <strong>only</strong> exposed
 * type that users can access to perform <code>GET</code> operations on your Api.
 * <p>
 * To add fields for these queries, we can simply run the <code>addQuery</code> function to add
 * to the schema's <code>Query</code> type.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * InterfaceType filmConnection;
 * MappingTemplate dummyRequest;
 * MappingTemplate dummyResponse;
 * 
 * 
 * GraphqlType string = GraphqlType.string();
 * GraphqlType int = GraphqlType.int();
 * api.addQuery("allFilms", ResolvableField.Builder.create()
 *         .returnType(filmConnection.attribute())
 *         .args(Map.of("after", string, "first", int, "before", string, "last", int))
 *         .dataSource(api.addNoneDataSource("none"))
 *         .requestMappingTemplate(dummyRequest)
 *         .responseMappingTemplate(dummyResponse)
 *         .build());
 * </pre></blockquote>
 * <p>
 * To learn more about top level operations, check out the docs <a href="https://docs.aws.amazon.com/appsync/latest/devguide/graphql-overview.html">here</a>.
 * <p>
 * <h3>Mutation</h3>
 * <p>
 * Every schema <strong>can</strong> have a top level Mutation type. By default, the schema will look
 * for the <code>ObjectType</code> named <code>Mutation</code>. The top level <code>Mutation</code> Type is the only exposed
 * type that users can access to perform <code>mutable</code> operations on your Api.
 * <p>
 * To add fields for these mutations, we can simply run the <code>addMutation</code> function to add
 * to the schema's <code>Mutation</code> type.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * ObjectType filmNode;
 * MappingTemplate dummyRequest;
 * MappingTemplate dummyResponse;
 * 
 * 
 * GraphqlType string = GraphqlType.string();
 * GraphqlType int = GraphqlType.int();
 * api.addMutation("addFilm", ResolvableField.Builder.create()
 *         .returnType(filmNode.attribute())
 *         .args(Map.of("name", string, "film_number", int))
 *         .dataSource(api.addNoneDataSource("none"))
 *         .requestMappingTemplate(dummyRequest)
 *         .responseMappingTemplate(dummyResponse)
 *         .build());
 * </pre></blockquote>
 * <p>
 * To learn more about top level operations, check out the docs <a href="https://docs.aws.amazon.com/appsync/latest/devguide/graphql-overview.html">here</a>.
 * <p>
 * <h3>Subscription</h3>
 * <p>
 * Every schema <strong>can</strong> have a top level Subscription type. The top level <code>Subscription</code> Type
 * is the only exposed type that users can access to invoke a response to a mutation. <code>Subscriptions</code>
 * notify users when a mutation specific mutation is called. This means you can make any data source
 * real time by specify a GraphQL Schema directive on a mutation.
 * <p>
 * <strong>Note</strong>: The AWS AppSync client SDK automatically handles subscription connection management.
 * <p>
 * To add fields for these subscriptions, we can simply run the <code>addSubscription</code> function to add
 * to the schema's <code>Subscription</code> type.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * InterfaceType film;
 * 
 * 
 * api.addSubscription("addedFilm", Field.Builder.create()
 *         .returnType(film.attribute())
 *         .args(Map.of("id", GraphqlType.id(BaseTypeOptions.builder().isRequired(true).build())))
 *         .directives(List.of(Directive.subscribe("addFilm")))
 *         .build());
 * </pre></blockquote>
 * <p>
 * To learn more about top level operations, check out the docs <a href="https://docs.aws.amazon.com/appsync/latest/devguide/real-time-data.html">here</a>.
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.amazon.awscdk.services.appsync;
