/**
 * <h1>AWS App Mesh Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * AWS App Mesh is a service mesh based on the <a href="https://www.envoyproxy.io/">Envoy</a> proxy that makes it easy to monitor and control microservices. App Mesh standardizes how your microservices communicate, giving you end-to-end visibility and helping to ensure high-availability for your applications.
 * <p>
 * App Mesh gives you consistent visibility and network traffic controls for every microservice in an application.
 * <p>
 * App Mesh supports microservice applications that use service discovery naming for their components. To use App Mesh, you must have an existing application running on AWS Fargate, Amazon ECS, Amazon EKS, Kubernetes on AWS, or Amazon EC2.
 * <p>
 * For further information on <strong>AWS AppMesh</strong> visit the <a href="https://docs.aws.amazon.com/app-mesh/index.html">AWS Docs for AppMesh</a>.
 * <p>
 * <h2>Create the App and Stack</h2>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object app = new App();
 * Object stack = new Stack(app, "stack");
 * </pre></blockquote>
 * <p>
 * <h2>Creating the Mesh</h2>
 * <p>
 * A service mesh is a logical boundary for network traffic between the services that reside within it.
 * <p>
 * After you create your service mesh, you can create virtual services, virtual nodes, virtual routers, and routes to distribute traffic between the applications in your mesh.
 * <p>
 * The following example creates the <code>AppMesh</code> service mesh with the default filter of <code>DROP_ALL</code>, see <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-egressfilter.html">docs</a> here for more info on egress filters.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object mesh = Mesh.Builder.create(stack, "AppMesh")
 *         .meshName("myAwsmMesh")
 *         .build();
 * </pre></blockquote>
 * <p>
 * The mesh can also be created with the "ALLOW_ALL" egress filter by overwritting the property.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object mesh = Mesh.Builder.create(stack, "AppMesh")
 *         .meshName("myAwsmMesh")
 *         .egressFilter(MeshFilterType.getALLOW_ALL())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Adding VirtualRouters</h2>
 * <p>
 * The <em>Mesh</em> needs <em>VirtualRouters</em> as logical units to route requests to <em>VirtualNodes</em>.
 * <p>
 * Virtual routers handle traffic for one or more virtual services within your mesh.
 * After you create a virtual router, you can create and associate routes to your virtual router that direct incoming requests to different virtual nodes.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object router = mesh.addVirtualRouter("router", Map.of(
 *         "listeners", asList(appmesh.VirtualRouterListener.http(8080))));
 * </pre></blockquote>
 * <p>
 * Note that creating the router using the <code>addVirtualRouter()</code> method places it in the same Stack that the mesh belongs to
 * (which might be different from the current Stack).
 * The router can also be created using the constructor of <code>VirtualRouter</code> and passing in the mesh instead of calling the <code>addVirtualRouter()</code> method.
 * This is particularly useful when splitting your resources between many Stacks,
 * like creating the <code>mesh</code> as part of an infrastructure stack,
 * but the other resources, such as routers, in the application stack:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object mesh = Mesh.Builder.create(infraStack, "AppMesh")
 *         .meshName("myAwsmMesh")
 *         .egressFilter(MeshFilterType.getALLOW_ALL())
 *         .build();
 * 
 * // the VirtualRouter will belong to 'appStack',
 * // even though the Mesh belongs to 'infraStack'
 * Object router = VirtualRouter.Builder.create(appStack, "router")
 *         .mesh(mesh)// notice that mesh is a required property when creating a router with the 'new' statement
 *         .listeners(asList(appmesh.VirtualRouterListener.http(8081)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The same is true for other <code>add*()</code> methods in the AppMesh library.
 * <p>
 * The <em>VirtualRouterListener</em> class provides an easy interface for defining new protocol specific listeners.
 * The <code>http()</code>, <code>http2()</code>, <code>grpc()</code> and <code>tcp()</code> methods are available for use.
 * They accept a single port parameter, that is used to define what port to match requests on.
 * The port parameter can be omitted, and it will default to port 8080.
 * <p>
 * <h2>Adding VirtualService</h2>
 * <p>
 * A virtual service is an abstraction of a real service that is provided by a virtual node directly or indirectly by means of a virtual router. Dependent services call your virtual service by its virtualServiceName, and those requests are routed to the virtual node or virtual router that is specified as the provider for the virtual service.
 * <p>
 * We recommend that you use the service discovery name of the real service that you're targeting (such as <code>my-service.default.svc.cluster.local</code>).
 * <p>
 * When creating a virtual service:
 * <p>
 * <ul>
 * <li>If you want the virtual service to spread traffic across multiple virtual nodes, specify a Virtual router.</li>
 * <li>If you want the virtual service to reach a virtual node directly, without a virtual router, specify a Virtual node.</li>
 * </ul>
 * <p>
 * Adding a virtual router as the provider:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * VirtualService.Builder.create("virtual-service")
 *         .virtualServiceName("my-service.default.svc.cluster.local")// optional
 *         .virtualServiceProvider(appmesh.VirtualServiceProvider.virtualRouter(router))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Adding a virtual node as the provider:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * VirtualService.Builder.create("virtual-service")
 *         .virtualServiceName("my-service.default.svc.cluster.local")// optional
 *         .virtualServiceProvider(appmesh.VirtualServiceProvider.virtualNode(node))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Adding a VirtualNode</h2>
 * <p>
 * A <code>virtual node</code> acts as a logical pointer to a particular task group, such as an Amazon ECS service or a Kubernetes deployment.
 * <p>
 * When you create a <code>virtual node</code>, any inbound traffic that your <code>virtual node</code> expects should be specified as a listener. Any outbound traffic that your <code>virtual node</code> expects to reach should be specified as a backend.
 * <p>
 * The response metadata for your new <code>virtual node</code> contains the Amazon Resource Name (ARN) that is associated with the <code>virtual node</code>. Set this value (either the full ARN or the truncated resource name) as the APPMESH_VIRTUAL_NODE_NAME environment variable for your task group's Envoy proxy container in your task definition or pod spec. For example, the value could be mesh/default/virtualNode/simpleapp. This is then mapped to the node.id and node.cluster Envoy parameters.
 * <p>
 * <blockquote>
 * <p>
 * Note
 * If you require your Envoy stats or tracing to use a different name, you can override the node.cluster value that is set by APPMESH_VIRTUAL_NODE_NAME with the APPMESH_VIRTUAL_NODE_CLUSTER environment variable.
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object vpc = new Vpc(stack, "vpc");
 * Object namespace = PrivateDnsNamespace.Builder.create(this, "test-namespace")
 *         .vpc(vpc)
 *         .name("domain.local")
 *         .build();
 * Object service = namespace.createService("Svc");
 * 
 * Object node = mesh.addVirtualNode("virtual-node", Map.of(
 *         "serviceDiscovery", appmesh.ServiceDiscovery.cloudMap(service),
 *         "listeners", asList(appmesh.VirtualNodeListener.httpNodeListener(Map.of(
 *                 "port", 8081,
 *                 "healthCheck", appmesh.HealthCheck.http(Map.of(
 *                         "healthyThreshold", 3,
 *                         "interval", Duration.seconds(5), // minimum
 *                         "path", "/health-check-path",
 *                         "timeout", Duration.seconds(2), // minimum
 *                         "unhealthyThreshold", 2))))),
 *         "accessLog", appmesh.AccessLog.fromFilePath("/dev/stdout")));
 * </pre></blockquote>
 * <p>
 * Create a <code>VirtualNode</code> with the constructor and add tags.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object node = VirtualNode.Builder.create(this, "node")
 *         .mesh(mesh)
 *         .serviceDiscovery(appmesh.ServiceDiscovery.cloudMap(service))
 *         .listeners(asList(appmesh.VirtualNodeListener.http(Map.of(
 *                 "port", 8080,
 *                 "healthCheck", appmesh.HealthCheck.http(Map.of(
 *                         "healthyThreshold", 3,
 *                         "interval", Duration.seconds(5), // min
 *                         "path", "/ping",
 *                         "timeout", Duration.seconds(2), // min
 *                         "unhealthyThreshold", 2)),
 *                 "timeout", Map.of(
 *                         "idle", cdk.Duration.seconds(5))))))
 *         .backendDefaults(Map.of(
 *                 "tlsClientPolicy", Map.of(
 *                         "validation", Map.of(
 *                                 "trust", appmesh.TlsValidationTrust.file("/keys/local_cert_chain.pem")))))
 *         .accessLog(appmesh.AccessLog.fromFilePath("/dev/stdout"))
 *         .build();
 * 
 * cdk.Tag.add(node, "Environment", "Dev");
 * </pre></blockquote>
 * <p>
 * Create a <code>VirtualNode</code> with the constructor and add backend virtual service.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object node = VirtualNode.Builder.create(this, "node")
 *         .mesh(mesh)
 *         .serviceDiscovery(appmesh.ServiceDiscovery.cloudMap(service))
 *         .listeners(asList(appmesh.VirtualNodeListener.httpNodeListener(Map.of(
 *                 "port", 8080,
 *                 "healthCheck", appmesh.HealthCheck.http(Map.of(
 *                         "healthyThreshold", 3,
 *                         "interval", Duration.seconds(5), // min
 *                         "path", "/ping",
 *                         "timeout", Duration.seconds(2), // min
 *                         "unhealthyThreshold", 2)),
 *                 "timeout", Map.of(
 *                         "idle", cdk.Duration.seconds(5))))))
 *         .accessLog(appmesh.AccessLog.fromFilePath("/dev/stdout"))
 *         .build();
 * 
 * Object virtualService = VirtualService.Builder.create(stack, "service-1")
 *         .virtualServiceProvider(appmesh.VirtualServiceProvider.virtualRouter(router))
 *         .virtualServiceName("service1.domain.local")
 *         .build();
 * 
 * node.addBackend(appmesh.Backend.virtualService(virtualService));
 * </pre></blockquote>
 * <p>
 * The <code>listeners</code> property can be left blank and added later with the <code>node.addListener()</code> method. The <code>healthcheck</code> and <code>timeout</code> properties are optional but if specifying a listener, the <code>port</code> must be added.
 * <p>
 * The <code>backends</code> property can be added with <code>node.addBackend()</code>. We define a virtual service and add it to the virtual node to allow egress traffic to other node.
 * <p>
 * The <code>backendDefaults</code> property are added to the node while creating the virtual node. These are virtual node's default settings for all backends.
 * <p>
 * <h2>Adding TLS to a listener</h2>
 * <p>
 * The <code>tls</code> property can be provided when creating a Virtual Node listener, or a Virtual Gateway listener to add TLS configuration.
 * App Mesh allows you to provide the TLS certificate to the proxy in the following ways:
 * <p>
 * <ul>
 * <li>A certificate from AWS Certificate Manager (ACM) can be used.</li>
 * <li>A customer provided certificate can be specified with a <code>certificateChain</code> path file and a <code>privateKey</code> file path.</li>
 * <li>A certificate provided by a Secrets Discovery Service (SDS) endpoint over local Unix Domain Socket can be specified with its <code>secretName</code>.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_certificatemanager;
 * 
 * 
 * // A Virtual Node with listener TLS from an ACM provided certificate
 * Certificate cert = new Certificate(this, "cert", new CertificateProps()...);
 * 
 * Object node = VirtualNode.Builder.create(stack, "node")
 *         .mesh(mesh)
 *         .serviceDiscovery(appmesh.ServiceDiscovery.dns("node"))
 *         .listeners(asList(appmesh.VirtualNodeListener.grpc(Map.of(
 *                 "port", 80,
 *                 "tls", Map.of(
 *                         "mode", appmesh.TlsMode.getSTRICT(),
 *                         "certificate", appmesh.TlsCertificate.acm(cert))))))
 *         .build();
 * 
 * // A Virtual Gateway with listener TLS from a customer provided file certificate
 * Object gateway = VirtualGateway.Builder.create(this, "gateway")
 *         .mesh(mesh)
 *         .listeners(asList(appmesh.VirtualGatewayListener.grpc(Map.of(
 *                 "port", 8080,
 *                 "tls", Map.of(
 *                         "mode", appmesh.TlsMode.getSTRICT(),
 *                         "certificate", appmesh.TlsCertificate.file("path/to/certChain", "path/to/privateKey"))))))
 *         .virtualGatewayName("gateway")
 *         .build();
 * 
 * // A Virtual Gateway with listener TLS from a SDS provided certificate
 * Object gateway2 = VirtualGateway.Builder.create(this, "gateway2")
 *         .mesh(mesh)
 *         .listeners(asList(appmesh.VirtualGatewayListener.http2(Map.of(
 *                 "port", 8080,
 *                 "tls", Map.of(
 *                         "mode", appmesh.TlsMode.getSTRICT(),
 *                         "certificate", appmesh.TlsCertificate.sds("secrete_certificate"))))))
 *         .virtualGatewayName("gateway2")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Adding mutual TLS authentication</h2>
 * <p>
 * Mutual TLS authentication is an optional component of TLS that offers two-way peer authentication.
 * To enable mutual TLS authentication,
 * add <code>mutualTlsCertificate</code> property to TLS Client Policy and/or <code>mutualTlsValidation</code> property to TLS Listener.
 * <p>
 * <code>tls.mutualTlsValidation</code> and <code>tlsClientPolicy.mutualTlsCertificate</code> can be sourced from either:
 * <p>
 * <ul>
 * <li>A customer provided certificate can be specified with a <code>certificateChain</code> path file and a <code>privateKey</code> file path.</li>
 * <li>A certificate provided by a Secrets Discovery Service (SDS) endpoint over local Unix Domain Socket can be specified with its <code>secretName</code>.</li>
 * </ul>
 * <p>
 * <strong>Note</strong>: Currently, a certificate from AWS Certificate Manager (ACM) cannot be sourced for above two properties.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_certificatemanager;
 * 
 * 
 * Object node1 = VirtualNode.Builder.create(stack, "node1")
 *         .mesh(mesh)
 *         .serviceDiscovery(appmesh.ServiceDiscovery.dns("node"))
 *         .listeners(asList(appmesh.VirtualNodeListener.grpc(Map.of(
 *                 "port", 80,
 *                 "tls", Map.of(
 *                         "mode", appmesh.TlsMode.getSTRICT(),
 *                         "certificate", appmesh.TlsCertificate.file("path/to/certChain", "path/to/privateKey"),
 *                         // Validate a file client certificates to enable mutual TLS authentication when a client provides a certificate.
 *                         "mutualTlsValidation", Map.of(
 *                                 "trust", appmesh.TlsValidationTrust.file("path-to-certificate")))))))
 *         .build();
 * 
 * Object node2 = VirtualNode.Builder.create(stack, "node2")
 *         .mesh(mesh)
 *         .serviceDiscovery(appmesh.ServiceDiscovery.dns("node2"))
 *         .backendDefaults(Map.of(
 *                 "tlsClientPolicy", Map.of(
 *                         "ports", asList(8080, 8081),
 *                         "validation", Map.of(
 *                                 "subjectAlternativeNames", appmesh.SubjectAlternativeNames.matchingExactly("mesh-endpoint.apps.local"),
 *                                 "trust", appmesh.TlsValidationTrust.acm(asList(acmpca.CertificateAuthority.fromCertificateAuthorityArn(stack, "certificate", certificateAuthorityArn)))),
 *                         // Provide a SDS client certificate when a server requests it and enable mutual TLS authentication.
 *                         "mutualTlsCertificate", appmesh.TlsCertificate.sds("secret_certificate"))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Adding outlier detection to a Virtual Node listener</h2>
 * <p>
 * The <code>outlierDetection</code> property can be added to a Virtual Node listener to add outlier detection. The 4 parameters
 * (<code>baseEjectionDuration</code>, <code>interval</code>, <code>maxEjectionPercent</code>, <code>maxServerErrors</code>) are required.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Cloud Map service discovery is currently required for host ejection by outlier detection
 * Object vpc = new Vpc(stack, "vpc");
 * Object namespace = PrivateDnsNamespace.Builder.create(this, "test-namespace")
 *         .vpc(vpc)
 *         .name("domain.local")
 *         .build();
 * Object service = namespace.createService("Svc");
 * 
 * Object node = mesh.addVirtualNode("virtual-node", Map.of(
 *         "serviceDiscovery", appmesh.ServiceDiscovery.cloudMap(service),
 *         "outlierDetection", Map.of(
 *                 "baseEjectionDuration", cdk.Duration.seconds(10),
 *                 "interval", cdk.Duration.seconds(30),
 *                 "maxEjectionPercent", 50,
 *                 "maxServerErrors", 5)));
 * </pre></blockquote>
 * <p>
 * <h2>Adding a connection pool to a listener</h2>
 * <p>
 * The <code>connectionPool</code> property can be added to a Virtual Node listener or Virtual Gateway listener to add a request connection pool. There are different
 * connection pool properties per listener protocol types.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // A Virtual Node with a gRPC listener with a connection pool set
 * Object node = VirtualNode.Builder.create(stack, "node")
 *         .mesh(mesh)
 *         // DNS service discovery can optionally specify the DNS response type as either LOAD_BALANCER or ENDPOINTS.
 *         // LOAD_BALANCER means that the DNS resolver returns a loadbalanced set of endpoints,
 *         // whereas ENDPOINTS means that the DNS resolver is returning all the endpoints.
 *         // By default, the response type is assumed to be LOAD_BALANCER
 *         .serviceDiscovery(appmesh.ServiceDiscovery.dns("node", appmesh.ResponseType.getENDPOINTS()))
 *         .listeners(asList(appmesh.VirtualNodeListener.http(Map.of(
 *                 "port", 80,
 *                 "connectionPool", Map.of(
 *                         "maxConnections", 100,
 *                         "maxPendingRequests", 10)))))
 *         .build();
 * 
 * // A Virtual Gateway with a gRPC listener with a connection pool set
 * Object gateway = VirtualGateway.Builder.create(this, "gateway")
 *         .mesh(mesh)
 *         .listeners(asList(appmesh.VirtualGatewayListener.grpc(Map.of(
 *                 "port", 8080,
 *                 "connectionPool", Map.of(
 *                         "maxRequests", 10)))))
 *         .virtualGatewayName("gateway")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Adding a Route</h2>
 * <p>
 * A <code>route</code> is associated with a virtual router, and it's used to match requests for a virtual router and distribute traffic accordingly to its associated virtual nodes.
 * <p>
 * If your <code>route</code> matches a request, you can distribute traffic to one or more target virtual nodes with relative weighting.
 * <p>
 * The <em>RouteSpec</em> class provides an easy interface for defining new protocol specific route specs.
 * The <code>tcp()</code>, <code>http()</code>, <code>http2()</code>, and <code>grpc()</code> methods provide the spec necessary to define a protocol specific spec.
 * <p>
 * For HTTP based routes, the match field can be used to match on
 * path (prefix, exact, or regex), HTTP method, scheme, HTTP headers, and query parameters.
 * By default, an HTTP based route will match all requests.
 * <p>
 * For gRPC based routes, the match field can be used to match on service name, method name, and metadata.
 * When specifying the method name, service name must also be specified.
 * <p>
 * For example, here's how to add an HTTP route that matches based on a prefix of the URL path:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * router.addRoute("route-http", Map.of(
 *         "routeSpec", appmesh.RouteSpec.http(Map.of(
 *                 "weightedTargets", asList(Map.of(
 *                         "virtualNode", node)),
 *                 "match", Map.of(
 *                         // Path that is passed to this method must start with '/'.
 *                         "path", appmesh.HttpRoutePathMatch.startsWith("/path-to-app"))))));
 * </pre></blockquote>
 * <p>
 * Add an HTTP2 route that matches based on exact path, method, scheme, headers, and query parameters:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * router.addRoute("route-http2", Map.of(
 *         "routeSpec", appmesh.RouteSpec.http2(Map.of(
 *                 "weightedTargets", asList(Map.of(
 *                         "virtualNode", node)),
 *                 "match", Map.of(
 *                         "path", appmesh.HttpRoutePathMatch.exactly("/exact"),
 *                         "method", appmesh.HttpRouteMethod.getPOST(),
 *                         "protocol", appmesh.HttpRouteProtocol.getHTTPS(),
 *                         "headers", asList(appmesh.HeaderMatch.valueIs("Content-Type", "application/json"), appmesh.HeaderMatch.valueIsNot("Content-Type", "application/json")),
 *                         "queryParameters", asList(appmesh.QueryParameterMatch.valueIs("query-field", "value")))))));
 * </pre></blockquote>
 * <p>
 * Add a single route with multiple targets and split traffic 50/50
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * router.addRoute("route-http", Map.of(
 *         "routeSpec", appmesh.RouteSpec.http(Map.of(
 *                 "weightedTargets", asList(Map.of(
 *                         "virtualNode", node,
 *                         "weight", 50), Map.of(
 *                         "virtualNode", node,
 *                         "weight", 50)),
 *                 "match", Map.of(
 *                         "path", appmesh.HttpRoutePathMatch.startsWith("/path-to-app"))))));
 * </pre></blockquote>
 * <p>
 * Add an http2 route with retries:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * router.addRoute("route-http2-retry", Map.of(
 *         "routeSpec", appmesh.RouteSpec.http2(Map.of(
 *                 "weightedTargets", asList(Map.of("virtualNode", node)),
 *                 "retryPolicy", Map.of(
 *                         // Retry if the connection failed
 *                         "tcpRetryEvents", asList(appmesh.TcpRetryEvent.getCONNECTION_ERROR()),
 *                         // Retry if HTTP responds with a gateway error (502, 503, 504)
 *                         "httpRetryEvents", asList(appmesh.HttpRetryEvent.getGATEWAY_ERROR()),
 *                         // Retry five times
 *                         "retryAttempts", 5,
 *                         // Use a 1 second timeout per retry
 *                         "retryTimeout", cdk.Duration.seconds(1))))));
 * </pre></blockquote>
 * <p>
 * Add a gRPC route with retries:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * router.addRoute("route-grpc-retry", Map.of(
 *         "routeSpec", appmesh.RouteSpec.grpc(Map.of(
 *                 "weightedTargets", asList(Map.of("virtualNode", node)),
 *                 "match", Map.of("serviceName", "servicename"),
 *                 "retryPolicy", Map.of(
 *                         "tcpRetryEvents", asList(appmesh.TcpRetryEvent.getCONNECTION_ERROR()),
 *                         "httpRetryEvents", asList(appmesh.HttpRetryEvent.getGATEWAY_ERROR()),
 *                         // Retry if gRPC responds that the request was cancelled, a resource
 *                         // was exhausted, or if the service is unavailable
 *                         "grpcRetryEvents", asList(appmesh.GrpcRetryEvent.getCANCELLED(), appmesh.GrpcRetryEvent.getRESOURCE_EXHAUSTED(), appmesh.GrpcRetryEvent.getUNAVAILABLE()),
 *                         "retryAttempts", 5,
 *                         "retryTimeout", cdk.Duration.seconds(1))))));
 * </pre></blockquote>
 * <p>
 * Add an gRPC route that matches based on method name and metadata:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * router.addRoute("route-grpc-retry", Map.of(
 *         "routeSpec", appmesh.RouteSpec.grpc(Map.of(
 *                 "weightedTargets", asList(Map.of("virtualNode", node)),
 *                 "match", Map.of(
 *                         // When method name is specified, service name must be also specified.
 *                         "methodName", "methodname",
 *                         "serviceName", "servicename",
 *                         "metadata", asList(appmesh.HeaderMatch.valueStartsWith("Content-Type", "application/"), appmesh.HeaderMatch.valueDoesNotStartWith("Content-Type", "text/")))))));
 * </pre></blockquote>
 * <p>
 * Add a gRPC route with time out:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * router.addRoute("route-http", Map.of(
 *         "routeSpec", appmesh.RouteSpec.grpc(Map.of(
 *                 "weightedTargets", asList(Map.of(
 *                         "virtualNode", node)),
 *                 "match", Map.of(
 *                         "serviceName", "my-service.default.svc.cluster.local"),
 *                 "timeout", Map.of(
 *                         "idle", Duration.seconds(2),
 *                         "perRequest", Duration.seconds(1))))));
 * </pre></blockquote>
 * <p>
 * <h2>Adding a Virtual Gateway</h2>
 * <p>
 * A <em>virtual gateway</em> allows resources outside your mesh to communicate to resources that are inside your mesh.
 * The virtual gateway represents an Envoy proxy running in an Amazon ECS task, in a Kubernetes service, or on an Amazon EC2 instance.
 * Unlike a virtual node, which represents an Envoy running with an application, a virtual gateway represents Envoy deployed by itself.
 * <p>
 * A virtual gateway is similar to a virtual node in that it has a listener that accepts traffic for a particular port and protocol (HTTP, HTTP2, GRPC).
 * The traffic that the virtual gateway receives, is directed to other services in your mesh,
 * using rules defined in gateway routes which can be added to your virtual gateway.
 * <p>
 * Create a virtual gateway with the constructor:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * String certificateAuthorityArn = "arn:aws:acm-pca:us-east-1:123456789012:certificate-authority/12345678-1234-1234-1234-123456789012";
 * 
 * Object gateway = VirtualGateway.Builder.create(stack, "gateway")
 *         .mesh(mesh)
 *         .listeners(asList(appmesh.VirtualGatewayListener.http(Map.of(
 *                 "port", 443,
 *                 "healthCheck", appmesh.HealthCheck.http(Map.of(
 *                         "interval", cdk.Duration.seconds(10)))))))
 *         .backendDefaults(Map.of(
 *                 "tlsClientPolicy", Map.of(
 *                         "ports", asList(8080, 8081),
 *                         "validation", Map.of(
 *                                 "trust", appmesh.TlsValidationTrust.acm(asList(acmpca.CertificateAuthority.fromCertificateAuthorityArn(stack, "certificate", certificateAuthorityArn)))))))
 *         .accessLog(appmesh.AccessLog.fromFilePath("/dev/stdout"))
 *         .virtualGatewayName("virtualGateway")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Add a virtual gateway directly to the mesh:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object gateway = mesh.addVirtualGateway("gateway", Map.of(
 *         "accessLog", appmesh.AccessLog.fromFilePath("/dev/stdout"),
 *         "virtualGatewayName", "virtualGateway",
 *         "listeners", asList(appmesh.VirtualGatewayListener.http(Map.of(
 *                 "port", 443,
 *                 "healthCheck", appmesh.HealthCheck.http(Map.of(
 *                         "interval", cdk.Duration.seconds(10))))))));
 * </pre></blockquote>
 * <p>
 * The listeners field can be omitted which will default to an HTTP Listener on port 8080.
 * A gateway route can be added using the <code>gateway.addGatewayRoute()</code> method.
 * <p>
 * The <code>backendDefaults</code> property is added to the node while creating the virtual gateway. These are virtual gateway's default settings for all backends.
 * <p>
 * <h2>Adding a Gateway Route</h2>
 * <p>
 * A <em>gateway route</em> is attached to a virtual gateway and routes traffic to an existing virtual service.
 * If a route matches a request, it can distribute traffic to a target virtual service.
 * <p>
 * For HTTP based gateway routes, the match field can be used to match on
 * path (prefix, exact, or regex), HTTP method, host name, HTTP headers, and query parameters.
 * By default, an HTTP based route will match all requests.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * gateway.addGatewayRoute("gateway-route-http", Map.of(
 *         "routeSpec", appmesh.GatewayRouteSpec.http(Map.of(
 *                 "routeTarget", virtualService,
 *                 "match", Map.of(
 *                         "path", appmesh.HttpGatewayRoutePathMatch.regex("regex"))))));
 * </pre></blockquote>
 * <p>
 * For gRPC based gateway routes, the match field can be used to match on service name, host name, and metadata.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * gateway.addGatewayRoute("gateway-route-grpc", Map.of(
 *         "routeSpec", appmesh.GatewayRouteSpec.grpc(Map.of(
 *                 "routeTarget", virtualService,
 *                 "match", Map.of(
 *                         "hostname", appmesh.GatewayRouteHostnameMatch.endsWith(".example.com"))))));
 * </pre></blockquote>
 * <p>
 * For HTTP based gateway routes, App Mesh automatically rewrites the matched prefix path in Gateway Route to “/”.
 * This automatic rewrite configuration can be overwritten in following ways:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * gateway.addGatewayRoute("gateway-route-http", Map.of(
 *         "routeSpec", appmesh.GatewayRouteSpec.http(Map.of(
 *                 "routeTarget", virtualService,
 *                 "match", Map.of(
 *                         // This disables the default rewrite to '/', and retains original path.
 *                         "path", appmesh.HttpGatewayRoutePathMatch.startsWith("/path-to-app/", ""))))));
 * 
 * gateway.addGatewayRoute("gateway-route-http-1", Map.of(
 *         "routeSpec", appmesh.GatewayRouteSpec.http(Map.of(
 *                 "routeTarget", virtualService,
 *                 "match", Map.of(
 *                         // If the request full path is '/path-to-app/xxxxx', this rewrites the path to '/rewrittenUri/xxxxx'.
 *                         // Please note both `prefixPathMatch` and `rewriteTo` must start and end with the `/` character.
 *                         "path", appmesh.HttpGatewayRoutePathMatch.startsWith("/path-to-app/", "/rewrittenUri/"))))));
 * </pre></blockquote>
 * <p>
 * If matching other path (exact or regex), only specific rewrite path can be specified.
 * Unlike <code>startsWith()</code> method above, no default rewrite is performed.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * gateway.addGatewayRoute("gateway-route-http-2", Map.of(
 *         "routeSpec", appmesh.GatewayRouteSpec.http(Map.of(
 *                 "routeTarget", virtualService,
 *                 "match", Map.of(
 *                         // This rewrites the path from '/test' to '/rewrittenPath'.
 *                         "path", appmesh.HttpGatewayRoutePathMatch.exactly("/test", "/rewrittenPath"))))));
 * </pre></blockquote>
 * <p>
 * For HTTP/gRPC based routes, App Mesh automatically rewrites
 * the original request received at the Virtual Gateway to the destination Virtual Service name.
 * This default host name rewrite can be configured by specifying the rewrite rule as one of the <code>match</code> property:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * gateway.addGatewayRoute("gateway-route-grpc", Map.of(
 *         "routeSpec", appmesh.GatewayRouteSpec.grpc(Map.of(
 *                 "routeTarget", virtualService,
 *                 "match", Map.of(
 *                         "hostname", appmesh.GatewayRouteHostnameMatch.exactly("example.com"),
 *                         // This disables the default rewrite to virtual service name and retain original request.
 *                         "rewriteRequestHostname", false)))));
 * </pre></blockquote>
 * <p>
 * <h2>Importing Resources</h2>
 * <p>
 * Each mesh resource comes with two static methods for importing a reference to an existing App Mesh resource.
 * These imported resources can be used as references for other resources in your mesh.
 * There are two static methods, <code>from&lt;Resource&gt;Arn</code> and <code>from&lt;Resource&gt;Attributes</code> where the <code>&lt;Resource&gt;</code> is replaced with the resource name.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * String arn = "arn:aws:appmesh:us-east-1:123456789012:mesh/testMesh/virtualNode/testNode";
 * appmesh.VirtualNode.fromVirtualNodeArn(stack, "importedVirtualNode", arn);
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * appmesh.VirtualNode.fromVirtualNodeAttributes(stack, "imported-virtual-node", Map.of(
 *         "mesh", appmesh.Mesh.fromMeshName(stack, "Mesh", "testMesh"),
 *         "virtualNodeName", virtualNodeName));
 * </pre></blockquote>
 * <p>
 * To import a mesh, there are two static methods, <code>fromMeshArn</code> and <code>fromMeshName</code>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * String arn = "arn:aws:appmesh:us-east-1:123456789012:mesh/testMesh";
 * appmesh.Mesh.fromMeshArn(stack, "imported-mesh", arn);
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * appmesh.Mesh.fromMeshName(stack, "imported-mesh", "abc");
 * </pre></blockquote>
 * <p>
 * <h2>IAM Grants</h2>
 * <p>
 * Virtual Node and Virtual Gateway implement <code>grantStreamAggregatedResources</code> that will grant identities that are running
 * Envoy access to stream generated config from App Mesh.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object gateway = VirtualGateway.Builder.create(stack, "testGateway").mesh(mesh).build();
 * User envoyUser = new User(stack, "envoyUser");
 * 
 * /**
 *  * This will grant `grantStreamAggregatedResources` ONLY for this gateway.
 *  *{@literal /}
 * gateway.grantStreamAggregatedResources(envoyUser);
 * </pre></blockquote>
 * <p>
 * <h2>Adding Resources to shared meshes</h2>
 * <p>
 * A shared mesh allows resources created by different accounts to communicate with each other in the same mesh:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // This is the ARN for the mesh from different AWS IAM account ID.
 * // Ensure mesh is properly shared with your account. For more details, see: https://github.com/aws/aws-cdk/issues/15404
 * String arn = "arn:aws:appmesh:us-east-1:123456789012:mesh/testMesh";
 * sharedMesh = appmesh.Mesh.fromMeshArn(stack, "imported-mesh", arn);
 * 
 * // This VirtualNode resource can communicate with the resources in the mesh from different AWS IAM account ID.
 * // This VirtualNode resource can communicate with the resources in the mesh from different AWS IAM account ID.
 * VirtualNode.Builder.create(stack, "test-node")
 *         .mesh(sharedMesh)
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.appmesh;
