/**
 * <h1>AWS Lambda Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * This construct library allows you to define AWS Lambda Functions.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.getNODEJS_12_X())
 *         .handler("index.handler")
 *         .code(Code.fromAsset(path.join(__dirname, "lambda-handler")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Handler Code</h2>
 * <p>
 * The <code>lambda.Code</code> class includes static convenience methods for various types of
 * runtime code.
 * <p>
 * <ul>
 * <li><code>lambda.Code.fromBucket(bucket, key[, objectVersion])</code> - specify an S3 object
 * that contains the archive of your runtime code.</li>
 * <li><code>lambda.Code.fromInline(code)</code> - inline the handle code as a string. This is
 * limited to supported runtimes and the code cannot exceed 4KiB.</li>
 * <li><code>lambda.Code.fromAsset(path)</code> - specify a directory or a .zip file in the local
 * filesystem which will be zipped and uploaded to S3 before deployment. See also
 * <a href="#bundling-asset-code">bundling asset code</a>.</li>
 * <li><code>lambda.Code.fromDockerBuild(path, options)</code> - use the result of a Docker
 * build as code. The runtime code is expected to be located at <code>/asset</code> in the
 * image and will be zipped and uploaded to S3 as an asset.</li>
 * </ul>
 * <p>
 * The following example shows how to define a Python function and deploy the code
 * from the local directory <code>my-lambda-handler</code> to it:
 * <p>
 * <a href="test/integ.assets.lit.ts">Example of Lambda Code from Local Assets</a>
 * <p>
 * When deploying a stack that contains this code, the directory will be zip
 * archived and then uploaded to an S3 bucket, then the exact location of the S3
 * objects will be passed when the stack is deployed.
 * <p>
 * During synthesis, the CDK expects to find a directory on disk at the asset
 * directory specified. Note that we are referencing the asset directory relatively
 * to our CDK project directory. This is especially important when we want to share
 * this construct through a library. Different programming languages will have
 * different techniques for bundling resources into libraries.
 * <p>
 * <h2>Docker Images</h2>
 * <p>
 * Lambda functions allow specifying their handlers within docker images. The docker
 * image can be an image from ECR or a local asset that the CDK will package and load
 * into ECR.
 * <p>
 * The following <code>DockerImageFunction</code> construct uses a local folder with a
 * Dockerfile as the asset that will be used as the function handler.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * DockerImageFunction.Builder.create(this, "AssetFunction")
 *         .code(DockerImageCode.fromImageAsset(path.join(__dirname, "docker-handler")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also specify an image that already exists in ECR as the function handler.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_ecr;
 * 
 * Repository repo = new Repository(this, "Repository");
 * 
 * DockerImageFunction.Builder.create(this, "ECRFunction")
 *         .code(DockerImageCode.fromEcr(repo))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Execution Role</h2>
 * <p>
 * Lambda functions assume an IAM role during execution. In CDK by default, Lambda
 * functions will use an autogenerated Role if one is not provided.
 * <p>
 * The autogenerated Role is automatically given permissions to execute the Lambda
 * function. To reference the autogenerated Role:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.getNODEJS_12_X())
 *         .handler("index.handler")
 *         .code(Code.fromAsset(path.join(__dirname, "lambda-handler")))
 *         .build();
 * 
 * fn.getRole();
 * </pre></blockquote>
 * <p>
 * You can also provide your own IAM role. Provided IAM roles will not automatically
 * be given permissions to execute the Lambda function. To provide a role and grant
 * it appropriate permissions:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_iam;
 * 
 * Role myRole = new Role(this, "My Role", new RoleProps()
 *         .assumedBy(new ServicePrincipal("sns.amazonaws.com")));
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.getNODEJS_12_X())
 *         .handler("index.handler")
 *         .code(Code.fromAsset(path.join(__dirname, "lambda-handler")))
 *         .role(myRole)
 *         .build();
 * 
 * myRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AWSLambdaBasicExecutionRole"));
 * myRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AWSLambdaVPCAccessExecutionRole"));
 * </pre></blockquote>
 * <p>
 * <h2>Resource-based Policies</h2>
 * <p>
 * AWS Lambda supports resource-based policies for controlling access to Lambda
 * functions and layers on a per-resource basis. In particular, this allows you to
 * give permission to AWS services and other AWS accounts to modify and invoke your
 * functions. You can also restrict permissions given to AWS services by providing
 * a source account or ARN (representing the account and identifier of the resource
 * that accesses the function or layer).
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_iam;
 * 
 * ServicePrincipal principal = new ServicePrincipal("my-service");
 * 
 * fn.grantInvoke(principal);
 * 
 * // Equivalent to:
 * fn.addPermission("my-service Invocation", new Permission()
 *         .principal(principal));
 * </pre></blockquote>
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/lambda/latest/dg/access-control-resource-based.html">Resource-based
 * policies</a>
 * in the AWS Lambda Developer Guide.
 * <p>
 * Providing an unowned principal (such as account principals, generic ARN
 * principals, service principals, and principals in other accounts) to a call to
 * <code>fn.grantInvoke</code> will result in a resource-based policy being created. If the
 * principal in question has conditions limiting the source account or ARN of the
 * operation (see above), these conditions will be automatically added to the
 * resource policy.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_iam;
 * 
 * ServicePrincipal servicePrincipal = new ServicePrincipal("my-service");
 * String sourceArn = "arn:aws:s3:::my-bucket";
 * String sourceAccount = "111122223333";
 * IPrincipal servicePrincipalWithConditions = servicePrincipal.withConditions(Map.of(
 *         "ArnLike", Map.of(
 *                 "aws:SourceArn", sourceArn),
 *         "StringEquals", Map.of(
 *                 "aws:SourceAccount", sourceAccount)));
 * 
 * fn.grantInvoke(servicePrincipalWithConditions);
 * 
 * // Equivalent to:
 * fn.addPermission("my-service Invocation", new Permission()
 *         .principal(servicePrincipal)
 *         .sourceArn(sourceArn)
 *         .sourceAccount(sourceAccount));
 * </pre></blockquote>
 * <p>
 * <h2>Versions</h2>
 * <p>
 * You can use
 * <a href="https://docs.aws.amazon.com/lambda/latest/dg/configuration-versions.html">versions</a>
 * to manage the deployment of your AWS Lambda functions. For example, you can
 * publish a new version of a function for beta testing without affecting users of
 * the stable production version.
 * <p>
 * The function version includes the following information:
 * <p>
 * <ul>
 * <li>The function code and all associated dependencies.</li>
 * <li>The Lambda runtime that executes the function.</li>
 * <li>All of the function settings, including the environment variables.</li>
 * <li>A unique Amazon Resource Name (ARN) to identify this version of the function.</li>
 * </ul>
 * <p>
 * You could create a version to your lambda function using the <code>Version</code> construct.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Function fn = new Function(this, "MyFunction", ...);
 * Object version = Version.Builder.create(this, "MyVersion")
 *         .lambda(fn)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The major caveat to know here is that a function version must always point to a
 * specific 'version' of the function. When the function is modified, the version
 * will continue to point to the 'then version' of the function.
 * <p>
 * One way to ensure that the <code>lambda.Version</code> always points to the latest version
 * of your <code>lambda.Function</code> is to set an environment variable which changes at
 * least as often as your code does. This makes sure the function always has the
 * latest code. For instance -
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * String codeVersion = "stringOrMethodToGetCodeVersion";
 * Function fn = new Function(this, "MyFunction", new FunctionProps()
 *         .environment(Map.of(
 *                 "CodeVersionString", codeVersion)));
 * </pre></blockquote>
 * <p>
 * The <code>fn.latestVersion</code> property returns a <code>lambda.IVersion</code> which represents
 * the <code>$LATEST</code> pseudo-version.
 * <p>
 * However, most AWS services require a specific AWS Lambda version,
 * and won't allow you to use <code>$LATEST</code>. Therefore, you would normally want
 * to use <code>lambda.currentVersion</code>.
 * <p>
 * The <code>fn.currentVersion</code> property can be used to obtain a <code>lambda.Version</code>
 * resource that represents the AWS Lambda function defined in your application.
 * Any change to your function's code or configuration will result in the creation
 * of a new version resource. You can specify options for this version through the
 * <code>currentVersionOptions</code> property.
 * <p>
 * NOTE: The <code>currentVersion</code> property is only supported when your AWS Lambda function
 * uses either <code>lambda.Code.fromAsset</code> or <code>lambda.Code.fromInline</code>. Other types
 * of code providers (such as <code>lambda.Code.fromBucket</code>) require that you define a
 * <code>lambda.Version</code> resource directly since the CDK is unable to determine if
 * their contents had changed.
 * <p>
 * <h3><code>currentVersion</code>: Updated hashing logic</h3>
 * <p>
 * To produce a new lambda version each time the lambda function is modified, the
 * <code>currentVersion</code> property under the hood, computes a new logical id based on the
 * properties of the function. This informs CloudFormation that a new
 * <code>AWS::Lambda::Version</code> resource should be created pointing to the updated Lambda
 * function.
 * <p>
 * However, a bug was introduced in this calculation that caused the logical id to
 * change when it was not required (ex: when the Function's <code>Tags</code> property, or
 * when the <code>DependsOn</code> clause was modified). This caused the deployment to fail
 * since the Lambda service does not allow creating duplicate versions.
 * <p>
 * This has been fixed in the AWS CDK but <em>existing</em> users need to opt-in via a
 * <a href="https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html">feature flag</a>. Users who have run <code>cdk init</code> since this fix will be opted in,
 * by default.
 * <p>
 * Existing users will need to enable the <a href="https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html">feature flag</a>
 * <code>&#64;aws-cdk/aws-lambda:recognizeVersionProps</code>. Since CloudFormation does not
 * allow duplicate versions, they will also need to make some modification to
 * their function so that a new version can be created. Any trivial change such as
 * a whitespace change in the code or a no-op environment variable will suffice.
 * <p>
 * When the new logic is in effect, you may rarely come across the following error:
 * <code>The following properties are not recognized as version properties</code>. This will
 * occur, typically when <a href="https://docs.aws.amazon.com/cdk/latest/guide/cfn_layer.html#cfn_layer_raw">property overrides</a> are used, when a new property
 * introduced in <code>AWS::Lambda::Function</code> is used that CDK is still unaware of.
 * <p>
 * To overcome this error, use the API <code>Function.classifyVersionProperty()</code> to
 * record whether a new version should be generated when this property is changed.
 * This can be typically determined by checking whether the property can be
 * modified using the <em><a href="https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionConfiguration.html">UpdateFunctionConfiguration</a></em> API or not.
 * <p>
 * <h2>Aliases</h2>
 * <p>
 * You can define one or more
 * <a href="https://docs.aws.amazon.com/lambda/latest/dg/configuration-aliases.html">aliases</a>
 * for your AWS Lambda function. A Lambda alias is like a pointer to a specific
 * Lambda function version. Users can access the function version using the alias
 * ARN.
 * <p>
 * The <code>version.addAlias()</code> method can be used to define an AWS Lambda alias that
 * points to a specific version.
 * <p>
 * The following example defines an alias named <code>live</code> which will always point to a
 * version that represents the function as defined in your CDK app. When you change
 * your lambda code or configuration, a new resource will be created. You can
 * specify options for the current version through the <code>currentVersionOptions</code>
 * property.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.*;
 * 
 * 
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .currentVersionOptions(Map.of(
 *                 "removalPolicy", cdk.RemovalPolicy.getRETAIN(), // retain old versions
 *                 "retryAttempts", 1))
 *         .runtime(Runtime.getNODEJS_12_X())
 *         .handler("index.handler")
 *         .code(Code.fromAsset(path.join(__dirname, "lambda-handler")))
 *         .build();
 * 
 * fn.currentVersion.addAlias("live");
 * </pre></blockquote>
 * <p>
 * <h2>Layers</h2>
 * <p>
 * The <code>lambda.LayerVersion</code> class can be used to define Lambda layers and manage
 * granting permissions to other AWS accounts or organizations.
 * <p>
 * <a href="test/integ.layer-version.lit.ts">Example of Lambda Layer usage</a>
 * <p>
 * By default, updating a layer creates a new layer version, and CloudFormation will delete the old version as part of the stack update.
 * <p>
 * Alternatively, a removal policy can be used to retain the old version:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.*;
 * 
 * 
 * LayerVersion.Builder.create(this, "MyLayer")
 *         .removalPolicy(cdk.RemovalPolicy.getRETAIN())
 *         .code(Code.fromAsset(path.join(__dirname, "lambda-handler")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Event Rule Target</h2>
 * <p>
 * You can use an AWS Lambda function as a target for an Amazon CloudWatch event
 * rule:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_events;
 * import software.amazon.awscdk.aws_events_targets;
 * 
 * Rule rule = new Rule(this, "Schedule Rule", new RuleProps()
 *         .schedule(events.Schedule.cron(new CronOptions().minute("0").hour("4"))));
 * rule.addTarget(new LambdaFunction(fn));
 * </pre></blockquote>
 * <p>
 * <h2>Event Sources</h2>
 * <p>
 * AWS Lambda supports a <a href="https://docs.aws.amazon.com/lambda/latest/dg/invoking-lambda-function.html">variety of event sources</a>.
 * <p>
 * In most cases, it is possible to trigger a function as a result of an event by
 * using one of the <code>add&lt;Event&gt;Notification</code> methods on the source construct. For
 * example, the <code>s3.Bucket</code> construct has an <code>onEvent</code> method which can be used to
 * trigger a Lambda when an event, such as PutObject occurs on an S3 bucket.
 * <p>
 * An alternative way to add event sources to a function is to use <code>function.addEventSource(source)</code>.
 * This method accepts an <code>IEventSource</code> object. The module <strong>&#64;aws-cdk/aws-lambda-event-sources</strong>
 * includes classes for the various event sources supported by AWS Lambda.
 * <p>
 * For example, the following code adds an SQS queue as an event source for a function:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_lambda_event_sources;
 * import software.amazon.awscdk.aws_sqs;
 * 
 * Queue queue = new Queue(this, "Queue");
 * fn.addEventSource(new SqsEventSource(queue));
 * </pre></blockquote>
 * <p>
 * The following code adds an S3 bucket notification as an event source:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_lambda_event_sources;
 * import software.amazon.awscdk.aws_s3;
 * 
 * Bucket bucket = new Bucket(this, "Bucket");
 * fn.addEventSource(new S3EventSource(bucket, new S3EventSourceProps()
 *         .events(asList(s3.EventType.getOBJECT_CREATED(), s3.EventType.getOBJECT_REMOVED()))
 *         .filters(asList(new NotificationKeyFilter().prefix("subdir/")))));
 * </pre></blockquote>
 * <p>
 * See the documentation for the <strong>&#64;aws-cdk/aws-lambda-event-sources</strong> module for more details.
 * <p>
 * <h2>Lambda with DLQ</h2>
 * <p>
 * A dead-letter queue can be automatically created for a Lambda function by
 * setting the <code>deadLetterQueueEnabled: true</code> configuration.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.getNODEJS_12_X())
 *         .handler("index.handler")
 *         .code(Code.fromInline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"))
 *         .deadLetterQueueEnabled(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * It is also possible to provide a dead-letter queue instead of getting a new queue created:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_sqs;
 * 
 * 
 * Queue dlq = new Queue(this, "DLQ");
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.getNODEJS_12_X())
 *         .handler("index.handler")
 *         .code(Code.fromInline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"))
 *         .deadLetterQueue(dlq)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/lambda/latest/dg/dlq.html">the AWS documentation</a>
 * to learn more about AWS Lambdas and DLQs.
 * <p>
 * <h2>Lambda with X-Ray Tracing</h2>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.getNODEJS_12_X())
 *         .handler("index.handler")
 *         .code(Code.fromInline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"))
 *         .tracing(Tracing.getACTIVE())
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/lambda/latest/dg/lambda-x-ray.html">the AWS documentation</a>
 * to learn more about AWS Lambda's X-Ray support.
 * <p>
 * <h2>Lambda with Profiling</h2>
 * <p>
 * The following code configures the lambda function with CodeGuru profiling. By default, this creates a new CodeGuru
 * profiling group -
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_lambda;
 * 
 * 
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.getPYTHON_3_6())
 *         .handler("index.handler")
 *         .code(Code.fromAsset("lambda-handler"))
 *         .profiling(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>profilingGroup</code> property can be used to configure an existing CodeGuru profiler group.
 * <p>
 * CodeGuru profiling is supported for all Java runtimes and Python3.6+ runtimes.
 * <p>
 * See <a href="https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html">the AWS documentation</a>
 * to learn more about AWS Lambda's Profiling support.
 * <p>
 * <h2>Lambda with Reserved Concurrent Executions</h2>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.getNODEJS_12_X())
 *         .handler("index.handler")
 *         .code(Code.fromInline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"))
 *         .reservedConcurrentExecutions(100)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/lambda/latest/dg/concurrent-executions.html">the AWS documentation</a>
 * managing concurrency.
 * <p>
 * <h2>AutoScaling</h2>
 * <p>
 * You can use Application AutoScaling to automatically configure the provisioned concurrency for your functions. AutoScaling can be set to track utilization or be based on a schedule. To configure AutoScaling on a function alias:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_autoscaling;
 * 
 * Alias alias = new Alias(this, "Alias", new AliasProps()
 *         .aliasName("prod")
 *         .version(fn.getLatestVersion()));
 * 
 * // Create AutoScaling target
 * IScalableFunctionAttribute as = alias.addAutoScaling(new AutoScalingOptions().maxCapacity(50));
 * 
 * // Configure Target Tracking
 * as.scaleOnUtilization(new UtilizationScalingOptions()
 *         .utilizationTarget(0.5));
 * 
 * // Configure Scheduled Scaling
 * as.scaleOnSchedule("ScaleUpInTheMorning", new ScalingSchedule()
 *         .schedule(autoscaling.Schedule.cron(new CronOptions().hour("8").minute("0")))
 *         .minCapacity(20));
 * </pre></blockquote>
 * <p>
 * <a href="test/integ.autoscaling.lit.ts">Example of Lambda AutoScaling usage</a>
 * <p>
 * See <a href="https://docs.aws.amazon.com/lambda/latest/dg/invocation-scaling.html">the AWS documentation</a> on autoscaling lambda functions.
 * <p>
 * <h2>Log Group</h2>
 * <p>
 * Lambda functions automatically create a log group with the name <code>/aws/lambda/&lt;function-name&gt;</code> upon first execution with
 * log data set to never expire.
 * <p>
 * The <code>logRetention</code> property can be used to set a different expiration period.
 * <p>
 * It is possible to obtain the function's log group as a <code>logs.ILogGroup</code> by calling the <code>logGroup</code> property of the
 * <code>Function</code> construct.
 * <p>
 * By default, CDK uses the AWS SDK retry options when creating a log group. The <code>logRetentionRetryOptions</code> property
 * allows you to customize the maximum number of retries and base backoff duration.
 * <p>
 * <em>Note</em> that, if either <code>logRetention</code> is set or <code>logGroup</code> property is called, a <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cfn-customresource.html">CloudFormation custom
 * resource</a> is added
 * to the stack that pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the
 * correct log retention period (never expire, by default).
 * <p>
 * <em>Further note</em> that, if the log group already exists and the <code>logRetention</code> is not set, the custom resource will reset
 * the log retention to never expire even if it was configured with a different value.
 * <p>
 * <h2>FileSystem Access</h2>
 * <p>
 * You can configure a function to mount an Amazon Elastic File System (Amazon EFS) to a
 * directory in your runtime environment with the <code>filesystem</code> property. To access Amazon EFS
 * from lambda function, the Amazon EFS access point will be required.
 * <p>
 * The following sample allows the lambda function to mount the Amazon EFS access point to <code>/mnt/msg</code> in the runtime environment and access the filesystem with the POSIX identity defined in <code>posixUser</code>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_ec2;
 * import software.amazon.awscdk.aws_efs;
 * 
 * 
 * // create a new VPC
 * Vpc vpc = new Vpc(this, "VPC");
 * 
 * // create a new Amazon EFS filesystem
 * FileSystem fileSystem = new FileSystem(this, "Efs", new FileSystemProps().vpc(vpc));
 * 
 * // create a new access point from the filesystem
 * AccessPoint accessPoint = fileSystem.addAccessPoint("AccessPoint", new AccessPointOptions()
 *         // set /export/lambda as the root of the access point
 *         .path("/export/lambda")
 *         // as /export/lambda does not exist in a new efs filesystem, the efs will create the directory with the following createAcl
 *         .createAcl(new Acl()
 *                 .ownerUid("1001")
 *                 .ownerGid("1001")
 *                 .permissions("750"))
 *         // enforce the POSIX identity so lambda function will access with this identity
 *         .posixUser(new PosixUser()
 *                 .uid("1001")
 *                 .gid("1001")));
 * 
 * Function fn = Function.Builder.create(this, "MyLambda")
 *         // mount the access point to /mnt/msg in the lambda runtime environment
 *         .filesystem(FileSystem.fromEfsAccessPoint(accessPoint, "/mnt/msg"))
 *         .runtime(Runtime.getNODEJS_12_X())
 *         .handler("index.handler")
 *         .code(Code.fromAsset(path.join(__dirname, "lambda-handler")))
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Singleton Function</h2>
 * <p>
 * The <code>SingletonFunction</code> construct is a way to guarantee that a lambda function will be guaranteed to be part of the stack,
 * once and only once, irrespective of how many times the construct is declared to be part of the stack. This is guaranteed
 * as long as the <code>uuid</code> property and the optional <code>lambdaPurpose</code> property stay the same whenever they're declared into the
 * stack.
 * <p>
 * A typical use case of this function is when a higher level construct needs to declare a Lambda function as part of it but
 * needs to guarantee that the function is declared once. However, a user of this higher level construct can declare it any
 * number of times and with different properties. Using <code>SingletonFunction</code> here with a fixed <code>uuid</code> will guarantee this.
 * <p>
 * For example, the <code>LogRetention</code> construct requires only one single lambda function for all different log groups whose
 * retention it seeks to manage.
 * <p>
 * <h2>Bundling Asset Code</h2>
 * <p>
 * When using <code>lambda.Code.fromAsset(path)</code> it is possible to bundle the code by running a
 * command in a Docker container. The asset path will be mounted at <code>/asset-input</code>. The
 * Docker container is responsible for putting content at <code>/asset-output</code>. The content at
 * <code>/asset-output</code> will be zipped and used as Lambda code.
 * <p>
 * Example with Python:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Function.Builder.create(this, "Function")
 *         .code(Code.fromAsset(path.join(__dirname, "my-python-handler"), Map.of(
 *                 "bundling", Map.of(
 *                         "image", Runtime.PYTHON_3_8.getBundlingImage(),
 *                         "command", asList("bash", "-c", "pip install -r requirements.txt -t /asset-output &amp;&amp; cp -au . /asset-output")))))
 *         .runtime(Runtime.getPYTHON_3_8())
 *         .handler("index.handler")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Runtimes expose a <code>bundlingImage</code> property that points to the <a href="https://github.com/awslabs/aws-sam-cli">AWS SAM</a> build image.
 * <p>
 * Use <code>cdk.DockerImage.fromRegistry(image)</code> to use an existing image or
 * <code>cdk.DockerImage.fromBuild(path)</code> to build a specific image:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.*;
 * 
 * 
 * Function.Builder.create(this, "Function")
 *         .code(Code.fromAsset("/path/to/handler", Map.of(
 *                 "bundling", Map.of(
 *                         "image", cdk.DockerImage.fromBuild("/path/to/dir/with/DockerFile", new DockerBuildOptions()
 *                                 .buildArgs(Map.of(
 *                                         "ARG1", "value1"))),
 *                         "command", asList("my", "cool", "command")))))
 *         .runtime(Runtime.getPYTHON_3_8())
 *         .handler("index.handler")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Language-specific APIs</h2>
 * <p>
 * Language-specific higher level constructs are provided in separate modules:
 * <p>
 * <ul>
 * <li><code>&#64;aws-cdk/aws-lambda-nodejs</code>: <a href="https://github.com/aws/aws-cdk/tree/master/packages/%40aws-cdk/aws-lambda-nodejs">Github</a> &amp; <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-lambda-nodejs-readme.html">CDK Docs</a></li>
 * <li><code>&#64;aws-cdk/aws-lambda-python</code>: <a href="https://github.com/aws/aws-cdk/tree/master/packages/%40aws-cdk/aws-lambda-python">Github</a> &amp; <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-lambda-python-readme.html">CDK Docs</a></li>
 * </ul>
 * <p>
 * <h2>Code Signing</h2>
 * <p>
 * Code signing for AWS Lambda helps to ensure that only trusted code runs in your Lambda functions.
 * When enabled, AWS Lambda checks every code deployment and verifies that the code package is signed by a trusted source.
 * For more information, see <a href="https://docs.aws.amazon.com/lambda/latest/dg/configuration-codesigning.html">Configuring code signing for AWS Lambda</a>.
 * The following code configures a function with code signing.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_signer;
 * 
 * 
 * SigningProfile signingProfile = new SigningProfile(this, "SigningProfile", new SigningProfileProps()
 *         .platform(signer.Platform.getAWS_LAMBDA_SHA384_ECDSA()));
 * 
 * Object codeSigningConfig = CodeSigningConfig.Builder.create(this, "CodeSigningConfig")
 *         .signingProfiles(asList(signingProfile))
 *         .build();
 * 
 * Function.Builder.create(this, "Function")
 *         .codeSigningConfig(codeSigningConfig)
 *         .runtime(Runtime.getNODEJS_12_X())
 *         .handler("index.handler")
 *         .code(Code.fromAsset(path.join(__dirname, "lambda-handler")))
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.lambda;
