/**
 * <h1>Amazon CloudFront Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * Amazon CloudFront is a web service that speeds up distribution of your static and dynamic web content, such as .html, .css, .js, and image files, to
 * your users. CloudFront delivers your content through a worldwide network of data centers called edge locations. When a user requests content that
 * you're serving with CloudFront, the user is routed to the edge location that provides the lowest latency, so that content is delivered with the best
 * possible performance.
 * <p>
 * <h2>Distribution API</h2>
 * <p>
 * The <code>Distribution</code> API is currently being built to replace the existing <code>CloudFrontWebDistribution</code> API. The <code>Distribution</code> API is optimized for the
 * most common use cases of CloudFront distributions (e.g., single origin and behavior, few customizations) while still providing the ability for more
 * advanced use cases. The API focuses on simplicity for the common use cases, and convenience methods for creating the behaviors and origins necessary
 * for more complex use cases.
 * <p>
 * <h3>Creating a distribution</h3>
 * <p>
 * CloudFront distributions deliver your content from one or more origins; an origin is the location where you store the original version of your
 * content. Origins can be created from S3 buckets or a custom origin (HTTP server). Constructs to define origins are in the <code>&#64;aws-cdk/aws-cloudfront-origins</code> module.
 * <p>
 * Each distribution has a default behavior which applies to all requests to that distribution, and routes requests to a primary origin.
 * Additional behaviors may be specified for an origin with a given URL path pattern. Behaviors allow routing with multiple origins,
 * controlling which HTTP methods to support, whether to require users to use HTTPS, and what query strings or cookies to forward to your origin,
 * among other settings.
 * <p>
 * <h4>From an S3 Bucket</h4>
 * <p>
 * An S3 bucket can be added as an origin. If the bucket is configured as a website endpoint, the distribution can use S3 redirects and S3 custom error
 * documents.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_cloudfront;
 * import software.amazon.awscdk.aws_cloudfront_origins;
 * 
 * 
 * // Creates a distribution for a S3 bucket.
 * Bucket myBucket = new Bucket(this, "myBucket");
 * new Distribution(this, "myDist", new DistributionProps()
 *         .defaultBehavior(new BehaviorOptions().origin(new S3Origin(myBucket))));
 * </pre></blockquote>
 * <p>
 * The above will treat the bucket differently based on if <code>IBucket.isWebsite</code> is set or not. If the bucket is configured as a website, the bucket is
 * treated as an HTTP origin, and the built-in S3 redirects and error pages can be used. Otherwise, the bucket is handled as a bucket origin and
 * CloudFront's redirect and error handling will be used. In the latter case, the Origin will create an origin access identity and grant it access to the
 * underlying bucket. This can be used in conjunction with a bucket that is not public to require that your users access your content using CloudFront
 * URLs and not S3 URLs directly.
 * <p>
 * <h4>ELBv2 Load Balancer</h4>
 * <p>
 * An Elastic Load Balancing (ELB) v2 load balancer may be used as an origin. In order for a load balancer to serve as an origin, it must be publicly
 * accessible (<code>internetFacing</code> is true). Both Application and Network load balancers are supported.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_ec2;
 * import software.amazon.awscdk.aws_elasticloadbalancingv2;
 * 
 * 
 * Vpc vpc = new Vpc(...);
 * // Create an application load balancer in a VPC. 'internetFacing' must be 'true'
 * // for CloudFront to access the load balancer and use it as an origin.
 * ApplicationLoadBalancer lb = new ApplicationLoadBalancer(this, "LB", new ApplicationLoadBalancerProps()
 *         .vpc(vpc)
 *         .internetFacing(true));
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(Map.of("origin", new LoadBalancerV2Origin(lb)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>From an HTTP endpoint</h4>
 * <p>
 * Origins can also be created from any other HTTP endpoint, given the domain name, and optionally, other origin properties.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(Map.of("origin", new HttpOrigin("www.example.com")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Domain Names and Certificates</h3>
 * <p>
 * When you create a distribution, CloudFront assigns a domain name for the distribution, for example: <code>d111111abcdef8.cloudfront.net</code>; this value can
 * be retrieved from <code>distribution.distributionDomainName</code>. CloudFront distributions use a default certificate (<code>*.cloudfront.net</code>) to support HTTPS by
 * default. If you want to use your own domain name, such as <code>www.example.com</code>, you must associate a certificate with your distribution that contains
 * your domain name, and provide one (or more) domain names from the certificate for the distribution.
 * <p>
 * The certificate must be present in the AWS Certificate Manager (ACM) service in the US East (N. Virginia) region; the certificate
 * may either be created by ACM, or created elsewhere and imported into ACM. When a certificate is used, the distribution will support HTTPS connections
 * from SNI only and a minimum protocol version of TLSv1.2_2021 if the '&#64;aws-cdk/aws-cloudfront:defaultSecurityPolicyTLSv1.2_2021' feature flag is set, and TLSv1.2_2019 otherwise.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object myCertificate = DnsValidatedCertificate.Builder.create(this, "mySiteCert")
 *         .domainName("www.example.com")
 *         .hostedZone(hostedZone)
 *         .build();
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(Map.of("origin", new S3Origin(myBucket)))
 *         .domainNames(asList("www.example.com"))
 *         .certificate(myCertificate)
 *         .build();
 * </pre></blockquote>
 * <p>
 * However, you can customize the minimum protocol version for the certificate while creating the distribution using <code>minimumProtocolVersion</code> property.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(Map.of("origin", new S3Origin(myBucket)))
 *         .domainNames(asList("www.example.com"))
 *         .minimumProtocolVersion(SecurityPolicyProtocol.getTLS_V1_2016())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Multiple Behaviors &amp; Origins</h3>
 * <p>
 * Each distribution has a default behavior which applies to all requests to that distribution; additional behaviors may be specified for a
 * given URL path pattern. Behaviors allow routing with multiple origins, controlling which HTTP methods to support, whether to require users to
 * use HTTPS, and what query strings or cookies to forward to your origin, among others.
 * <p>
 * The properties of the default behavior can be adjusted as part of the distribution creation. The following example shows configuring the HTTP
 * methods and viewer protocol policy of the cache.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object myWebDistribution = Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(Map.of(
 *                 "origin", new S3Origin(myBucket),
 *                 "allowedMethods", AllowedMethods.getALLOW_ALL(),
 *                 "viewerProtocolPolicy", ViewerProtocolPolicy.getREDIRECT_TO_HTTPS()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Additional behaviors can be specified at creation, or added after the initial creation. Each additional behavior is associated with an origin,
 * and enable customization for a specific set of resources based on a URL path pattern. For example, we can add a behavior to <code>myWebDistribution</code> to
 * override the default viewer protocol policy for all of the images.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * myWebDistribution.addBehavior("/images/*.jpg", new S3Origin(myBucket), Map.of(
 *         "viewerProtocolPolicy", ViewerProtocolPolicy.getREDIRECT_TO_HTTPS()));
 * </pre></blockquote>
 * <p>
 * These behaviors can also be specified at distribution creation time.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bucketOrigin = new S3Origin(myBucket);
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(Map.of(
 *                 "origin", bucketOrigin,
 *                 "allowedMethods", AllowedMethods.getALLOW_ALL(),
 *                 "viewerProtocolPolicy", ViewerProtocolPolicy.getREDIRECT_TO_HTTPS()))
 *         .additionalBehaviors(Map.of(
 *                 "/images/*.jpg", Map.of(
 *                         "origin", bucketOrigin,
 *                         "viewerProtocolPolicy", ViewerProtocolPolicy.getREDIRECT_TO_HTTPS())))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Customizing Cache Keys and TTLs with Cache Policies</h3>
 * <p>
 * You can use a cache policy to improve your cache hit ratio by controlling the values (URL query strings, HTTP headers, and cookies)
 * that are included in the cache key, and/or adjusting how long items remain in the cache via the time-to-live (TTL) settings.
 * CloudFront provides some predefined cache policies, known as managed policies, for common use cases. You can use these managed policies,
 * or you can create your own cache policy that’s specific to your needs.
 * See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html for more details.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Using an existing cache policy
 * // Using an existing cache policy
 * Distribution.Builder.create(this, "myDistManagedPolicy")
 *         .defaultBehavior(Map.of(
 *                 "origin", bucketOrigin,
 *                 "cachePolicy", cloudfront.CachePolicy.getCACHING_OPTIMIZED()))
 *         .build();
 * 
 * // Creating a custom cache policy  -- all parameters optional
 * Object myCachePolicy = CachePolicy.Builder.create(this, "myCachePolicy")
 *         .cachePolicyName("MyPolicy")
 *         .comment("A default policy")
 *         .defaultTtl(Duration.days(2))
 *         .minTtl(Duration.minutes(1))
 *         .maxTtl(Duration.days(10))
 *         .cookieBehavior(cloudfront.CacheCookieBehavior.all())
 *         .headerBehavior(cloudfront.CacheHeaderBehavior.allowList("X-CustomHeader"))
 *         .queryStringBehavior(cloudfront.CacheQueryStringBehavior.denyList("username"))
 *         .enableAcceptEncodingGzip(true)
 *         .enableAcceptEncodingBrotli(true)
 *         .build();
 * Distribution.Builder.create(this, "myDistCustomPolicy")
 *         .defaultBehavior(Map.of(
 *                 "origin", bucketOrigin,
 *                 "cachePolicy", myCachePolicy))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Customizing Origin Requests with Origin Request Policies</h3>
 * <p>
 * When CloudFront makes a request to an origin, the URL path, request body (if present), and a few standard headers are included.
 * Other information from the viewer request, such as URL query strings, HTTP headers, and cookies, is not included in the origin request by default.
 * You can use an origin request policy to control the information that’s included in an origin request.
 * CloudFront provides some predefined origin request policies, known as managed policies, for common use cases. You can use these managed policies,
 * or you can create your own origin request policy that’s specific to your needs.
 * See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html for more details.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Using an existing origin request policy
 * // Using an existing origin request policy
 * Distribution.Builder.create(this, "myDistManagedPolicy")
 *         .defaultBehavior(Map.of(
 *                 "origin", bucketOrigin,
 *                 "originRequestPolicy", cloudfront.OriginRequestPolicy.getCORS_S3_ORIGIN()))
 *         .build();
 * // Creating a custom origin request policy -- all parameters optional
 * Object myOriginRequestPolicy = OriginRequestPolicy.Builder.create(stack, "OriginRequestPolicy")
 *         .originRequestPolicyName("MyPolicy")
 *         .comment("A default policy")
 *         .cookieBehavior(cloudfront.OriginRequestCookieBehavior.none())
 *         .headerBehavior(cloudfront.OriginRequestHeaderBehavior.all("CloudFront-Is-Android-Viewer"))
 *         .queryStringBehavior(cloudfront.OriginRequestQueryStringBehavior.allowList("username"))
 *         .build();
 * Distribution.Builder.create(this, "myDistCustomPolicy")
 *         .defaultBehavior(Map.of(
 *                 "origin", bucketOrigin,
 *                 "cachePolicy", myCachePolicy,
 *                 "originRequestPolicy", myOriginRequestPolicy))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Validating signed URLs or signed cookies with Trusted Key Groups</h3>
 * <p>
 * CloudFront Distribution now supports validating signed URLs or signed cookies using key groups. When a cache behavior contains trusted key groups, CloudFront requires signed URLs or signed cookies for all requests that match the cache behavior.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // public key in PEM format
 * Object pubKey = PublicKey.Builder.create(stack, "MyPubKey")
 *         .encodedKey(publicKey)
 *         .build();
 * 
 * Object keyGroup = KeyGroup.Builder.create(stack, "MyKeyGroup")
 *         .items(asList(pubKey))
 *         .build();
 * 
 * Distribution.Builder.create(stack, "Dist")
 *         .defaultBehavior(Map.of(
 *                 "origin", new HttpOrigin("www.example.com"),
 *                 "trustedKeyGroups", asList(keyGroup)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Lambda&#64;Edge</h3>
 * <p>
 * Lambda&#64;Edge is an extension of AWS Lambda, a compute service that lets you execute functions that customize the content that CloudFront delivers.
 * You can author Node.js or Python functions in the US East (N. Virginia) region,
 * and then execute them in AWS locations globally that are closer to the viewer,
 * without provisioning or managing servers.
 * Lambda&#64;Edge functions are associated with a specific behavior and event type.
 * Lambda&#64;Edge can be used to rewrite URLs,
 * alter responses based on headers or cookies,
 * or authorize requests based on headers or authorization tokens.
 * <p>
 * The following shows a Lambda&#64;Edge function added to the default behavior and triggered on every request:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object myFunc = EdgeFunction.Builder.create(this, "MyFunction")
 *         .runtime(lambda.Runtime.getNODEJS_12_X())
 *         .handler("index.handler")
 *         .code(lambda.Code.fromAsset(path.join(__dirname, "lambda-handler")))
 *         .build();
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(Map.of(
 *                 "origin", new S3Origin(myBucket),
 *                 "edgeLambdas", asList(Map.of(
 *                         "functionVersion", myFunc.getCurrentVersion(),
 *                         "eventType", cloudfront.LambdaEdgeEventType.getVIEWER_REQUEST()))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * <strong>Note:</strong> Lambda&#64;Edge functions must be created in the <code>us-east-1</code> region, regardless of the region of the CloudFront distribution and stack.
 * To make it easier to request functions for Lambda&#64;Edge, the <code>EdgeFunction</code> construct can be used.
 * The <code>EdgeFunction</code> construct will automatically request a function in <code>us-east-1</code>, regardless of the region of the current stack.
 * <code>EdgeFunction</code> has the same interface as <code>Function</code> and can be created and used interchangeably.
 * Please note that using <code>EdgeFunction</code> requires that the <code>us-east-1</code> region has been bootstrapped.
 * See https://docs.aws.amazon.com/cdk/latest/guide/bootstrapping.html for more about bootstrapping regions.
 * <p>
 * </blockquote>
 * <p>
 * If the stack is in <code>us-east-1</code>, a "normal" <code>lambda.Function</code> can be used instead of an <code>EdgeFunction</code>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Function myFunc = new Function(this, "MyFunction", new FunctionProps()
 *         .runtime(lambda.Runtime.getNODEJS_12_X())
 *         .handler("index.handler")
 *         .code(lambda.Code.fromAsset(path.join(__dirname, "lambda-handler"))));
 * </pre></blockquote>
 * <p>
 * If the stack is not in <code>us-east-1</code>, and you need references from different applications on the same account,
 * you can also set a specific stack ID for each Lambda&#64;Edge.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object myFunc1 = EdgeFunction.Builder.create(this, "MyFunction1")
 *         .runtime(lambda.Runtime.getNODEJS_12_X())
 *         .handler("index.handler")
 *         .code(lambda.Code.fromAsset(path.join(__dirname, "lambda-handler1")))
 *         .stackId("edge-lambda-stack-id-1")
 *         .build();
 * 
 * Object myFunc2 = EdgeFunction.Builder.create(this, "MyFunction2")
 *         .runtime(lambda.Runtime.getNODEJS_12_X())
 *         .handler("index.handler")
 *         .code(lambda.Code.fromAsset(path.join(__dirname, "lambda-handler2")))
 *         .stackId("edge-lambda-stack-id-2")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Lambda&#64;Edge functions can also be associated with additional behaviors,
 * either at or after Distribution creation time.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // assigning at Distribution creation
 * Object myOrigin = new S3Origin(myBucket);
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(Map.of("origin", myOrigin))
 *         .additionalBehaviors(Map.of(
 *                 "images/*", Map.of(
 *                         "origin", myOrigin,
 *                         "edgeLambdas", asList(Map.of(
 *                                 "functionVersion", myFunc.getCurrentVersion(),
 *                                 "eventType", cloudfront.LambdaEdgeEventType.getORIGIN_REQUEST(),
 *                                 "includeBody", true)))))
 *         .build();
 * 
 * // assigning after creation
 * myDistribution.addBehavior("images/*", myOrigin, Map.of(
 *         "edgeLambdas", asList(Map.of(
 *                 "functionVersion", myFunc.getCurrentVersion(),
 *                 "eventType", cloudfront.LambdaEdgeEventType.getVIEWER_RESPONSE()))));
 * </pre></blockquote>
 * <p>
 * Adding an existing Lambda&#64;Edge function created in a different stack to a CloudFront distribution.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * IVersion functionVersion = lambda.Version.fromVersionArn(this, "Version", "arn:aws:lambda:us-east-1:123456789012:function:functionName:1");
 * 
 * Distribution.Builder.create(this, "distro")
 *         .defaultBehavior(Map.of(
 *                 "origin", new S3Origin(s3Bucket),
 *                 "edgeLambdas", asList(Map.of(
 *                         "functionVersion", functionVersion,
 *                         "eventType", cloudfront.LambdaEdgeEventType.getVIEWER_REQUEST()))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>CloudFront Function</h3>
 * <p>
 * You can also deploy CloudFront functions and add them to a CloudFront distribution.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object cfFunction = Function.Builder.create(stack, "Function")
 *         .code(cloudfront.FunctionCode.fromInline("function handler(event) { return event.request }"))
 *         .build();
 * 
 * Distribution.Builder.create(stack, "distro")
 *         .defaultBehavior(Map.of(
 *                 "origin", new S3Origin(s3Bucket),
 *                 "functionAssociations", asList(Map.of(
 *                         "function", cfFunction,
 *                         "eventType", cloudfront.FunctionEventType.getVIEWER_REQUEST()))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * It will auto-generate the name of the function and deploy it to the <code>live</code> stage.
 * <p>
 * Additionally, you can load the function's code from a file using the <code>FunctionCode.fromFile()</code> method.
 * <p>
 * <h3>Logging</h3>
 * <p>
 * You can configure CloudFront to create log files that contain detailed information about every user request that CloudFront receives.
 * The logs can go to either an existing bucket, or a bucket will be created for you.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Simplest form - creates a new bucket and logs to it.
 * // Simplest form - creates a new bucket and logs to it.
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(Map.of("origin", new HttpOrigin("www.example.com")))
 *         .enableLogging(true)
 *         .build();
 * 
 * // You can optionally log to a specific bucket, configure whether cookies are logged, and give the log files a prefix.
 * // You can optionally log to a specific bucket, configure whether cookies are logged, and give the log files a prefix.
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(Map.of("origin", new HttpOrigin("www.example.com")))
 *         .enableLogging(true)// Optional, this is implied if logBucket is specified
 *         .logBucket(new Bucket(this, "LogBucket"))
 *         .logFilePrefix("distribution-access-logs/")
 *         .logIncludesCookies(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Importing Distributions</h3>
 * <p>
 * Existing distributions can be imported as well; note that like most imported constructs, an imported distribution cannot be modified.
 * However, it can be used as a reference for other higher-level constructs.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object distribution = cloudfront.Distribution.fromDistributionAttributes(scope, "ImportedDist", Map.of(
 *         "domainName", "d111111abcdef8.cloudfront.net",
 *         "distributionId", "012345ABCDEF"));
 * </pre></blockquote>
 * <p>
 * <h2>CloudFrontWebDistribution API</h2>
 * <p>
 * <blockquote>
 * <p>
 * The <code>CloudFrontWebDistribution</code> construct is the original construct written for working with CloudFront distributions.
 * Users are encouraged to use the newer <code>Distribution</code> instead, as it has a simpler interface and receives new features faster.
 * <p>
 * </blockquote>
 * <p>
 * Example usage:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object sourceBucket = new Bucket(this, "Bucket");
 * 
 * Object distribution = CloudFrontWebDistribution.Builder.create(this, "MyDistribution")
 *         .originConfigs(asList(Map.of(
 *                 "s3OriginSource", Map.of(
 *                         "s3BucketSource", sourceBucket),
 *                 "behaviors", asList(Map.of("isDefaultBehavior", true)))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Viewer certificate</h3>
 * <p>
 * By default, CloudFront Web Distributions will answer HTTPS requests with CloudFront's default certificate, only containing the distribution <code>domainName</code> (e.g. d111111abcdef8.cloudfront.net).
 * You can customize the viewer certificate property to provide a custom certificate and/or list of domain name aliases to fit your needs.
 * <p>
 * See <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-https-alternate-domain-names.html">Using Alternate Domain Names and HTTPS</a> in the CloudFront User Guide.
 * <p>
 * <h4>Default certificate</h4>
 * <p>
 * You can customize the default certificate aliases. This is intended to be used in combination with CNAME records in your DNS zone.
 * <p>
 * Example:
 * <p>
 * <a href="test/example.default-cert-alias.lit.ts">create a distribution with an default certificate example</a>
 * <p>
 * <h4>ACM certificate</h4>
 * <p>
 * You can change the default certificate by one stored AWS Certificate Manager, or ACM.
 * Those certificate can either be generated by AWS, or purchased by another CA imported into ACM.
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-certificatemanager-readme.html">the aws-certificatemanager module documentation</a> or <a href="https://docs.aws.amazon.com/acm/latest/userguide/import-certificate.html">Importing Certificates into AWS Certificate Manager</a> in the AWS Certificate Manager User Guide.
 * <p>
 * Example:
 * <p>
 * <a href="test/example.acm-cert-alias.lit.ts">create a distribution with an acm certificate example</a>
 * <p>
 * <h4>IAM certificate</h4>
 * <p>
 * You can also import a certificate into the IAM certificate store.
 * <p>
 * See <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-and-https-procedures.html#cnames-and-https-uploading-certificates">Importing an SSL/TLS Certificate</a> in the CloudFront User Guide.
 * <p>
 * Example:
 * <p>
 * <a href="test/example.iam-cert-alias.lit.ts">create a distribution with an iam certificate example</a>
 * <p>
 * <h3>Trusted Key Groups</h3>
 * <p>
 * CloudFront Web Distributions supports validating signed URLs or signed cookies using key groups. When a cache behavior contains trusted key groups, CloudFront requires signed URLs or signed cookies for all requests that match the cache behavior.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object pubKey = PublicKey.Builder.create(stack, "MyPubKey")
 *         .encodedKey(publicKey)
 *         .build();
 * 
 * Object keyGroup = KeyGroup.Builder.create(stack, "MyKeyGroup")
 *         .items(asList(pubKey))
 *         .build();
 * 
 * CloudFrontWebDistribution.Builder.create(stack, "AnAmazingWebsiteProbably")
 *         .originConfigs(asList(Map.of(
 *                 "s3OriginSource", Map.of(
 *                         "s3BucketSource", sourceBucket),
 *                 "behaviors", asList(Map.of(
 *                         "isDefaultBehavior", true,
 *                         "trustedKeyGroups", asList(keyGroup))))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Restrictions</h3>
 * <p>
 * CloudFront supports adding restrictions to your distribution.
 * <p>
 * See <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/georestrictions.html">Restricting the Geographic Distribution of Your Content</a> in the CloudFront User Guide.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * CloudFrontWebDistribution.Builder.create(stack, "MyDistribution")
 *         //...
 *         .geoRestriction(GeoRestriction.whitelist("US", "UK"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Connection behaviors between CloudFront and your origin</h3>
 * <p>
 * CloudFront provides you even more control over the connection behaviors between CloudFront and your origin. You can now configure the number of connection attempts CloudFront will make to your origin and the origin connection timeout for each attempt.
 * <p>
 * See <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-attempts">Origin Connection Attempts</a>
 * <p>
 * See <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-timeout">Origin Connection Timeout</a>
 * <p>
 * Example usage:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object distribution = CloudFrontWebDistribution.Builder.create(this, "MyDistribution")
 *         .originConfigs(asList(Map.of(...,
 *                 "connectionAttempts", 3,
 *                 "connectionTimeout", cdk.Duration.seconds(10))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Origin Fallback</h4>
 * <p>
 * In case the origin source is not available and answers with one of the
 * specified status code the failover origin source will be used.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * CloudFrontWebDistribution.Builder.create(stack, "ADistribution")
 *         .originConfigs(asList(Map.of(
 *                 "s3OriginSource", Map.of(
 *                         "s3BucketSource", s3.Bucket.fromBucketName(stack, "aBucket", "myoriginbucket"),
 *                         "originPath", "/",
 *                         "originHeaders", Map.of(
 *                                 "myHeader", "42"),
 *                         "originShieldRegion", "us-west-2"),
 *                 "failoverS3OriginSource", Map.of(
 *                         "s3BucketSource", s3.Bucket.fromBucketName(stack, "aBucketFallback", "myoriginbucketfallback"),
 *                         "originPath", "/somewhere",
 *                         "originHeaders", Map.of(
 *                                 "myHeader2", "21"),
 *                         "originShieldRegion", "us-east-1"),
 *                 "failoverCriteriaStatusCodes", asList(FailoverStatusCode.getINTERNAL_SERVER_ERROR()),
 *                 "behaviors", asList(Map.of(
 *                         "isDefaultBehavior", true)))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>KeyGroup &amp; PublicKey API</h2>
 * <p>
 * Now you can create a key group to use with CloudFront signed URLs and signed cookies. You can add public keys to use with CloudFront features such as signed URLs, signed cookies, and field-level encryption.
 * <p>
 * The following example command uses OpenSSL to generate an RSA key pair with a length of 2048 bits and save to the file named <code>private_key.pem</code>.
 * <p>
 * <blockquote><pre>
 * openssl genrsa -out private_key.pem 2048
 * </pre></blockquote>
 * <p>
 * The resulting file contains both the public and the private key. The following example command extracts the public key from the file named <code>private_key.pem</code> and stores it in <code>public_key.pem</code>.
 * <p>
 * <blockquote><pre>
 * openssl rsa -pubout -in private_key.pem -out public_key.pem
 * </pre></blockquote>
 * <p>
 * Note: Don't forget to copy/paste the contents of <code>public_key.pem</code> file including <code>-----BEGIN PUBLIC KEY-----</code> and <code>-----END PUBLIC KEY-----</code> lines into <code>encodedKey</code> parameter when creating a <code>PublicKey</code>.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * KeyGroup.Builder.create(stack, "MyKeyGroup")
 *         .items(asList(
 *             PublicKey.Builder.create(stack, "MyPublicKey")
 *                     .encodedKey("...")
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * See:
 * <p>
 * <ul>
 * <li>https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html</li>
 * <li>https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-trusted-signers.html</li>
 * </ul>
 */
package software.amazon.awscdk.services.cloudfront;
