/**
 * <h1>Amazon S3 Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * Define an unencrypted S3 bucket.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * new Bucket(this, "MyFirstBucket");
 * </pre></blockquote>
 * <p>
 * <code>Bucket</code> constructs expose the following deploy-time attributes:
 * <p>
 * <ul>
 * <li><code>bucketArn</code> - the ARN of the bucket (i.e. <code>arn:aws:s3:::bucket_name</code>)</li>
 * <li><code>bucketName</code> - the name of the bucket (i.e. <code>bucket_name</code>)</li>
 * <li><code>bucketWebsiteUrl</code> - the Website URL of the bucket (i.e.
 * <code>http://bucket_name.s3-website-us-west-1.amazonaws.com</code>)</li>
 * <li><code>bucketDomainName</code> - the URL of the bucket (i.e. <code>bucket_name.s3.amazonaws.com</code>)</li>
 * <li><code>bucketDualStackDomainName</code> - the dual-stack URL of the bucket (i.e.
 * <code>bucket_name.s3.dualstack.eu-west-1.amazonaws.com</code>)</li>
 * <li><code>bucketRegionalDomainName</code> - the regional URL of the bucket (i.e.
 * <code>bucket_name.s3.eu-west-1.amazonaws.com</code>)</li>
 * <li><code>arnForObjects(pattern)</code> - the ARN of an object or objects within the bucket (i.e.
 * <code>arn:aws:s3:::bucket_name/exampleobject.png</code> or
 * <code>arn:aws:s3:::bucket_name/Development/*</code>)</li>
 * <li><code>urlForObject(key)</code> - the HTTP URL of an object within the bucket (i.e.
 * <code>https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey</code>)</li>
 * <li><code>virtualHostedUrlForObject(key)</code> - the virtual-hosted style HTTP URL of an object
 * within the bucket (i.e. <code>https://china-bucket-s3.cn-north-1.amazonaws.com.cn/mykey</code>)</li>
 * <li><code>s3UrlForObject(key)</code> - the S3 URL of an object within the bucket (i.e.
 * <code>s3://bucket/mykey</code>)</li>
 * </ul>
 * <p>
 * <h2>Encryption</h2>
 * <p>
 * Define a KMS-encrypted bucket:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bucket = Bucket.Builder.create(this, "MyEncryptedBucket")
 *         .encryption(BucketEncryption.getKMS())
 *         .build();
 * 
 * // you can access the encryption key:
 * assert(bucket.getEncryptionKey() instanceof kms.getKey());
 * </pre></blockquote>
 * <p>
 * You can also supply your own key:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object myKmsKey = new Key(this, "MyKey");
 * 
 * Object bucket = Bucket.Builder.create(this, "MyEncryptedBucket")
 *         .encryption(BucketEncryption.getKMS())
 *         .encryptionKey(myKmsKey)
 *         .build();
 * 
 * assert(bucket.getEncryptionKey() === myKmsKey);
 * </pre></blockquote>
 * <p>
 * Enable KMS-SSE encryption via <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/bucket-key.html">S3 Bucket Keys</a>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bucket = Bucket.Builder.create(this, "MyEncryptedBucket")
 *         .encryption(BucketEncryption.getKMS())
 *         .bucketKeyEnabled(true)
 *         .build();
 * 
 * assert(bucket.getBucketKeyEnabled() === true);
 * </pre></blockquote>
 * <p>
 * Use <code>BucketEncryption.ManagedKms</code> to use the S3 master KMS key:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bucket = Bucket.Builder.create(this, "Buck")
 *         .encryption(BucketEncryption.getKMS_MANAGED())
 *         .build();
 * 
 * assert(bucket.getEncryptionKey() == null);
 * </pre></blockquote>
 * <p>
 * <h2>Permissions</h2>
 * <p>
 * A bucket policy will be automatically created for the bucket upon the first call to
 * <code>addToResourcePolicy(statement)</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bucket = new Bucket(this, "MyBucket");
 * Object result = bucket.addToResourcePolicy(new PolicyStatement(new PolicyStatementProps()
 *         .actions(asList("s3:GetObject"))
 *         .resources(asList(bucket.arnForObjects("file.txt")))
 *         .principals(asList(new AccountRootPrincipal()))));
 * </pre></blockquote>
 * <p>
 * If you try to add a policy statement to an existing bucket, this method will
 * not do anything:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bucket = Bucket.fromBucketName(this, "existingBucket", "bucket-name");
 * 
 * // Nothing will change here
 * Object result = bucket.addToResourcePolicy(new PolicyStatement(new PolicyStatementProps()...));
 * </pre></blockquote>
 * <p>
 * That's because it's not possible to tell whether the bucket
 * already has a policy attached, let alone to re-use that policy to add more
 * statements to it. We recommend that you always check the result of the call:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object result = bucket.addToResourcePolicy(...);
 * if (~result.getStatementAdded()) {
 * }
 * </pre></blockquote>
 * <p>
 * The bucket policy can be directly accessed after creation to add statements or
 * adjust the removal policy.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * bucket.policy.applyRemovalPolicy(RemovalPolicy.getRETAIN());
 * </pre></blockquote>
 * <p>
 * Most of the time, you won't have to manipulate the bucket policy directly.
 * Instead, buckets have "grant" methods called to give prepackaged sets of permissions
 * to other resources. For example:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object lambda = Function.Builder.create(this, "Lambda").build();
 * 
 * Object bucket = new Bucket(this, "MyBucket");
 * bucket.grantReadWrite(lambda);
 * </pre></blockquote>
 * <p>
 * Will give the Lambda's execution role permissions to read and write
 * from the bucket.
 * <p>
 * <h2>AWS Foundational Security Best Practices</h2>
 * <p>
 * <h3>Enforcing SSL</h3>
 * <p>
 * To require all requests use Secure Socket Layer (SSL):
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bucket = Bucket.Builder.create(this, "Bucket")
 *         .enforceSSL(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Sharing buckets between stacks</h2>
 * <p>
 * To use a bucket in a different stack in the same CDK application, pass the object to the other stack:
 * <p>
 * <a href="test/integ.bucket-sharing.lit.ts">sharing bucket between stacks</a>
 * <p>
 * <h2>Importing existing buckets</h2>
 * <p>
 * To import an existing bucket into your CDK application, use the <code>Bucket.fromBucketAttributes</code>
 * factory method. This method accepts <code>BucketAttributes</code> which describes the properties of an already
 * existing bucket:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bucket = Bucket.fromBucketAttributes(this, "ImportedBucket", Map.of(
 *         "bucketArn", "arn:aws:s3:::my-bucket"));
 * 
 * // now you can just call methods on the bucket
 * bucket.addEventNotification(EventType.getOBJECT_CREATED(), ...);
 * </pre></blockquote>
 * <p>
 * Alternatively, short-hand factories are available as <code>Bucket.fromBucketName</code> and
 * <code>Bucket.fromBucketArn</code>, which will derive all bucket attributes from the bucket
 * name or ARN respectively:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object byName = Bucket.fromBucketName(this, "BucketByName", "my-bucket");
 * Object byArn = Bucket.fromBucketArn(this, "BucketByArn", "arn:aws:s3:::my-bucket");
 * </pre></blockquote>
 * <p>
 * The bucket's region defaults to the current stack's region, but can also be explicitly set in cases where one of the bucket's
 * regional properties needs to contain the correct values.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object myCrossRegionBucket = Bucket.fromBucketAttributes(this, "CrossRegionImport", Map.of(
 *         "bucketArn", "arn:aws:s3:::my-bucket",
 *         "region", "us-east-1"));
 * </pre></blockquote>
 * <p>
 * <h2>Bucket Notifications</h2>
 * <p>
 * The Amazon S3 notification feature enables you to receive notifications when
 * certain events happen in your bucket as described under [S3 Bucket
 * Notifications] of the S3 Developer Guide.
 * <p>
 * To subscribe for bucket notifications, use the <code>bucket.addEventNotification</code> method. The
 * <code>bucket.addObjectCreatedNotification</code> and <code>bucket.addObjectRemovedNotification</code> can also be used for
 * these common use cases.
 * <p>
 * The following example will subscribe an SNS topic to be notified of all <code>s3:ObjectCreated:*</code> events:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_s3_notifications;
 * 
 * 
 * Topic topic = new Topic(this, "MyTopic");
 * bucket.addEventNotification(s3.EventType.getOBJECT_CREATED(), new SnsDestination(topic));
 * </pre></blockquote>
 * <p>
 * This call will also ensure that the topic policy can accept notifications for
 * this specific bucket.
 * <p>
 * Supported S3 notification targets are exposed by the <code>&#64;aws-cdk/aws-s3-notifications</code> package.
 * <p>
 * It is also possible to specify S3 object key filters when subscribing. The
 * following example will notify <code>myQueue</code> when objects prefixed with <code>foo/</code> and
 * have the <code>.jpg</code> suffix are removed from the bucket.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * bucket.addEventNotification(s3.EventType.getOBJECT_REMOVED(),
 * new SqsDestination(myQueue), Map.of("prefix", "foo/", "suffix", ".jpg"));
 * </pre></blockquote>
 * <p>
 * Adding notifications on existing buckets:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bucket = Bucket.fromBucketAttributes(this, "ImportedBucket", Map.of(
 *         "bucketArn", "arn:aws:s3:::my-bucket"));
 * bucket.addEventNotification(s3.EventType.getOBJECT_CREATED(), new SnsDestination(topic));
 * </pre></blockquote>
 * <p>
 * <h2>Block Public Access</h2>
 * <p>
 * Use <code>blockPublicAccess</code> to specify <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html">block public access settings</a> on the bucket.
 * <p>
 * Enable all block public access settings:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bucket = Bucket.Builder.create(this, "MyBlockedBucket")
 *         .blockPublicAccess(BlockPublicAccess.getBLOCK_ALL())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Block and ignore public ACLs:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bucket = Bucket.Builder.create(this, "MyBlockedBucket")
 *         .blockPublicAccess(BlockPublicAccess.getBLOCK_ACLS())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, specify the settings manually:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bucket = Bucket.Builder.create(this, "MyBlockedBucket")
 *         .blockPublicAccess(BlockPublicAccess.Builder.create().blockPublicPolicy(true).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * When <code>blockPublicPolicy</code> is set to <code>true</code>, <code>grantPublicRead()</code> throws an error.
 * <p>
 * <h2>Logging configuration</h2>
 * <p>
 * Use <code>serverAccessLogsBucket</code> to describe where server access logs are to be stored.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object accessLogsBucket = new Bucket(this, "AccessLogsBucket");
 * 
 * Object bucket = Bucket.Builder.create(this, "MyBucket")
 *         .serverAccessLogsBucket(accessLogsBucket)
 *         .build();
 * </pre></blockquote>
 * <p>
 * It's also possible to specify a prefix for Amazon S3 to assign to all log object keys.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bucket = Bucket.Builder.create(this, "MyBucket")
 *         .serverAccessLogsBucket(accessLogsBucket)
 *         .serverAccessLogsPrefix("logs")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>S3 Inventory</h2>
 * <p>
 * An <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage-inventory.html">inventory</a> contains a list of the objects in the source bucket and metadata for each object. The inventory lists are stored in the destination bucket as a CSV file compressed with GZIP, as an Apache optimized row columnar (ORC) file compressed with ZLIB, or as an Apache Parquet (Parquet) file compressed with Snappy.
 * <p>
 * You can configure multiple inventory lists for a bucket. You can configure what object metadata to include in the inventory, whether to list all object versions or only current versions, where to store the inventory list file output, and whether to generate the inventory on a daily or weekly basis.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Bucket inventoryBucket = new Bucket(this, "InventoryBucket");
 * 
 * Bucket dataBucket = new Bucket(this, "DataBucket", new BucketProps()
 *         .inventories(asList(Map.of(
 *                 "frequency", s3.InventoryFrequency.getDAILY(),
 *                 "includeObjectVersions", s3.InventoryObjectVersion.getCURRENT(),
 *                 "destination", Map.of(
 *                         "bucket", inventoryBucket)), Map.of(
 *                 "frequency", s3.InventoryFrequency.getWEEKLY(),
 *                 "includeObjectVersions", s3.InventoryObjectVersion.getALL(),
 *                 "destination", Map.of(
 *                         "bucket", inventoryBucket,
 *                         "prefix", "with-all-versions")))));
 * </pre></blockquote>
 * <p>
 * If the destination bucket is created as part of the same CDK application, the necessary permissions will be automatically added to the bucket policy.
 * However, if you use an imported bucket (i.e <code>Bucket.fromXXX()</code>), you'll have to make sure it contains the following policy document:
 * <p>
 * <blockquote><pre>
 * {
 *   "Version": "2012-10-17",
 *   "Statement": [
 *     {
 *       "Sid": "InventoryAndAnalyticsExamplePolicy",
 *       "Effect": "Allow",
 *       "Principal": { "Service": "s3.amazonaws.com" },
 *       "Action": "s3:PutObject",
 *       "Resource": ["arn:aws:s3:::destinationBucket/*"]
 *     }
 *   ]
 * }
 * </pre></blockquote>
 * <p>
 * <h2>Website redirection</h2>
 * <p>
 * You can use the two following properties to specify the bucket <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/how-to-page-redirect.html#advanced-conditional-redirects">redirection policy</a>. Please note that these methods cannot both be applied to the same bucket.
 * <p>
 * <h3>Static redirection</h3>
 * <p>
 * You can statically redirect a to a given Bucket URL or any other host name with <code>websiteRedirect</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bucket = Bucket.Builder.create(this, "MyRedirectedBucket")
 *         .websiteRedirect(Map.of("hostName", "www.example.com"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Routing rules</h3>
 * <p>
 * Alternatively, you can also define multiple <code>websiteRoutingRules</code>, to define complex, conditional redirections:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bucket = Bucket.Builder.create(this, "MyRedirectedBucket")
 *         .websiteRoutingRules(asList(Map.of(
 *                 "hostName", "www.example.com",
 *                 "httpRedirectCode", "302",
 *                 "protocol", RedirectProtocol.getHTTPS(),
 *                 "replaceKey", ReplaceKey.prefixWith("test/"),
 *                 "condition", Map.of(
 *                         "httpErrorCodeReturnedEquals", "200",
 *                         "keyPrefixEquals", "prefix"))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Filling the bucket as part of deployment</h2>
 * <p>
 * To put files into a bucket as part of a deployment (for example, to host a
 * website), see the <code>&#64;aws-cdk/aws-s3-deployment</code> package, which provides a
 * resource that can do just that.
 * <p>
 * <h2>The URL for objects</h2>
 * <p>
 * S3 provides two types of URLs for accessing objects via HTTP(S). Path-Style and
 * <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/VirtualHosting.html">Virtual Hosted-Style</a>
 * URL. Path-Style is a classic way and will be
 * <a href="https://aws.amazon.com/jp/blogs/aws/amazon-s3-path-deprecation-plan-the-rest-of-the-story">deprecated</a>.
 * We recommend to use Virtual Hosted-Style URL for newly made bucket.
 * <p>
 * You can generate both of them.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * bucket.urlForObject("objectname");// Path-Style URL
 * bucket.virtualHostedUrlForObject("objectname");// Virtual Hosted-Style URL
 * bucket.virtualHostedUrlForObject("objectname", Map.of("regional", false));
 * </pre></blockquote>
 * <p>
 * <h3>Object Ownership</h3>
 * <p>
 * You can use the two following properties to specify the bucket <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/about-object-ownership.html">object Ownership</a>.
 * <p>
 * <h4>Object writer</h4>
 * <p>
 * The Uploading account will own the object.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * new Bucket(this, "MyBucket", new BucketProps()
 *         .objectOwnership(s3.ObjectOwnership.getOBJECT_WRITER()));
 * </pre></blockquote>
 * <p>
 * <h4>Bucket owner preferred</h4>
 * <p>
 * The bucket owner will own the object if the object is uploaded with the bucket-owner-full-control canned ACL. Without this setting and canned ACL, the object is uploaded and remains owned by the uploading account.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * new Bucket(this, "MyBucket", new BucketProps()
 *         .objectOwnership(s3.ObjectOwnership.getBUCKET_OWNER_PREFERRED()));
 * </pre></blockquote>
 * <p>
 * <h3>Bucket deletion</h3>
 * <p>
 * When a bucket is removed from a stack (or the stack is deleted), the S3
 * bucket will be removed according to its removal policy (which by default will
 * simply orphan the bucket and leave it in your AWS account). If the removal
 * policy is set to <code>RemovalPolicy.DESTROY</code>, the bucket will be deleted as long
 * as it does not contain any objects.
 * <p>
 * To override this and force all objects to get deleted during bucket deletion,
 * enable the<code>autoDeleteObjects</code> option.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object bucket = Bucket.Builder.create(this, "MyTempFileBucket")
 *         .removalPolicy(RemovalPolicy.getDESTROY())
 *         .autoDeleteObjects(true)
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.s3;
