/**
 * <h1>AWS Step Functions Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * The <code>&#64;aws-cdk/aws-stepfunctions</code> package contains constructs for building
 * serverless workflows using objects. Use this in conjunction with the
 * <code>&#64;aws-cdk/aws-stepfunctions-tasks</code> package, which contains classes used
 * to call other AWS services.
 * <p>
 * Defining a workflow looks like this (for the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/job-status-poller-sample.html">Step Functions Job Poller
 * example</a>):
 * <p>
 * <h2>Example</h2>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_stepfunctions;
 * import software.amazon.awscdk.aws_stepfunctions_tasks;
 * import software.amazon.awscdk.aws_lambda;
 * 
 * 
 * Function submitLambda = new Function(this, "SubmitLambda", new FunctionProps()...);
 * Function getStatusLambda = new Function(this, "CheckLambda", new FunctionProps()...);
 * 
 * LambdaInvoke submitJob = new LambdaInvoke(this, "Submit Job", new LambdaInvokeProps()
 *         .lambdaFunction(submitLambda)
 *         // Lambda's result is in the attribute `Payload`
 *         .outputPath("$.Payload"));
 * 
 * Wait waitX = new Wait(this, "Wait X Seconds", new WaitProps()
 *         .time(sfn.WaitTime.secondsPath("$.waitSeconds")));
 * 
 * LambdaInvoke getStatus = new LambdaInvoke(this, "Get Job Status", new LambdaInvokeProps()
 *         .lambdaFunction(getStatusLambda)
 *         // Pass just the field named "guid" into the Lambda, put the
 *         // Lambda's result in a field called "status" in the response
 *         .inputPath("$.guid")
 *         .outputPath("$.Payload"));
 * 
 * Fail jobFailed = new Fail(this, "Job Failed", new FailProps()
 *         .cause("AWS Batch Job Failed")
 *         .error("DescribeJob returned FAILED"));
 * 
 * LambdaInvoke finalStatus = new LambdaInvoke(this, "Get Final Job Status", new LambdaInvokeProps()
 *         .lambdaFunction(getStatusLambda)
 *         // Use "guid" field as input
 *         .inputPath("$.guid")
 *         .outputPath("$.Payload"));
 * 
 * Chain definition = submitJob
 *     .next(waitX)
 *     .next(getStatus).next(new sfn.Choice(this, 'Job Complete?')
 *         // Look at the "status" field
 *         .when(sfn.Condition.stringEquals('$.status', 'FAILED'), jobFailed)
 *         .when(sfn.Condition.stringEquals('$.status', 'SUCCEEDED'), finalStatus).otherwise(waitX));
 * 
 * new StateMachine(this, "StateMachine", new StateMachineProps()
 *         .definition(definition)
 *         .timeout(Duration.minutes(5)));
 * </pre></blockquote>
 * <p>
 * You can find more sample snippets and learn more about the service integrations
 * in the <code>&#64;aws-cdk/aws-stepfunctions-tasks</code> package.
 * <p>
 * <h2>State Machine</h2>
 * <p>
 * A <code>stepfunctions.StateMachine</code> is a resource that takes a state machine
 * definition. The definition is specified by its start state, and encompasses
 * all states reachable from the start state:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object startState = new Pass(this, "StartState");
 * 
 * StateMachine.Builder.create(this, "StateMachine")
 *         .definition(startState)
 *         .build();
 * </pre></blockquote>
 * <p>
 * State machines execute using an IAM Role, which will automatically have all
 * permissions added that are required to make all state machine tasks execute
 * properly (for example, permissions to invoke any Lambda functions you add to
 * your workflow). A role will be created by default, but you can supply an
 * existing one as well.
 * <p>
 * <h2>Amazon States Language</h2>
 * <p>
 * This library comes with a set of classes that model the <a href="https://states-language.net/spec.html">Amazon States
 * Language</a>. The following State classes
 * are supported:
 * <p>
 * <ul>
 * <li><a href="#task"><code>Task</code></a></li>
 * <li><a href="#pass"><code>Pass</code></a></li>
 * <li><a href="#wait"><code>Wait</code></a></li>
 * <li><a href="#choice"><code>Choice</code></a></li>
 * <li><a href="#parallel"><code>Parallel</code></a></li>
 * <li><a href="#succeed"><code>Succeed</code></a></li>
 * <li><a href="#fail"><code>Fail</code></a></li>
 * <li><a href="#map"><code>Map</code></a></li>
 * <li><a href="#custom-state"><code>Custom State</code></a></li>
 * </ul>
 * <p>
 * An arbitrary JSON object (specified at execution start) is passed from state to
 * state and transformed during the execution of the workflow. For more
 * information, see the States Language spec.
 * <p>
 * <h3>Task</h3>
 * <p>
 * A <code>Task</code> represents some work that needs to be done. The exact work to be
 * done is determine by a class that implements <code>IStepFunctionsTask</code>, a collection
 * of which can be found in the <code>&#64;aws-cdk/aws-stepfunctions-tasks</code> module.
 * <p>
 * The tasks in the <code>&#64;aws-cdk/aws-stepfunctions-tasks</code> module support the
 * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html">service integration pattern</a> that integrates Step Functions with services
 * directly in the Amazon States language.
 * <p>
 * <h3>Pass</h3>
 * <p>
 * A <code>Pass</code> state passes its input to its output, without performing work.
 * Pass states are useful when constructing and debugging state machines.
 * <p>
 * The following example injects some fixed data into the state machine through
 * the <code>result</code> field. The <code>result</code> field will be added to the input and the result
 * will be passed as the state's output.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Makes the current JSON state { ..., "subObject": { "hello": "world" } }
 * Object pass = Pass.Builder.create(this, "Add Hello World")
 *         .result(sfn.Result.fromObject(Map.of("hello", "world")))
 *         .resultPath("$.subObject")
 *         .build();
 * 
 * // Set the next state
 * pass.next(nextState);
 * </pre></blockquote>
 * <p>
 * The <code>Pass</code> state also supports passing key-value pairs as input. Values can
 * be static, or selected from the input with a path.
 * <p>
 * The following example filters the <code>greeting</code> field from the state input
 * and also injects a field called <code>otherData</code>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object pass = Pass.Builder.create(this, "Filter input and inject data")
 *         .parameters(Map.of(// input to the pass state
 *                 "input", sfn.JsonPath.stringAt("$.input.greeting"),
 *                 "otherData", "some-extra-stuff"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The object specified in <code>parameters</code> will be the input of the <code>Pass</code> state.
 * Since neither <code>Result</code> nor <code>ResultPath</code> are supplied, the <code>Pass</code> state copies
 * its input through to its output.
 * <p>
 * Learn more about the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-pass-state.html">Pass state</a>
 * <p>
 * <h3>Wait</h3>
 * <p>
 * A <code>Wait</code> state waits for a given number of seconds, or until the current time
 * hits a particular time. The time to wait may be taken from the execution's JSON
 * state.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Wait until it's the time mentioned in the the state object's "triggerTime"
 * // field.
 * Object wait = Wait.Builder.create(this, "Wait For Trigger Time")
 *         .time(sfn.WaitTime.timestampPath("$.triggerTime"))
 *         .build();
 * 
 * // Set the next state
 * wait.next(startTheWork);
 * </pre></blockquote>
 * <p>
 * <h3>Choice</h3>
 * <p>
 * A <code>Choice</code> state can take a different path through the workflow based on the
 * values in the execution's JSON state:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object choice = new Choice(this, "Did it work?");
 * 
 * // Add conditions with .when()
 * choice.when(sfn.Condition.stringEquals("$.status", "SUCCESS"), successState);
 * choice.when(sfn.Condition.numberGreaterThan("$.attempts", 5), failureState);
 * 
 * // Use .otherwise() to indicate what should be done if none of the conditions match
 * choice.otherwise(tryAgainState);
 * </pre></blockquote>
 * <p>
 * If you want to temporarily branch your workflow based on a condition, but have
 * all branches come together and continuing as one (similar to how an <code>if ... then ... else</code> works in a programming language), use the <code>.afterwards()</code> method:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object choice = new Choice(this, "What color is it?");
 * choice.when(sfn.Condition.stringEquals("$.color", "BLUE"), handleBlueItem);
 * choice.when(sfn.Condition.stringEquals("$.color", "RED"), handleRedItem);
 * choice.otherwise(handleOtherItemColor);
 * 
 * // Use .afterwards() to join all possible paths back together and continue
 * choice.afterwards().next(shipTheItem);
 * </pre></blockquote>
 * <p>
 * If your <code>Choice</code> doesn't have an <code>otherwise()</code> and none of the conditions match
 * the JSON state, a <code>NoChoiceMatched</code> error will be thrown. Wrap the state machine
 * in a <code>Parallel</code> state if you want to catch and recover from this.
 * <p>
 * <h4>Available Conditions</h4>
 * <p>
 * see <a href="https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-choice-state.html#amazon-states-language-choice-state-rules">step function comparison operators</a>
 * <p>
 * <ul>
 * <li><code>Condition.isPresent</code> - matches if a json path is present</li>
 * <li><code>Condition.isNotPresent</code> - matches if a json path is not present</li>
 * <li><code>Condition.isString</code> - matches if a json path contains a string</li>
 * <li><code>Condition.isNotString</code> - matches if a json path is not a string</li>
 * <li><code>Condition.isNumeric</code> - matches if a json path is numeric</li>
 * <li><code>Condition.isNotNumeric</code> - matches if a json path is not numeric</li>
 * <li><code>Condition.isBoolean</code> - matches if a json path is boolean</li>
 * <li><code>Condition.isNotBoolean</code> - matches if a json path is not boolean</li>
 * <li><code>Condition.isTimestamp</code> - matches if a json path is a timestamp</li>
 * <li><code>Condition.isNotTimestamp</code> - matches if a json path is not a timestamp</li>
 * <li><code>Condition.isNotNull</code> - matches if a json path is not null</li>
 * <li><code>Condition.isNull</code> - matches if a json path is null</li>
 * <li><code>Condition.booleanEquals</code> - matches if a boolean field has a given value</li>
 * <li><code>Condition.booleanEqualsJsonPath</code> - matches if a boolean field equals a value in a given mapping path</li>
 * <li><code>Condition.stringEqualsJsonPath</code> - matches if a string field equals a given mapping path</li>
 * <li><code>Condition.stringEquals</code> - matches if a field equals a string value</li>
 * <li><code>Condition.stringLessThan</code> - matches if a string field sorts before a given value</li>
 * <li><code>Condition.stringLessThanJsonPath</code> - matches if a string field sorts before a value at given mapping path</li>
 * <li><code>Condition.stringLessThanEquals</code> - matches if a string field sorts equal to or before a given value</li>
 * <li><code>Condition.stringLessThanEqualsJsonPath</code> - matches if a string field sorts equal to or before a given mapping</li>
 * <li><code>Condition.stringGreaterThan</code> - matches if a string field sorts after a given value</li>
 * <li><code>Condition.stringGreaterThanJsonPath</code> - matches if a string field sorts after a value at a given mapping path</li>
 * <li><code>Condition.stringGreaterThanEqualsJsonPath</code> - matches if a string field sorts after or equal to value at a given mapping path</li>
 * <li><code>Condition.stringGreaterThanEquals</code> - matches if a string field sorts after or equal to a given value</li>
 * <li><code>Condition.numberEquals</code> - matches if a numeric field has the given value</li>
 * <li><code>Condition.numberEqualsJsonPath</code> - matches if a numeric field has the value in a given mapping path</li>
 * <li><code>Condition.numberLessThan</code> - matches if a numeric field is less than the given value</li>
 * <li><code>Condition.numberLessThanJsonPath</code> - matches if a numeric field is less than the value at the given mapping path</li>
 * <li><code>Condition.numberLessThanEquals</code> - matches if a numeric field is less than or equal to the given value</li>
 * <li><code>Condition.numberLessThanEqualsJsonPath</code> - matches if a numeric field is less than or equal to the numeric value at given mapping path</li>
 * <li><code>Condition.numberGreaterThan</code> - matches if a numeric field is greater than the given value</li>
 * <li><code>Condition.numberGreaterThanJsonPath</code> - matches if a numeric field is greater than the value at a given mapping path</li>
 * <li><code>Condition.numberGreaterThanEquals</code> - matches if a numeric field is greater than or equal to the given value</li>
 * <li><code>Condition.numberGreaterThanEqualsJsonPath</code> - matches if a numeric field is greater than or equal to the value at a given mapping path</li>
 * <li><code>Condition.timestampEquals</code> - matches if a timestamp field is the same time as the given timestamp</li>
 * <li><code>Condition.timestampEqualsJsonPath</code> - matches if a timestamp field is the same time as the timestamp at a given mapping path</li>
 * <li><code>Condition.timestampLessThan</code> - matches if a timestamp field is before the given timestamp</li>
 * <li><code>Condition.timestampLessThanJsonPath</code> - matches if a timestamp field is before the timestamp at a given mapping path</li>
 * <li><code>Condition.timestampLessThanEquals</code> - matches if a timestamp field is before or equal to the given timestamp</li>
 * <li><code>Condition.timestampLessThanEqualsJsonPath</code> - matches if a timestamp field is before or equal to the timestamp at a given mapping path</li>
 * <li><code>Condition.timestampGreaterThan</code> - matches if a timestamp field is after the timestamp at a given mapping path</li>
 * <li><code>Condition.timestampGreaterThanJsonPath</code> - matches if a timestamp field is after the timestamp at a given mapping path</li>
 * <li><code>Condition.timestampGreaterThanEquals</code> - matches if a timestamp field is after or equal to the given timestamp</li>
 * <li><code>Condition.timestampGreaterThanEqualsJsonPath</code> - matches if a timestamp field is after or equal to the timestamp at a given mapping path</li>
 * <li><code>Condition.stringMatches</code> - matches if a field matches a string pattern that can contain a wild card (<em>) e.g: log-</em>.txt or <em>LATEST</em>. No other characters other than "<em>" have any special meaning - * can be escaped: \</em></li>
 * </ul>
 * <p>
 * <h3>Parallel</h3>
 * <p>
 * A <code>Parallel</code> state executes one or more subworkflows in parallel. It can also
 * be used to catch and recover from errors in subworkflows.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object parallel = new Parallel(this, "Do the work in parallel");
 * 
 * // Add branches to be executed in parallel
 * parallel.branch(shipItem);
 * parallel.branch(sendInvoice);
 * parallel.branch(restock);
 * 
 * // Retry the whole workflow if something goes wrong
 * parallel.addRetry(Map.of("maxAttempts", 1));
 * 
 * // How to recover from errors
 * parallel.addCatch(sendFailureNotification);
 * 
 * // What to do in case everything succeeded
 * parallel.next(closeOrder);
 * </pre></blockquote>
 * <p>
 * <h3>Succeed</h3>
 * <p>
 * Reaching a <code>Succeed</code> state terminates the state machine execution with a
 * successful status.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object success = new Succeed(this, "We did it!");
 * </pre></blockquote>
 * <p>
 * <h3>Fail</h3>
 * <p>
 * Reaching a <code>Fail</code> state terminates the state machine execution with a
 * failure status. The fail state should report the reason for the failure.
 * Failures can be caught by encompassing <code>Parallel</code> states.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object success = Fail.Builder.create(this, "Fail")
 *         .error("WorkflowFailure")
 *         .cause("Something went wrong")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Map</h3>
 * <p>
 * A <code>Map</code> state can be used to run a set of steps for each element of an input array.
 * A <code>Map</code> state will execute the same steps for multiple entries of an array in the state input.
 * <p>
 * While the <code>Parallel</code> state executes multiple branches of steps using the same input, a <code>Map</code> state will
 * execute the same steps for multiple entries of an array in the state input.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object map = Map.Builder.create(this, "Map State")
 *         .maxConcurrency(1)
 *         .itemsPath(sfn.JsonPath.stringAt("$.inputForMap"))
 *         .build();
 * map.iterator(new Pass(this, "Pass State"));
 * </pre></blockquote>
 * <p>
 * <h3>Custom State</h3>
 * <p>
 * It's possible that the high-level constructs for the states or <code>stepfunctions-tasks</code> do not have
 * the states or service integrations you are looking for. The primary reasons for this lack of
 * functionality are:
 * <p>
 * <ul>
 * <li>A <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-service-integrations.html">service integration</a> is available through Amazon States Langauge, but not available as construct
 * classes in the CDK.</li>
 * <li>The state or state properties are available through Step Functions, but are not configurable
 * through constructs</li>
 * </ul>
 * <p>
 * If a feature is not available, a <code>CustomState</code> can be used to supply any Amazon States Language
 * JSON-based object as the state definition.
 * <p>
 * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/tutorial-code-snippet.html#tutorial-code-snippet-1">Code Snippets</a> are available and can be plugged in as the state definition.
 * <p>
 * Custom states can be chained together with any of the other states to create your state machine
 * definition. You will also need to provide any permissions that are required to the <code>role</code> that
 * the State Machine uses.
 * <p>
 * The following example uses the <code>DynamoDB</code> service integration to insert data into a DynamoDB table.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_dynamodb;
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.aws_stepfunctions;
 * 
 * 
 * // create a table
 * Table table = new Table(this, "montable", new TableProps()
 *         .partitionKey(new Attribute()
 *                 .name("id")
 *                 .type(ddb.AttributeType.getSTRING())));
 * 
 * Pass finalStatus = new Pass(stack, "final step");
 * 
 * // States language JSON to put an item into DynamoDB
 * // snippet generated from https://docs.aws.amazon.com/step-functions/latest/dg/tutorial-code-snippet.html#tutorial-code-snippet-1
 * __object stateJson = Map.of(
 *         "Type", "Task",
 *         "Resource", "arn:aws:states:::dynamodb:putItem",
 *         "Parameters", Map.of(
 *                 "TableName", table.getTableName(),
 *                 "Item", Map.of(
 *                         "id", Map.of(
 *                                 "S", "MyEntry"))),
 *         "ResultPath", null);
 * 
 * // custom state which represents a task to insert data into DynamoDB
 * CustomState custom = new CustomState(this, "my custom task", new CustomStateProps()
 *         .stateJson(stateJson));
 * 
 * Chain chain = sfn.Chain.start(custom).next(finalStatus);
 * 
 * StateMachine sm = new StateMachine(this, "StateMachine", new StateMachineProps()
 *         .definition(chain)
 *         .timeout(cdk.Duration.seconds(30)));
 * 
 * // don't forget permissions. You need to assign them
 * table.grantWriteData(sm);
 * </pre></blockquote>
 * <p>
 * <h2>Task Chaining</h2>
 * <p>
 * To make defining work flows as convenient (and readable in a top-to-bottom way)
 * as writing regular programs, it is possible to chain most methods invocations.
 * In particular, the <code>.next()</code> method can be repeated. The result of a series of
 * <code>.next()</code> calls is called a <strong>Chain</strong>, and can be used when defining the jump
 * targets of <code>Choice.on</code> or <code>Parallel.branch</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object definition = step1
 *     .next(step2)
 *     .next(choice
 *         .when(condition1, step3.next(step4).next(step5))
 *         .otherwise(step6)
 *         .afterwards())
 *     .next(parallel
 *         .branch(step7.next(step8))
 *         .branch(step9.next(step10))).next(finish);
 * 
 * StateMachine.Builder.create(this, "StateMachine")
 *         .definition(definition)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you don't like the visual look of starting a chain directly off the first
 * step, you can use <code>Chain.start</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object definition = sfn.Chain
 *     .start(step1)
 *     .next(step2).next(step3);
 * </pre></blockquote>
 * <p>
 * <h2>State Machine Fragments</h2>
 * <p>
 * It is possible to define reusable (or abstracted) mini-state machines by
 * defining a construct that implements <code>IChainable</code>, which requires you to define
 * two fields:
 * <p>
 * <ul>
 * <li><code>startState: State</code>, representing the entry point into this state machine.</li>
 * <li><code>endStates: INextable[]</code>, representing the (one or more) states that outgoing
 * transitions will be added to if you chain onto the fragment.</li>
 * </ul>
 * <p>
 * Since states will be named after their construct IDs, you may need to prefix the
 * IDs of states if you plan to instantiate the same state machine fragment
 * multiples times (otherwise all states in every instantiation would have the same
 * name).
 * <p>
 * The class <code>StateMachineFragment</code> contains some helper functions (like
 * <code>prefixStates()</code>) to make it easier for you to do this. If you define your state
 * machine as a subclass of this, it will be convenient to use:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * public class MyJobProps {
 *     private String jobFlavor;
 *     public String getJobFlavor() {
 *         return this.jobFlavor;
 *     }
 *     public MyJobProps jobFlavor(String jobFlavor) {
 *         this.jobFlavor = jobFlavor;
 *         return this;
 *     }
 * }
 * 
 * public class MyJob extends StateMachineFragment {
 *     public final Object startState;
 *     public final Array endStates;
 * 
 *     public MyJob(Construct parent, String id, MyJobProps props) {
 *         super(parent, id);
 * 
 *         Object first = Task.Builder.create(this, "First")....build();
 *         // ...
 *         Object last = Task.Builder.create(this, "Last")....build();
 * 
 *         this.startState = first;
 *         this.endStates = asList(last);
 *     }
 * }
 * 
 * // Do 3 different variants of MyJob in parallel
 * new sfn.Parallel(this, 'All jobs')
 *     .branch(new MyJob(this, 'Quick', { jobFlavor: 'quick' }).prefixStates())
 *     .branch(new MyJob(this, 'Medium', { jobFlavor: 'medium' }).prefixStates()).branch(new MyJob(this, 'Slow', { jobFlavor: 'slow' }).prefixStates());
 * </pre></blockquote>
 * <p>
 * A few utility functions are available to parse state machine fragments.
 * <p>
 * <ul>
 * <li><code>State.findReachableStates</code>: Retrieve the list of states reachable from a given state.</li>
 * <li><code>State.findReachableEndStates</code>: Retrieve the list of end or terminal states reachable from a given state.</li>
 * </ul>
 * <p>
 * <h2>Activity</h2>
 * <p>
 * <strong>Activities</strong> represent work that is done on some non-Lambda worker pool. The
 * Step Functions workflow will submit work to this Activity, and a worker pool
 * that you run yourself, probably on EC2, will pull jobs from the Activity and
 * submit the results of individual jobs back.
 * <p>
 * You need the ARN to do so, so if you use Activities be sure to pass the Activity
 * ARN into your worker pool:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object activity = new Activity(this, "Activity");
 * 
 * // Read this CloudFormation Output from your application and use it to poll for work on
 * // the activity.
 * // Read this CloudFormation Output from your application and use it to poll for work on
 * // the activity.
 * CfnOutput.Builder.create(this, "ActivityArn").value(activity.getActivityArn()).build();
 * </pre></blockquote>
 * <p>
 * <h3>Activity-Level Permissions</h3>
 * <p>
 * Granting IAM permissions to an activity can be achieved by calling the <code>grant(principal, actions)</code> API:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object activity = new Activity(this, "Activity");
 * 
 * Role role = new Role(stack, "Role", new RoleProps()
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com")));
 * 
 * activity.grant(role, "states:SendTaskSuccess");
 * </pre></blockquote>
 * <p>
 * This will grant the IAM principal the specified actions onto the activity.
 * <p>
 * <h2>Metrics</h2>
 * <p>
 * <code>Task</code> object expose various metrics on the execution of that particular task. For example,
 * to create an alarm on a particular task failing:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Alarm.Builder.create(this, "TaskAlarm")
 *         .metric(task.metricFailed())
 *         .threshold(1)
 *         .evaluationPeriods(1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * There are also metrics on the complete state machine:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Alarm.Builder.create(this, "StateMachineAlarm")
 *         .metric(stateMachine.metricFailed())
 *         .threshold(1)
 *         .evaluationPeriods(1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * And there are metrics on the capacity of all state machines in your account:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Alarm.Builder.create(this, "ThrottledAlarm")
 *         .metric(StateTransitionMetrics.metricThrottledEvents())
 *         .threshold(10)
 *         .evaluationPeriods(2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Error names</h2>
 * <p>
 * Step Functions identifies errors in the Amazon States Language using case-sensitive strings, known as error names.
 * The Amazon States Language defines a set of built-in strings that name well-known errors, all beginning with the <code>States.</code> prefix.
 * <p>
 * <ul>
 * <li><code>States.ALL</code> - A wildcard that matches any known error name.</li>
 * <li><code>States.Runtime</code> - An execution failed due to some exception that could not be processed. Often these are caused by errors at runtime, such as attempting to apply InputPath or OutputPath on a null JSON payload. A <code>States.Runtime</code> error is not retriable, and will always cause the execution to fail. A retry or catch on <code>States.ALL</code> will NOT catch States.Runtime errors.</li>
 * <li><code>States.DataLimitExceeded</code> - A States.DataLimitExceeded exception will be thrown for the following:
 * <p>
 * <ul>
 * <li>When the output of a connector is larger than payload size quota.</li>
 * <li>When the output of a state is larger than payload size quota.</li>
 * <li>When, after Parameters processing, the input of a state is larger than the payload size quota.</li>
 * <li>See <a href="https://docs.aws.amazon.com/step-functions/latest/dg/limits-overview.html">the AWS documentation</a> to learn more about AWS Step Functions Quotas.</li>
 * </ul></li>
 * <li><code>States.HeartbeatTimeout</code> - A Task state failed to send a heartbeat for a period longer than the HeartbeatSeconds value.</li>
 * <li><code>States.Timeout</code> - A Task state either ran longer than the TimeoutSeconds value, or failed to send a heartbeat for a period longer than the HeartbeatSeconds value.</li>
 * <li><code>States.TaskFailed</code>- A Task state failed during the execution. When used in a retry or catch, <code>States.TaskFailed</code> acts as a wildcard that matches any known error name except for <code>States.Timeout</code>.</li>
 * </ul>
 * <p>
 * <h2>Logging</h2>
 * <p>
 * Enable logging to CloudWatch by passing a logging configuration with a
 * destination LogGroup:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object logGroup = new LogGroup(stack, "MyLogGroup");
 * 
 * StateMachine.Builder.create(stack, "MyStateMachine")
 *         .definition(sfn.Chain.start(new Pass(stack, "Pass")))
 *         .logs(Map.of(
 *                 "destination", logGroup,
 *                 "level", sfn.LogLevel.getALL()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>X-Ray tracing</h2>
 * <p>
 * Enable X-Ray tracing for StateMachine:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object logGroup = new LogGroup(stack, "MyLogGroup");
 * 
 * StateMachine.Builder.create(stack, "MyStateMachine")
 *         .definition(sfn.Chain.start(new Pass(stack, "Pass")))
 *         .tracingEnabled(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-xray-tracing.html">the AWS documentation</a>
 * to learn more about AWS Step Functions's X-Ray support.
 * <p>
 * <h2>State Machine Permission Grants</h2>
 * <p>
 * IAM roles, users, or groups which need to be able to work with a State Machine should be granted IAM permissions.
 * <p>
 * Any object that implements the <code>IGrantable</code> interface (has an associated principal) can be granted permissions by calling:
 * <p>
 * <ul>
 * <li><code>stateMachine.grantStartExecution(principal)</code> - grants the principal the ability to execute the state machine</li>
 * <li><code>stateMachine.grantRead(principal)</code> - grants the principal read access</li>
 * <li><code>stateMachine.grantTaskResponse(principal)</code> - grants the principal the ability to send task tokens to the state machine</li>
 * <li><code>stateMachine.grantExecution(principal, actions)</code> - grants the principal execution-level permissions for the IAM actions specified</li>
 * <li><code>stateMachine.grant(principal, actions)</code> - grants the principal state-machine-level permissions for the IAM actions specified</li>
 * </ul>
 * <p>
 * <h3>Start Execution Permission</h3>
 * <p>
 * Grant permission to start an execution of a state machine by calling the <code>grantStartExecution()</code> API.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Role role = new Role(stack, "Role", new RoleProps()
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com")));
 * 
 * Object stateMachine = StateMachine.Builder.create(stack, "StateMachine")
 *         .definition(definition)
 *         .build();
 * 
 * // Give role permission to start execution of state machine
 * stateMachine.grantStartExecution(role);
 * </pre></blockquote>
 * <p>
 * The following permission is provided to a service principal by the <code>grantStartExecution()</code> API:
 * <p>
 * <ul>
 * <li><code>states:StartExecution</code> - to state machine</li>
 * </ul>
 * <p>
 * <h3>Read Permissions</h3>
 * <p>
 * Grant <code>read</code> access to a state machine by calling the <code>grantRead()</code> API.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Role role = new Role(stack, "Role", new RoleProps()
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com")));
 * 
 * Object stateMachine = StateMachine.Builder.create(stack, "StateMachine")
 *         .definition(definition)
 *         .build();
 * 
 * // Give role read access to state machine
 * stateMachine.grantRead(role);
 * </pre></blockquote>
 * <p>
 * The following read permissions are provided to a service principal by the <code>grantRead()</code> API:
 * <p>
 * <ul>
 * <li><code>states:ListExecutions</code> - to state machine</li>
 * <li><code>states:ListStateMachines</code> - to state machine</li>
 * <li><code>states:DescribeExecution</code> - to executions</li>
 * <li><code>states:DescribeStateMachineForExecution</code> - to executions</li>
 * <li><code>states:GetExecutionHistory</code> - to executions</li>
 * <li><code>states:ListActivities</code> - to <code>*</code></li>
 * <li><code>states:DescribeStateMachine</code> - to <code>*</code></li>
 * <li><code>states:DescribeActivity</code> - to <code>*</code></li>
 * </ul>
 * <p>
 * <h3>Task Response Permissions</h3>
 * <p>
 * Grant permission to allow task responses to a state machine by calling the <code>grantTaskResponse()</code> API:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Role role = new Role(stack, "Role", new RoleProps()
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com")));
 * 
 * Object stateMachine = StateMachine.Builder.create(stack, "StateMachine")
 *         .definition(definition)
 *         .build();
 * 
 * // Give role task response permissions to the state machine
 * stateMachine.grantTaskResponse(role);
 * </pre></blockquote>
 * <p>
 * The following read permissions are provided to a service principal by the <code>grantRead()</code> API:
 * <p>
 * <ul>
 * <li><code>states:SendTaskSuccess</code> - to state machine</li>
 * <li><code>states:SendTaskFailure</code> - to state machine</li>
 * <li><code>states:SendTaskHeartbeat</code> - to state machine</li>
 * </ul>
 * <p>
 * <h3>Execution-level Permissions</h3>
 * <p>
 * Grant execution-level permissions to a state machine by calling the <code>grantExecution()</code> API:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Role role = new Role(stack, "Role", new RoleProps()
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com")));
 * 
 * Object stateMachine = StateMachine.Builder.create(stack, "StateMachine")
 *         .definition(definition)
 *         .build();
 * 
 * // Give role permission to get execution history of ALL executions for the state machine
 * stateMachine.grantExecution(role, "states:GetExecutionHistory");
 * </pre></blockquote>
 * <p>
 * <h3>Custom Permissions</h3>
 * <p>
 * You can add any set of permissions to a state machine by calling the <code>grant()</code> API.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * User user = new User(stack, "MyUser");
 * 
 * Object stateMachine = StateMachine.Builder.create(stack, "StateMachine")
 *         .definition(definition)
 *         .build();
 * 
 * //give user permission to send task success to the state machine
 * stateMachine.grant(user, "states:SendTaskSuccess");
 * </pre></blockquote>
 * <p>
 * <h2>Import</h2>
 * <p>
 * Any Step Functions state machine that has been created outside the stack can be imported
 * into your CDK stack.
 * <p>
 * State machines can be imported by their ARN via the <code>StateMachine.fromStateMachineArn()</code> API
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import aws.stepfunctions.*;
 * 
 * 
 * Stack stack = new Stack(app, "MyStack");
 * sfn.StateMachine.fromStateMachineArn(stack, "ImportedStateMachine", "arn:aws:states:us-east-1:123456789012:stateMachine:StateMachine2E01A3A5-N5TJppzoevKQ");
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.stepfunctions;
