/**
 * <h1>Include CloudFormation templates in the CDK</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * This module contains a set of classes whose goal is to facilitate working
 * with existing CloudFormation templates in the CDK.
 * It can be thought of as an extension of the capabilities of the
 * <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_core.CfnInclude.html"><code>CfnInclude</code> class</a>.
 * <p>
 * <h2>Basic usage</h2>
 * <p>
 * Assume we have a file with an existing template.
 * It could be in JSON format, in a file <code>my-template.json</code>:
 * <p>
 * <blockquote><pre>
 * {
 *   "Resources": {
 *     "Bucket": {
 *       "Type": "AWS::S3::Bucket",
 *       "Properties": {
 *         "BucketName": "some-bucket-name"
 *       }
 *     }
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * Or it could by in YAML format, in a file <code>my-template.yaml</code>:
 * <p>
 * <blockquote><pre>
 * Resources:
 *   Bucket:
 *     Type: AWS::S3::Bucket
 *     Properties:
 *       BucketName: some-bucket-name
 * </pre></blockquote>
 * <p>
 * It can be included in a CDK application with the following code:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.cloudformation_include;
 * 
 * 
 * CfnInclude cfnTemplate = new CfnInclude(this, "Template", new CfnIncludeProps()
 *         .templateFile("my-template.json"));
 * </pre></blockquote>
 * <p>
 * Or, if your template uses YAML:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object cfnTemplate = CfnInclude.Builder.create(this, "Template")
 *         .templateFile("my-template.yaml")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: different YAML parsers sometimes don't agree on what exactly constitutes valid YAML.
 * If you get a YAML exception when including your template,
 * try converting it to JSON, and including that file instead.
 * If you're downloading your template from the CloudFormation AWS Console,
 * you can easily get it in JSON format by clicking the 'View in Designer'
 * button on the 'Template' tab -
 * once in Designer, select JSON in the "Choose template language"
 * radio buttons on the bottom pane.
 * <p>
 * This will add all resources from <code>my-template.json</code> / <code>my-template.yaml</code> into the CDK application,
 * preserving their original logical IDs from the template file.
 * <p>
 * Note that this including process will <em>not</em> execute any
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-macros.html">CloudFormation transforms</a> -
 * including the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/transform-aws-serverless.html">Serverless transform</a>.
 * <p>
 * Any resource from the included template can be retrieved by referring to it by its logical ID from the template.
 * If you know the class of the CDK object that corresponds to that resource,
 * you can cast the returned object to the correct type:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_s3;
 * 
 * 
 * CfnBucket cfnBucket = (CfnBucket)cfnTemplate.getResource("Bucket");
 * </pre></blockquote>
 * <p>
 * Note that any resources not present in the latest version of the CloudFormation schema
 * at the time of publishing the version of this module that you depend on,
 * including <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cfn-customresource.html">Custom Resources</a>,
 * will be returned as instances of the class <code>CfnResource</code>,
 * and so cannot be cast to a different resource type.
 * <p>
 * Any modifications made to that resource will be reflected in the resulting CDK template;
 * for example, the name of the bucket can be changed:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * cfnBucket.getBucketName() = "my-bucket-name";
 * </pre></blockquote>
 * <p>
 * You can also refer to the resource when defining other constructs,
 * including the higher-level ones
 * (those whose name does not start with <code>Cfn</code>),
 * for example:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_iam;
 * 
 * 
 * Role role = new Role(this, "Role", new RoleProps()
 *         .assumedBy(new AnyPrincipal()));
 * role.addToPolicy(new PolicyStatement(new PolicyStatementProps()
 *         .actions(asList("s3:*"))
 *         .resources(asList(cfnBucket.getAttrArn()))));
 * </pre></blockquote>
 * <p>
 * If you need, you can also convert the CloudFormation resource to a higher-level
 * resource by importing it:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * IBucket bucket = s3.Bucket.fromBucketName(this, "L2Bucket", cfnBucket.getRef());
 * </pre></blockquote>
 * <p>
 * <h2>Non-resource template elements</h2>
 * <p>
 * In addition to resources,
 * you can also retrieve and mutate all other template elements:
 * <p>
 * <ul>
 * <li><a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html">Parameters</a>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.*;
 * 
 * 
 * CfnParameter param = cfnTemplate.getParameter("MyParameter");
 * 
 * // mutating the parameter
 * param.getDefault() = "MyDefault";
 * </pre></blockquote></li>
 * <li><a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/conditions-section-structure.html">Conditions</a>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.*;
 * 
 * 
 * CfnCondition condition = cfnTemplate.getCondition("MyCondition");
 * 
 * // mutating the condition
 * condition.getExpression() = core.Fn.conditionEquals(1, 2);
 * </pre></blockquote></li>
 * <li><a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/mappings-section-structure.html">Mappings</a>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.*;
 * 
 * 
 * CfnMapping mapping = cfnTemplate.getMapping("MyMapping");
 * 
 * // mutating the mapping
 * mapping.setValue("my-region", "AMI", "ami-04681a1dbd79675a5");
 * </pre></blockquote></li>
 * <li><a href="https://docs.aws.amazon.com/servicecatalog/latest/adminguide/reference-template_constraint_rules.html">Service Catalog template Rules</a>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.*;
 * 
 * 
 * CfnRule rule = cfnTemplate.getRule("MyRule");
 * 
 * // mutating the rule
 * rule.addAssertion(core.Fn.conditionContains(asList("m1.small"), myParameter.getValue()), "MyParameter has to be m1.small");
 * </pre></blockquote></li>
 * <li><a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html">Outputs</a>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.*;
 * 
 * 
 * CfnOutput output = cfnTemplate.getOutput("MyOutput");
 * 
 * // mutating the output
 * output.getValue() = cfnBucket.getAttrArn();
 * </pre></blockquote></li>
 * <li><a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/blue-green.html">Hooks for blue-green deployments</a>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.*;
 * 
 * 
 * CfnHook hook = cfnTemplate.getHook("MyOutput");
 * 
 * // mutating the hook
 * CfnCodeDeployBlueGreenHook codeDeployHook = (CfnCodeDeployBlueGreenHook)hook;
 * codeDeployHook.getServiceRole() = myRole.getRoleArn();
 * </pre></blockquote></li>
 * </ul>
 * <p>
 * <h2>Parameter replacement</h2>
 * <p>
 * If your existing template uses CloudFormation Parameters,
 * you may want to remove them in favor of build-time values.
 * You can do that using the <code>parameters</code> property:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * CfnInclude.Builder.create(this, "includeTemplate")
 *         .templateFile("path/to/my/template")
 *         .parameters(Map.of(
 *                 "MyParam", "my-value"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * This will replace all references to <code>MyParam</code> with the string <code>'my-value'</code>,
 * and <code>MyParam</code> will be removed from the 'Parameters' section of the template.
 * <p>
 * <h2>Nested Stacks</h2>
 * <p>
 * This module also supports templates that use <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">nested stacks</a>.
 * <p>
 * For example, if you have the following parent template:
 * <p>
 * <blockquote><pre>
 * {
 *   "Resources": {
 *     "ChildStack": {
 *       "Type": "AWS::CloudFormation::Stack",
 *       "Properties": {
 *         "TemplateURL": "https://my-s3-template-source.s3.amazonaws.com/child-stack.json"
 *       }
 *     }
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * where the child template pointed to by <code>https://my-s3-template-source.s3.amazonaws.com/child-stack.json</code> is:
 * <p>
 * <blockquote><pre>
 * {
 *   "Resources": {
 *     "MyBucket": {
 *       "Type": "AWS::S3::Bucket"
 *     }
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * You can include both the parent stack,
 * and the nested stack in your CDK application as follows:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object parentTemplate = CfnInclude.Builder.create(this, "ParentStack")
 *         .templateFile("path/to/my-parent-template.json")
 *         .loadNestedStacks(Map.of(
 *                 "ChildStack", Map.of(
 *                         "templateFile", "path/to/my-nested-template.json")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Here, <code>path/to/my-nested-template.json</code>
 * represents the path on disk to the downloaded template file from the original template URL of the nested stack
 * (<code>https://my-s3-template-source.s3.amazonaws.com/child-stack.json</code>).
 * In the CDK application,
 * this file will be turned into an <a href="https://docs.aws.amazon.com/cdk/latest/guide/assets.html">Asset</a>,
 * and the <code>TemplateURL</code> property of the nested stack resource
 * will be modified to point to that asset.
 * <p>
 * The included nested stack can be accessed with the <code>getNestedStack</code> method:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object includedChildStack = parentTemplate.getNestedStack("ChildStack");
 * Object childStack = includedChildStack.getStack();
 * Object childTemplate = includedChildStack.getIncludedTemplate();
 * </pre></blockquote>
 * <p>
 * Now you can reference resources from <code>ChildStack</code>,
 * and modify them like any other included template:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * CfnBucket cfnBucket = (CfnBucket)childTemplate.getResource("MyBucket");
 * cfnBucket.getBucketName() = "my-new-bucket-name";
 * 
 * Role role = new Role(childStack, "MyRole", new RoleProps()
 *         .assumedBy(new AccountRootPrincipal()));
 * 
 * role.addToPolicy(new PolicyStatement(new PolicyStatementProps()
 *         .actions(asList("s3:GetObject*", "s3:GetBucket*", "s3:List*"))
 *         .resources(asList(cfnBucket.getAttrArn()))));
 * </pre></blockquote>
 * <p>
 * You can also include the nested stack after the <code>CfnInclude</code> object was created,
 * instead of doing it on construction:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object includedChildStack = parentTemplate.loadNestedStack("ChildTemplate", Map.of(
 *         "templateFile", "path/to/my-nested-template.json"));
 * </pre></blockquote>
 * <p>
 * <h2>Vending CloudFormation templates as Constructs</h2>
 * <p>
 * In many cases, there are existing CloudFormation templates that are not entire applications,
 * but more like specialized fragments, implementing a particular pattern or best practice.
 * If you have templates like that,
 * you can use the <code>CfnInclude</code> class to vend them as CDK Constructs:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import path.*;
 * 
 * 
 * public class MyConstruct extends Construct {
 *     public MyConstruct(Construct scope, String id) {
 *         super(scope, id);
 * 
 *         // include a template inside the Construct
 *         // include a template inside the Construct
 *         CfnInclude.Builder.create(this, "MyConstruct")
 *                 .templateFile(path.join(__dirname, "my-template.json"))
 *                 .preserveLogicalIds(false)
 *                 .build();
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * Notice the <code>preserveLogicalIds</code> parameter -
 * it makes sure the logical IDs of all the included template elements are re-named using CDK's algorithm,
 * guaranteeing they are unique within your application.
 * Without that parameter passed,
 * instantiating <code>MyConstruct</code> twice in the same Stack would result in duplicated logical IDs.
 */
package software.amazon.awscdk.cloudformation.include;
