/**
 * <h1>AWS Config Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * Features                                                                               | Stability
 * ---------------------------------------------------------------------------------------|------------
 * CFN Resources                                                                          | <img alt="Stable" src="https://img.shields.io/badge/stable-success.svg?style=for-the-badge">
 * Higher level constructs for Config Rules                                               | <img alt="Stable" src="https://img.shields.io/badge/stable-success.svg?style=for-the-badge">
 * Higher level constructs for initial set-up (delivery channel &amp; configuration recorder) | <img alt="Not Implemented" src="https://img.shields.io/badge/not--implemented-black.svg?style=for-the-badge">
 * <p>
 * <blockquote>
 * <p>
 * <strong>CFN Resources:</strong> All classes with the <code>Cfn</code> prefix in this module (<a href="https://docs.aws.amazon.com/cdk/latest/guide/constructs.html#constructs_lib">CFN Resources</a>) are always
 * stable and safe to use.
 * <p>
 * </blockquote><!-- --><blockquote>
 * <p>
 * <strong>Stable:</strong> Higher level constructs in this module that are marked stable will not undergo any
 * breaking changes. They will strictly follow the <a href="https://semver.org/">Semantic Versioning</a> model.
 * <p>
 * </blockquote>
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * <a href="https://docs.aws.amazon.com/config/latest/developerguide/WhatIsConfig.html">AWS Config</a> provides a detailed view of the configuration of AWS resources in your AWS account.
 * This includes how the resources are related to one another and how they were configured in the
 * past so that you can see how the configurations and relationships change over time.
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <h2>Initial Setup</h2>
 * <p>
 * Before using the constructs provided in this module, you need to set up AWS Config
 * in the region in which it will be used. This setup includes the one-time creation of the
 * following resources per region:
 * <p>
 * <ul>
 * <li><code>ConfigurationRecorder</code>: Configure which resources will be recorded for config changes.</li>
 * <li><code>DeliveryChannel</code>: Configure where to store the recorded data.</li>
 * </ul>
 * <p>
 * The following guides provide the steps for getting started with AWS Config:
 * <p>
 * <ul>
 * <li><a href="https://docs.aws.amazon.com/config/latest/developerguide/gs-console.html">Using the AWS Console</a></li>
 * <li><a href="https://docs.aws.amazon.com/config/latest/developerguide/gs-cli.html">Using the AWS CLI</a></li>
 * </ul>
 * <p>
 * <h2>Rules</h2>
 * <p>
 * AWS Config can evaluate the configuration settings of your AWS resources by creating AWS Config rules,
 * which represent your ideal configuration settings.
 * <p>
 * See <a href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config.html">Evaluating Resources with AWS Config Rules</a> to learn more about AWS Config rules.
 * <p>
 * <h3>AWS Managed Rules</h3>
 * <p>
 * AWS Config provides AWS managed rules, which are predefined, customizable rules that AWS Config
 * uses to evaluate whether your AWS resources comply with common best practices.
 * <p>
 * For example, you could create a managed rule that checks whether active access keys are rotated
 * within the number of days specified.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_config;
 * import software.amazon.awscdk.*;
 * 
 * 
 * // https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html
 * // https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html
 * new ManagedRule(this, "AccessKeysRotated", new ManagedRuleProps()
 *         .identifier(config.ManagedRuleIdentifiers.getACCESS_KEYS_ROTATED())
 *         .inputParameters(Map.of(
 *                 "maxAccessKeyAge", 60))
 *         .maximumExecutionFrequency(config.MaximumExecutionFrequency.getTWELVE_HOURS()));
 * </pre></blockquote>
 * <p>
 * Identifiers for AWS managed rules are available through static constants in the <code>ManagedRuleIdentifiers</code> class.
 * You can find supported input parameters in the <a href="https://docs.aws.amazon.com/config/latest/developerguide/managed-rules-by-aws-config.html">List of AWS Config Managed Rules</a>.
 * <p>
 * The following higher level constructs for AWS managed rules are available.
 * <p>
 * <h4>Access Key rotation</h4>
 * <p>
 * Checks whether your active access keys are rotated within the number of days specified.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_config;
 * import software.amazon.awscdk.aws_cdk;
 * 
 * 
 * // compliant if access keys have been rotated within the last 90 days
 * // compliant if access keys have been rotated within the last 90 days
 * new AccessKeysRotated(this, "AccessKeyRotated");
 * </pre></blockquote>
 * <p>
 * <h4>CloudFormation Stack drift detection</h4>
 * <p>
 * Checks whether your CloudFormation stack's actual configuration differs, or has drifted,
 * from it's expected configuration.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_config;
 * import software.amazon.awscdk.aws_cdk;
 * 
 * 
 * // compliant if stack's status is 'IN_SYNC'
 * // non-compliant if the stack's drift status is 'DRIFTED'
 * // compliant if stack's status is 'IN_SYNC'
 * // non-compliant if the stack's drift status is 'DRIFTED'
 * new CloudFormationStackDriftDetectionCheck(stack, "Drift", new CloudFormationStackDriftDetectionCheckProps()
 *         .ownStackOnly(true));
 * </pre></blockquote>
 * <p>
 * <h4>CloudFormation Stack notifications</h4>
 * <p>
 * Checks whether your CloudFormation stacks are sending event notifications to a SNS topic.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_config;
 * import software.amazon.awscdk.aws_cdk;
 * 
 * 
 * // topics to which CloudFormation stacks may send event notifications
 * Topic topic1 = new Topic(stack, "AllowedTopic1");
 * Topic topic2 = new Topic(stack, "AllowedTopic2");
 * 
 * // non-compliant if CloudFormation stack does not send notifications to 'topic1' or 'topic2'
 * // non-compliant if CloudFormation stack does not send notifications to 'topic1' or 'topic2'
 * new CloudFormationStackNotificationCheck(this, "NotificationCheck", new CloudFormationStackNotificationCheckProps()
 *         .topics(asList(topic1, topic2)));
 * </pre></blockquote>
 * <p>
 * <h3>Custom rules</h3>
 * <p>
 * You can develop custom rules and add them to AWS Config. You associate each custom rule with an
 * AWS Lambda function, which contains the logic that evaluates whether your AWS resources comply
 * with the rule.
 * <p>
 * <h3>Triggers</h3>
 * <p>
 * AWS Lambda executes functions in response to events that are published by AWS Services.
 * The function for a custom Config rule receives an event that is published by AWS Config,
 * and is responsible for evaluating the compliance of the rule.
 * <p>
 * Evaluations can be triggered by configuration changes, periodically, or both.
 * To create a custom rule, define a <code>CustomRule</code> and specify the Lambda Function
 * to run and the trigger types.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_config;
 * 
 * 
 * new CustomRule(this, "CustomRule", new CustomRuleProps()
 *         .lambdaFunction(evalComplianceFn)
 *         .configurationChanges(true)
 *         .periodic(true)
 *         .maximumExecutionFrequency(config.MaximumExecutionFrequency.getSIX_HOURS()));
 * </pre></blockquote>
 * <p>
 * When the trigger for a rule occurs, the Lambda function is invoked by publishing an event.
 * See <a href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_develop-rules_example-events.html">example events for AWS Config Rules</a>
 * <p>
 * The AWS documentation has examples of Lambda functions for evaluations that are
 * <a href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_develop-rules_nodejs-sample.html#event-based-example-rule">triggered by configuration changes</a> and <a href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_develop-rules_nodejs-sample.html#periodic-example-rule">triggered periodically</a>
 * <p>
 * <h3>Scope</h3>
 * <p>
 * By default rules are triggered by changes to all <a href="https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources">resources</a>.
 * <p>
 * Use the <code>RuleScope</code> APIs (<code>fromResource()</code>, <code>fromResources()</code> or <code>fromTag()</code>) to restrict
 * the scope of both managed and custom rules:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_config;
 * 
 * 
 * ManagedRule sshRule = new ManagedRule(this, "SSH", new ManagedRuleProps()
 *         .identifier(config.ManagedRuleIdentifiers.getEC2_SECURITY_GROUPS_INCOMING_SSH_DISABLED())
 *         .ruleScope(config.RuleScope.fromResource(config.ResourceType.getEC2_SECURITY_GROUP(), "sg-1234567890abcdefgh")));
 * 
 * CustomRule customRule = new CustomRule(this, "Lambda", new CustomRuleProps()
 *         .lambdaFunction(evalComplianceFn)
 *         .configurationChanges(true)
 *         .ruleScope(config.RuleScope.fromResources(asList(config.ResourceType.getCLOUDFORMATION_STACK(), config.ResourceType.getS3_BUCKET()))));
 * 
 * CustomRule tagRule = new CustomRule(this, "CostCenterTagRule", new CustomRuleProps()
 *         .lambdaFunction(evalComplianceFn)
 *         .configurationChanges(true)
 *         .ruleScope(config.RuleScope.fromTag("Cost Center", "MyApp")));
 * </pre></blockquote>
 * <p>
 * <h3>Events</h3>
 * <p>
 * You can define Amazon EventBridge event rules which trigger when a compliance check fails
 * or when a rule is re-evaluated.
 * <p>
 * Use the <code>onComplianceChange()</code> APIs to trigger an EventBridge event when a compliance check
 * of your AWS Config Rule fails:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_config;
 * import software.amazon.awscdk.aws_sns;
 * import software.amazon.awscdk.aws_events_targets;
 * 
 * 
 * // Topic to which compliance notification events will be published
 * Topic complianceTopic = new Topic(this, "ComplianceTopic");
 * 
 * CloudFormationStackDriftDetectionCheck rule = new CloudFormationStackDriftDetectionCheck(this, "Drift");
 * rule.onComplianceChange("TopicEvent", new OnEventOptions()
 *         .target(new SnsTopic(complianceTopic)));
 * </pre></blockquote>
 * <p>
 * Use the <code>onReEvaluationStatus()</code> status to trigger an EventBridge event when an AWS Config
 * rule is re-evaluated.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_config;
 * import software.amazon.awscdk.aws_sns;
 * import software.amazon.awscdk.aws_events_targets;
 * 
 * 
 * // Topic to which re-evaluation notification events will be published
 * Topic reEvaluationTopic = new Topic(this, "ComplianceTopic");
 * rule.onReEvaluationStatus("ReEvaluationEvent", Map.of(
 *         "target", new SnsTopic(reEvaluationTopic)));
 * </pre></blockquote>
 * <p>
 * <h3>Example</h3>
 * <p>
 * The following example creates a custom rule that evaluates whether EC2 instances are compliant.
 * Compliance events are published to an SNS topic.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_config;
 * import software.amazon.awscdk.aws_lambda;
 * import software.amazon.awscdk.aws_sns;
 * import software.amazon.awscdk.aws_events_targets;
 * 
 * 
 * // Lambda function containing logic that evaluates compliance with the rule.
 * Function evalComplianceFn = new Function(this, "CustomFunction", new FunctionProps()
 *         .code(lambda.AssetCode.fromInline("exports.handler = (event) =&gt; console.log(event);"))
 *         .handler("index.handler")
 *         .runtime(lambda.Runtime.getNODEJS_12_X()));
 * 
 * // A custom rule that runs on configuration changes of EC2 instances
 * CustomRule customRule = new CustomRule(this, "Custom", new CustomRuleProps()
 *         .configurationChanges(true)
 *         .lambdaFunction(evalComplianceFn)
 *         .ruleScope(config.RuleScope.fromResource(asList(config.ResourceType.getEC2_INSTANCE()))));
 * 
 * // A rule to detect stack drifts
 * CloudFormationStackDriftDetectionCheck driftRule = new CloudFormationStackDriftDetectionCheck(this, "Drift");
 * 
 * // Topic to which compliance notification events will be published
 * Topic complianceTopic = new Topic(this, "ComplianceTopic");
 * 
 * // Send notification on compliance change events
 * driftRule.onComplianceChange("ComplianceChange", new OnEventOptions()
 *         .target(new SnsTopic(complianceTopic)));
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.config;
