/**
 * <h1>Amazon ECS Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * This package contains constructs for working with <strong>Amazon Elastic Container
 * Service</strong> (Amazon ECS).
 * <p>
 * Amazon Elastic Container Service (Amazon ECS) is a fully managed container orchestration service.
 * <p>
 * For further information on Amazon ECS,
 * see the <a href="https://docs.aws.amazon.com/ecs">Amazon ECS documentation</a>
 * <p>
 * The following example creates an Amazon ECS cluster, adds capacity to it, and
 * runs a service on it:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_ecs;
 * 
 * 
 * // Create an ECS cluster
 * Cluster cluster = new Cluster(this, "Cluster", new ClusterProps()
 *         .vpc(vpc));
 * 
 * // Add capacity to it
 * cluster.addCapacity("DefaultAutoScalingGroupCapacity", new AddCapacityOptions()
 *         .instanceType(new InstanceType("t2.xlarge"))
 *         .desiredCapacity(3));
 * 
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * 
 * taskDefinition.addContainer("DefaultContainer", new ContainerDefinitionOptions()
 *         .image(ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryLimitMiB(512));
 * 
 * // Instantiate an Amazon ECS Service
 * Ec2Service ecsService = new Ec2Service(this, "Service", new Ec2ServiceProps()
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition));
 * </pre></blockquote>
 * <p>
 * For a set of constructs defining common ECS architectural patterns, see the <code>&#64;aws-cdk/aws-ecs-patterns</code> package.
 * <p>
 * <h2>Launch Types: AWS Fargate vs Amazon EC2</h2>
 * <p>
 * There are two sets of constructs in this library; one to run tasks on Amazon EC2 and
 * one to run tasks on AWS Fargate.
 * <p>
 * <ul>
 * <li>Use the <code>Ec2TaskDefinition</code> and <code>Ec2Service</code> constructs to run tasks on Amazon EC2 instances running in your account.</li>
 * <li>Use the <code>FargateTaskDefinition</code> and <code>FargateService</code> constructs to run tasks on
 * instances that are managed for you by AWS.</li>
 * </ul>
 * <p>
 * Here are the main differences:
 * <p>
 * <ul>
 * <li><strong>Amazon EC2</strong>: instances are under your control. Complete control of task to host
 * allocation. Required to specify at least a memory reservation or limit for
 * every container. Can use Host, Bridge and AwsVpc networking modes. Can attach
 * Classic Load Balancer. Can share volumes between container and host.</li>
 * <li><strong>AWS Fargate</strong>: tasks run on AWS-managed instances, AWS manages task to host
 * allocation for you. Requires specification of memory and cpu sizes at the
 * taskdefinition level. Only supports AwsVpc networking modes and
 * Application/Network Load Balancers. Only the AWS log driver is supported.
 * Many host features are not supported such as adding kernel capabilities
 * and mounting host devices/volumes inside the container.</li>
 * </ul>
 * <p>
 * For more information on Amazon EC2 vs AWS Fargate and networking see the AWS Documentation:
 * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS_Fargate.html">AWS Fargate</a> and
 * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html">Task Networking</a>.
 * <p>
 * <h2>Clusters</h2>
 * <p>
 * A <code>Cluster</code> defines the infrastructure to run your
 * tasks on. You can run many tasks on a single cluster.
 * <p>
 * The following code creates a cluster that can run AWS Fargate tasks:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object cluster = Cluster.Builder.create(this, "Cluster")
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To use tasks with Amazon EC2 launch-type, you have to add capacity to
 * the cluster in order for tasks to be scheduled on your instances.  Typically,
 * you add an AutoScalingGroup with instances running the latest
 * Amazon ECS-optimized AMI to the cluster. There is a method to build and add such an
 * AutoScalingGroup automatically, or you can supply a customized AutoScalingGroup
 * that you construct yourself. It's possible to add multiple AutoScalingGroups
 * with various instance types.
 * <p>
 * The following example creates an Amazon ECS cluster and adds capacity to it:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object cluster = Cluster.Builder.create(this, "Cluster")
 *         .vpc(vpc)
 *         .build();
 * 
 * // Either add default capacity
 * cluster.addCapacity("DefaultAutoScalingGroupCapacity", Map.of(
 *         "instanceType", new InstanceType("t2.xlarge"),
 *         "desiredCapacity", 3));
 * 
 * // Or add customized capacity. Be sure to start the Amazon ECS-optimized AMI.
 * Object autoScalingGroup = AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .instanceType(new InstanceType("t2.xlarge"))
 *         .machineImage(EcsOptimizedImage.amazonLinux())
 *         // Or use Amazon ECS-Optimized Amazon Linux 2 AMI
 *         // machineImage: EcsOptimizedImage.amazonLinux2(),
 *         .desiredCapacity(3)
 *         .build();
 * 
 * cluster.addAutoScalingGroup(autoScalingGroup);
 * </pre></blockquote>
 * <p>
 * If you omit the property <code>vpc</code>, the construct will create a new VPC with two AZs.
 * <p>
 * <h3>Bottlerocket</h3>
 * <p>
 * <a href="https://aws.amazon.com/bottlerocket/">Bottlerocket</a> is a Linux-based open source operating system that is
 * purpose-built by AWS for running containers. You can launch Amazon ECS container instances with the Bottlerocket AMI.
 * <p>
 * <blockquote>
 * <p>
 * <strong>NOTICE</strong>: The Bottlerocket AMI is in developer preview release for Amazon ECS and is subject to change.
 * <p>
 * </blockquote>
 * <p>
 * The following example will create a capacity with self-managed Amazon EC2 capacity of 2 <code>c5.large</code> Linux instances running with <code>Bottlerocket</code> AMI.
 * <p>
 * Note that you must specify either a <code>machineImage</code> or <code>machineImageType</code>, at least one, not both.
 * <p>
 * The following example adds Bottlerocket capacity to the cluster:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * cluster.addCapacity("bottlerocket-asg", Map.of(
 *         "minCapacity", 2,
 *         "instanceType", new InstanceType("c5.large"),
 *         "machineImageType", ecs.MachineImageType.getBOTTLEROCKET()));
 * </pre></blockquote>
 * <p>
 * <h3>ARM64 (Graviton) Instances</h3>
 * <p>
 * To launch instances with ARM64 hardware, you can use the Amazon ECS-optimized
 * Amazon Linux 2 (arm64) AMI. Based on Amazon Linux 2, this AMI is recommended
 * for use when launching your EC2 instances that are powered by Arm-based AWS
 * Graviton Processors.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * cluster.addCapacity("graviton-cluster", Map.of(
 *         "minCapacity", 2,
 *         "instanceType", new InstanceType("c6g.large"),
 *         "machineImage", ecs.EcsOptimizedImage.amazonLinux2(ecs.AmiHardwareType.getARM())));
 * </pre></blockquote>
 * <p>
 * <h3>Spot Instances</h3>
 * <p>
 * To add spot instances into the cluster, you must specify the <code>spotPrice</code> in the <code>ecs.AddCapacityOptions</code> and optionally enable the <code>spotInstanceDraining</code> property.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Add an AutoScalingGroup with spot instances to the existing cluster
 * cluster.addCapacity("AsgSpot", Map.of(
 *         "maxCapacity", 2,
 *         "minCapacity", 2,
 *         "desiredCapacity", 2,
 *         "instanceType", new InstanceType("c5.xlarge"),
 *         "spotPrice", "0.0735",
 *         // Enable the Automated Spot Draining support for Amazon ECS
 *         "spotInstanceDraining", true));
 * </pre></blockquote>
 * <p>
 * <h3>SNS Topic Encryption</h3>
 * <p>
 * When the <code>ecs.AddCapacityOptions</code> that you provide has a non-zero <code>taskDrainTime</code> (the default) then an SNS topic and Lambda are created to ensure that the
 * cluster's instances have been properly drained of tasks before terminating. The SNS Topic is sent the instance-terminating lifecycle event from the AutoScalingGroup,
 * and the Lambda acts on that event. If you wish to engage <a href="https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html">server-side encryption</a> for this SNS Topic
 * then you may do so by providing a KMS key for the <code>topicEncryptionKey</code> property of <code>ecs.AddCapacityOptions</code>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Given
 * Object key = kms.Key(...);
 * // Then, use that key to encrypt the lifecycle-event SNS Topic.
 * cluster.addCapacity("ASGEncryptedSNS", Map.of(
 *         "instanceType", new InstanceType("t2.xlarge"),
 *         "desiredCapacity", 3,
 *         "topicEncryptionKey", key));
 * </pre></blockquote>
 * <p>
 * <h2>Task definitions</h2>
 * <p>
 * A task definition describes what a single copy of a <strong>task</strong> should look like.
 * A task definition has one or more containers; typically, it has one
 * main container (the <em>default container</em> is the first one that's added
 * to the task definition, and it is marked <em>essential</em>) and optionally
 * some supporting containers which are used to support the main container,
 * doings things like upload logs or metrics to monitoring services.
 * <p>
 * To run a task or service with Amazon EC2 launch type, use the <code>Ec2TaskDefinition</code>. For AWS Fargate tasks/services, use the
 * <code>FargateTaskDefinition</code>. These classes provide a simplified API that only contain
 * properties relevant for that specific launch type.
 * <p>
 * For a <code>FargateTaskDefinition</code>, specify the task size (<code>memoryLimitMiB</code> and <code>cpu</code>):
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object fargateTaskDefinition = FargateTaskDefinition.Builder.create(this, "TaskDef")
 *         .memoryLimitMiB(512)
 *         .cpu(256)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To add containers to a task definition, call <code>addContainer()</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object container = fargateTaskDefinition.addContainer("WebContainer", Map.of(
 *         // Use an image from DockerHub
 *         "image", ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample")));
 * </pre></blockquote>
 * <p>
 * For a <code>Ec2TaskDefinition</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object ec2TaskDefinition = Ec2TaskDefinition.Builder.create(this, "TaskDef")
 *         .networkMode(NetworkMode.getBRIDGE())
 *         .build();
 * 
 * Object container = ec2TaskDefinition.addContainer("WebContainer", Map.of(
 *         // Use an image from DockerHub
 *         "image", ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
 *         "memoryLimitMiB", 1024));
 * </pre></blockquote>
 * <p>
 * You can specify container properties when you add them to the task definition, or with various methods, e.g.:
 * <p>
 * To add a port mapping when adding a container to the task definition, specify the <code>portMappings</code> option:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * taskDefinition.addContainer("WebContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
 *         "memoryLimitMiB", 1024,
 *         "portMappings", asList(Map.of("containerPort", 3000))));
 * </pre></blockquote>
 * <p>
 * To add port mappings directly to a container definition, call <code>addPortMappings()</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * container.addPortMappings(Map.of(
 *         "containerPort", 3000));
 * </pre></blockquote>
 * <p>
 * To add data volumes to a task definition, call <code>addVolume()</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * __object volume = Map.of(
 *         // Use an Elastic FileSystem
 *         "name", "mydatavolume",
 *         "efsVolumeConfiguration", ecs.EfsVolumeConfiguration(Map.of(
 *                 "fileSystemId", "EFS")));
 * 
 * Object container = fargateTaskDefinition.addVolume("mydatavolume");
 * </pre></blockquote>
 * <p>
 * To use a TaskDefinition that can be used with either Amazon EC2 or
 * AWS Fargate launch types, use the <code>TaskDefinition</code> construct.
 * <p>
 * When creating a task definition you have to specify what kind of
 * tasks you intend to run: Amazon EC2, AWS Fargate, or both.
 * The following example uses both:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object taskDefinition = TaskDefinition.Builder.create(this, "TaskDef")
 *         .memoryMiB("512")
 *         .cpu("256")
 *         .networkMode(NetworkMode.getAWS_VPC())
 *         .compatibility(ecs.Compatibility.getEC2_AND_FARGATE())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Images</h3>
 * <p>
 * Images supply the software that runs inside the container. Images can be
 * obtained from either DockerHub or from ECR repositories, or built directly from a local Dockerfile.
 * <p>
 * <ul>
 * <li><code>ecs.ContainerImage.fromRegistry(imageName)</code>: use a public image.</li>
 * <li><code>ecs.ContainerImage.fromRegistry(imageName, { credentials: mySecret })</code>: use a private image that requires credentials.</li>
 * <li><code>ecs.ContainerImage.fromEcrRepository(repo, tag)</code>: use the given ECR repository as the image
 * to start. If no tag is provided, "latest" is assumed.</li>
 * <li><code>ecs.ContainerImage.fromAsset('./image')</code>: build and upload an
 * image directly from a <code>Dockerfile</code> in your source directory.</li>
 * <li><code>ecs.ContainerImage.fromDockerImageAsset(asset)</code>: uses an existing
 * <code>&#64;aws-cdk/aws-ecr-assets.DockerImageAsset</code> as a container image.</li>
 * <li><code>new ecs.TagParameterContainerImage(repository)</code>: use the given ECR repository as the image
 * but a CloudFormation parameter as the tag.</li>
 * </ul>
 * <p>
 * <h3>Environment variables</h3>
 * <p>
 * To pass environment variables to the container, you can use the <code>environment</code>, <code>environmentFiles</code>, and <code>secrets</code> props.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * taskDefinition.addContainer("container", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
 *         "memoryLimitMiB", 1024,
 *         "environment", Map.of(// clear text, not for sensitive data
 *                 "STAGE", "prod"),
 *         "environmentFiles", asList(ecs.EnvironmentFile.fromAsset("./demo-env-file.env"), ecs.EnvironmentFile.fromBucket(s3Bucket, "assets/demo-env-file.env")),
 *         "secrets", Map.of(// Retrieved from AWS Secrets Manager or AWS Systems Manager Parameter Store at container start-up.
 *                 "SECRET", ecs.Secret.fromSecretsManager(secret),
 *                 "DB_PASSWORD", ecs.Secret.fromSecretsManager(dbSecret, "password"), // Reference a specific JSON field, (requires platform version 1.4.0 or later for Fargate tasks)
 *                 "PARAMETER", ecs.Secret.fromSsmParameter(parameter))));
 * </pre></blockquote>
 * <p>
 * The task execution role is automatically granted read permissions on the secrets/parameters. Support for environment
 * files is restricted to the EC2 launch type for files hosted on S3. Further details provided in the AWS documentation
 * about <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html">specifying environment variables</a>.
 * <p>
 * <h2>Service</h2>
 * <p>
 * A <code>Service</code> instantiates a <code>TaskDefinition</code> on a <code>Cluster</code> a given number of
 * times, optionally associating them with a load balancer.
 * If a task fails,
 * Amazon ECS automatically restarts the task.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object taskDefinition;
 * 
 * Object service = FargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .desiredCount(5)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <code>Services</code> by default will create a security group if not provided.
 * If you'd like to specify which security groups to use you can override the <code>securityGroups</code> property.
 * <p>
 * <h3>Deployment circuit breaker and rollback</h3>
 * <p>
 * Amazon ECS <a href="https://aws.amazon.com/tw/blogs/containers/announcing-amazon-ecs-deployment-circuit-breaker/">deployment circuit breaker</a>
 * automatically rolls back unhealthy service deployments without the need for manual intervention. Use <code>circuitBreaker</code> to enable
 * deployment circuit breaker and optionally enable <code>rollback</code> for automatic rollback. See <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html">Using the deployment circuit breaker</a>
 * for more details.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object service = FargateService.Builder.create(stack, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .circuitBreaker(Map.of("rollback", true))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Include an application/network load balancer</h3>
 * <p>
 * <code>Services</code> are load balancing targets and can be added to a target group, which will be attached to an application/network load balancers:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_elasticloadbalancingv2;
 * 
 * 
 * Object service = FargateService.Builder.create(this, "Service").build();
 * 
 * ApplicationLoadBalancer lb = new ApplicationLoadBalancer(this, "LB", new ApplicationLoadBalancerProps().vpc(vpc).internetFacing(true));
 * ApplicationListener listener = lb.addListener("Listener", new BaseApplicationListenerProps().port(80));
 * ApplicationTargetGroup targetGroup1 = listener.addTargets("ECS1", new AddApplicationTargetsProps()
 *         .port(80)
 *         .targets(asList(service)));
 * ApplicationTargetGroup targetGroup2 = listener.addTargets("ECS2", new AddApplicationTargetsProps()
 *         .port(80)
 *         .targets(asList(service.loadBalancerTarget(Map.of(
 *                 "containerName", "MyContainer",
 *                 "containerPort", 8080)))));
 * </pre></blockquote>
 * <p>
 * Note that in the example above, the default <code>service</code> only allows you to register the first essential container or the first mapped port on the container as a target and add it to a new target group. To have more control over which container and port to register as targets, you can use <code>service.loadBalancerTarget()</code> to return a load balancing target for a specific container and port.
 * <p>
 * Alternatively, you can also create all load balancer targets to be registered in this service, add them to target groups, and attach target groups to listeners accordingly.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_elasticloadbalancingv2;
 * 
 * 
 * Object service = FargateService.Builder.create(this, "Service").build();
 * 
 * ApplicationLoadBalancer lb = new ApplicationLoadBalancer(this, "LB", new ApplicationLoadBalancerProps().vpc(vpc).internetFacing(true));
 * ApplicationListener listener = lb.addListener("Listener", new BaseApplicationListenerProps().port(80));
 * service.registerLoadBalancerTargets(Map.of(
 *         "containerName", "web",
 *         "containerPort", 80,
 *         "newTargetGroupId", "ECS",
 *         "listener", ecs.ListenerConfig.applicationListener(listener, Map.of(
 *                 "protocol", elbv2.ApplicationProtocol.getHTTPS()))));
 * </pre></blockquote>
 * <p>
 * <h3>Using a Load Balancer from a different Stack</h3>
 * <p>
 * If you want to put your Load Balancer and the Service it is load balancing to in
 * different stacks, you may not be able to use the convenience methods
 * <code>loadBalancer.addListener()</code> and <code>listener.addTargets()</code>.
 * <p>
 * The reason is that these methods will create resources in the same Stack as the
 * object they're called on, which may lead to cyclic references between stacks.
 * Instead, you will have to create an <code>ApplicationListener</code> in the service stack,
 * or an empty <code>TargetGroup</code> in the load balancer stack that you attach your
 * service to.
 * <p>
 * See the <a href="https://github.com/aws-samples/aws-cdk-examples/tree/master/typescript/ecs/cross-stack-load-balancer/">ecs/cross-stack-load-balancer example</a>
 * for the alternatives.
 * <p>
 * <h3>Include a classic load balancer</h3>
 * <p>
 * <code>Services</code> can also be directly attached to a classic load balancer as targets:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_elasticloadbalancing;
 * 
 * 
 * Object service = Ec2Service.Builder.create(this, "Service").build();
 * 
 * LoadBalancer lb = new LoadBalancer(stack, "LB", new LoadBalancerProps().vpc(vpc));
 * lb.addListener(new LoadBalancerListener().externalPort(80));
 * lb.addTarget(service);
 * </pre></blockquote>
 * <p>
 * Similarly, if you want to have more control over load balancer targeting:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_elasticloadbalancing;
 * 
 * 
 * Object service = Ec2Service.Builder.create(this, "Service").build();
 * 
 * LoadBalancer lb = new LoadBalancer(stack, "LB", new LoadBalancerProps().vpc(vpc));
 * lb.addListener(new LoadBalancerListener().externalPort(80));
 * lb.addTarget(service.loadBalancerTarget(Map.of(
 *         "containerName", "MyContainer",
 *         "containerPort", 80)));
 * </pre></blockquote>
 * <p>
 * There are two higher-level constructs available which include a load balancer for you that can be found in the aws-ecs-patterns module:
 * <p>
 * <ul>
 * <li><code>LoadBalancedFargateService</code></li>
 * <li><code>LoadBalancedEc2Service</code></li>
 * </ul>
 * <p>
 * <h2>Task Auto-Scaling</h2>
 * <p>
 * You can configure the task count of a service to match demand. Task auto-scaling is
 * configured by calling <code>autoScaleTaskCount()</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object scaling = service.autoScaleTaskCount(Map.of("maxCapacity", 10));
 * scaling.scaleOnCpuUtilization("CpuScaling", Map.of(
 *         "targetUtilizationPercent", 50));
 * 
 * scaling.scaleOnRequestCount("RequestScaling", Map.of(
 *         "requestsPerTarget", 10000,
 *         "targetGroup", target));
 * </pre></blockquote>
 * <p>
 * Task auto-scaling is powered by <em>Application Auto-Scaling</em>.
 * See that section for details.
 * <p>
 * <h2>Integration with CloudWatch Events</h2>
 * <p>
 * To start an Amazon ECS task on an Amazon EC2-backed Cluster, instantiate an
 * <code>&#64;aws-cdk/aws-events-targets.EcsTask</code> instead of an <code>Ec2Service</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_events_targets;
 * 
 * 
 * // Create a Task Definition for the container to start
 * Object taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromAsset(path.resolve(__dirname, "..", "eventhandler-image")),
 *         "memoryLimitMiB", 256,
 *         "logging", AwsLogDriver.Builder.create().streamPrefix("EventDemo").mode(AwsLogDriverMode.getNON_BLOCKING()).build()));
 * 
 * // An Rule that describes the event trigger (in this case a scheduled run)
 * Object rule = Rule.Builder.create(this, "Rule")
 *         .schedule(events.Schedule.expression("rate(1 min)"))
 *         .build();
 * 
 * // Pass an environment variable to the container 'TheContainer' in the task
 * rule.addTarget(new EcsTask(new EcsTaskProps()
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .taskCount(1)
 *         .containerOverrides(asList(new ContainerOverride()
 *                 .containerName("TheContainer")
 *                 .environment(asList(new TaskEnvironmentVariable()
 *                         .name("I_WAS_TRIGGERED")
 *                         .value("From CloudWatch Events")))))));
 * </pre></blockquote>
 * <p>
 * <h2>Log Drivers</h2>
 * <p>
 * Currently Supported Log Drivers:
 * <p>
 * <ul>
 * <li>awslogs</li>
 * <li>fluentd</li>
 * <li>gelf</li>
 * <li>journald</li>
 * <li>json-file</li>
 * <li>splunk</li>
 * <li>syslog</li>
 * <li>awsfirelens</li>
 * </ul>
 * <p>
 * <h3>awslogs Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * Object taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", ecs.LogDrivers.awsLogs(Map.of("streamPrefix", "EventDemo"))));
 * </pre></blockquote>
 * <p>
 * <h3>fluentd Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * Object taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", ecs.LogDrivers.fluentd()));
 * </pre></blockquote>
 * <p>
 * <h3>gelf Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * Object taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", ecs.LogDrivers.gelf(Map.of("address", "my-gelf-address"))));
 * </pre></blockquote>
 * <p>
 * <h3>journald Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * Object taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", ecs.LogDrivers.journald()));
 * </pre></blockquote>
 * <p>
 * <h3>json-file Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * Object taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", ecs.LogDrivers.jsonFile()));
 * </pre></blockquote>
 * <p>
 * <h3>splunk Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * Object taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", ecs.LogDrivers.splunk(Map.of(
 *                 "token", cdk.SecretValue.secretsManager("my-splunk-token"),
 *                 "url", "my-splunk-url"))));
 * </pre></blockquote>
 * <p>
 * <h3>syslog Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * Object taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", ecs.LogDrivers.syslog()));
 * </pre></blockquote>
 * <p>
 * <h3>firelens Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * Object taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", ecs.LogDrivers.firelens(Map.of(
 *                 "options", Map.of(
 *                         "Name", "firehose",
 *                         "region", "us-west-2",
 *                         "delivery_stream", "my-stream")))));
 * </pre></blockquote>
 * <p>
 * <h3>Generic Log Driver</h3>
 * <p>
 * A generic log driver object exists to provide a lower level abstraction of the log driver configuration.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Create a Task Definition for the container to start
 * Object taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("example-image"),
 *         "memoryLimitMiB", 256,
 *         "logging", GenericLogDriver.Builder.create()
 *                 .logDriver("fluentd")
 *                 .options(Map.of(
 *                         "tag", "example-tag"))
 *                 .build()));
 * </pre></blockquote>
 * <p>
 * <h2>CloudMap Service Discovery</h2>
 * <p>
 * To register your ECS service with a CloudMap Service Registry, you may add the
 * <code>cloudMapOptions</code> property to your service:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object service = Ec2Service.Builder.create(stack, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .cloudMapOptions(Map.of(
 *                 // Create A records - useful for AWSVPC network mode.
 *                 "dnsRecordType", cloudmap.DnsRecordType.getA()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * With <code>bridge</code> or <code>host</code> network modes, only <code>SRV</code> DNS record types are supported.
 * By default, <code>SRV</code> DNS record types will target the default container and default
 * port. However, you may target a different container and port on the same ECS task:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // Add a container to the task definition
 * Object specificContainer = taskDefinition.addContainer(...);
 * 
 * // Add a port mapping
 * specificContainer.addPortMappings(Map.of(
 *         "containerPort", 7600,
 *         "protocol", ecs.Protocol.getTCP()));
 * 
 * Ec2Service.Builder.create(stack, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .cloudMapOptions(Map.of(
 *                 // Create SRV records - useful for bridge networking
 *                 "dnsRecordType", cloudmap.DnsRecordType.getSRV(),
 *                 // Targets port TCP port 7600 `specificContainer`
 *                 "container", specificContainer,
 *                 "containerPort", 7600))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Associate With a Specific CloudMap Service</h3>
 * <p>
 * You may associate an ECS service with a specific CloudMap service. To do
 * this, use the service's <code>associateCloudMapService</code> method:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object cloudMapService = new Service(...);
 * Object ecsService = new FargateService(...);
 * 
 * ecsService.associateCloudMapService(Map.of(
 *         "service", cloudMapService));
 * </pre></blockquote>
 * <p>
 * <h2>Capacity Providers</h2>
 * <p>
 * There are two major families of Capacity Providers: <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/fargate-capacity-providers.html">AWS
 * Fargate</a>
 * (including Fargate Spot) and EC2 <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/asg-capacity-providers.html">Auto Scaling
 * Group</a>
 * Capacity Providers. Both are supported.
 * <p>
 * <h3>Fargate Capacity Providers</h3>
 * <p>
 * To enable Fargate capacity providers, you can either set
 * <code>enableFargateCapacityProviders</code> to <code>true</code> when creating your cluster, or by
 * invoking the <code>enableFargateCapacityProviders()</code> method after creating your
 * cluster. This will add both <code>FARGATE</code> and <code>FARGATE_SPOT</code> as available capacity
 * providers on your cluster.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object cluster = Cluster.Builder.create(stack, "FargateCPCluster")
 *         .vpc(vpc)
 *         .enableFargateCapacityProviders(true)
 *         .build();
 * 
 * Object taskDefinition = new FargateTaskDefinition(stack, "TaskDef");
 * 
 * taskDefinition.addContainer("web", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample")));
 * 
 * FargateService.Builder.create(stack, "FargateService")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .capacityProviderStrategies(asList(Map.of(
 *                 "capacityProvider", "FARGATE_SPOT",
 *                 "weight", 2), Map.of(
 *                 "capacityProvider", "FARGATE",
 *                 "weight", 1)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Auto Scaling Group Capacity Providers</h3>
 * <p>
 * To add an Auto Scaling Group Capacity Provider, first create an EC2 Auto Scaling
 * Group. Then, create an <code>AsgCapacityProvider</code> and pass the Auto Scaling Group to
 * it in the constructor. Then add the Capacity Provider to the cluster. Finally,
 * you can refer to the Provider by its name in your service's or task's Capacity
 * Provider strategy.
 * <p>
 * By default, an Auto Scaling Group Capacity Provider will manage the Auto Scaling
 * Group's size for you. It will also enable managed termination protection, in
 * order to prevent EC2 Auto Scaling from terminating EC2 instances that have tasks
 * running on them. If you want to disable this behavior, set both
 * <code>enableManagedScaling</code> to and <code>enableManagedTerminationProtection</code> to <code>false</code>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object cluster = Cluster.Builder.create(stack, "Cluster")
 *         .vpc(vpc)
 *         .build();
 * 
 * Object autoScalingGroup = AutoScalingGroup.Builder.create(stack, "ASG")
 *         .vpc(vpc)
 *         .instanceType(new InstanceType("t2.micro"))
 *         .machineImage(ecs.EcsOptimizedImage.amazonLinux2())
 *         .minCapacity(0)
 *         .maxCapacity(100)
 *         .build();
 * 
 * Object capacityProvider = AsgCapacityProvider.Builder.create(stack, "AsgCapacityProvider")
 *         .autoScalingGroup(autoScalingGroup)
 *         .build();
 * cluster.addAsgCapacityProvider(capacityProvider);
 * 
 * Object taskDefinition = new Ec2TaskDefinition(stack, "TaskDef");
 * 
 * taskDefinition.addContainer("web", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample", memoryReservationMiB, 256)));
 * 
 * Ec2Service.Builder.create(stack, "EC2Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .capacityProviderStrategies(asList(Map.of(
 *                 "capacityProvider", capacityProvider.getCapacityProviderName(),
 *                 "weight", 1)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Elastic Inference Accelerators</h2>
 * <p>
 * Currently, this feature is only supported for services with EC2 launch types.
 * <p>
 * To add elastic inference accelerators to your EC2 instance, first add
 * <code>inferenceAccelerators</code> field to the Ec2TaskDefinition and set the <code>deviceName</code>
 * and <code>deviceType</code> properties.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Array inferenceAccelerators = asList(Map.of(
 *         "deviceName", "device1",
 *         "deviceType", "eia2.medium"));
 * 
 * Object taskDefinition = Ec2TaskDefinition.Builder.create(stack, "Ec2TaskDef")
 *         .inferenceAccelerators(inferenceAccelerators)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To enable using the inference accelerators in the containers, add <code>inferenceAcceleratorResources</code>
 * field and set it to a list of device names used for the inference accelerators. Each value in the
 * list should match a <code>DeviceName</code> for an <code>InferenceAccelerator</code> specified in the task definition.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Array inferenceAcceleratorResources = asList("device1");
 * 
 * taskDefinition.addContainer("cont", Map.of(
 *         "image", ecs.ContainerImage.fromRegistry("test"),
 *         "memoryLimitMiB", 1024,
 *         "inferenceAcceleratorResources", inferenceAcceleratorResources));
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.ecs;
