/**
 * <h1>Event Targets for Amazon EventBridge</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * This library contains integration classes to send Amazon EventBridge to any
 * number of supported AWS Services. Instances of these classes should be passed
 * to the <code>rule.addTarget()</code> method.
 * <p>
 * Currently supported are:
 * <p>
 * <ul>
 * <li><a href="#start-a-codebuild-build">Start a CodeBuild build</a></li>
 * <li><a href="#start-a-codepipeline-pipeline">Start a CodePipeline pipeline</a></li>
 * <li>Run an ECS task</li>
 * <li><a href="#invoke-a-lambda-function">Invoke a Lambda function</a></li>
 * <li><a href="#invoke-a-api-gateway-rest-api">Invoke a API Gateway REST API</a></li>
 * <li>Publish a message to an SNS topic</li>
 * <li>Send a message to an SQS queue</li>
 * <li><a href="#start-a-stepfunctions-state-machine">Start a StepFunctions state machine</a></li>
 * <li>Queue a Batch job</li>
 * <li>Make an AWS API call</li>
 * <li>Put a record to a Kinesis stream</li>
 * <li><a href="#log-an-event-into-a-loggroup">Log an event into a LogGroup</a></li>
 * <li>Put a record to a Kinesis Data Firehose stream</li>
 * <li>Put an event on an EventBridge bus</li>
 * </ul>
 * <p>
 * See the README of the <code>&#64;aws-cdk/aws-events</code> library for more information on
 * EventBridge.
 * <p>
 * <h2>Event retry policy and using dead-letter queues</h2>
 * <p>
 * The Codebuild, CodePipeline, Lambda, StepFunctions and LogGroup targets support attaching a <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/rule-dlq.html">dead letter queue and setting retry policies</a>. See the <a href="#invoke-a-lambda-function">lambda example</a>.
 * Use <a href="https://docs.aws.amazon.com/cdk/latest/guide/cfn_layer.html">escape hatches</a> for the other target types.
 * <p>
 * <h2>Invoke a Lambda function</h2>
 * <p>
 * Use the <code>LambdaFunction</code> target to invoke a lambda function.
 * <p>
 * The code snippet below creates an event rule with a Lambda function as a target
 * triggered for every events from <code>aws.ec2</code> source. You can optionally attach a
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/rule-dlq.html">dead letter queue</a>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_lambda;
 * import software.amazon.awscdk.aws_events;
 * import software.amazon.awscdk.aws_sqs;
 * import software.amazon.awscdk.aws_events_targets;
 * import software.amazon.awscdk.*;
 * 
 * 
 * Function fn = new Function(this, "MyFunc", new FunctionProps()
 *         .runtime(lambda.Runtime.getNODEJS_12_X())
 *         .handler("index.handler")
 *         .code(lambda.Code.fromInline(String.format("exports.handler = %s", handler.toString()))));
 * 
 * Rule rule = new Rule(this, "rule", new RuleProps()
 *         .eventPattern(new EventPattern()
 *                 .source(asList("aws.ec2"))));
 * 
 * Queue queue = new Queue(this, "Queue");
 * 
 * rule.addTarget(new LambdaFunction(fn, new LambdaFunctionProps()
 *         .deadLetterQueue(queue)// Optional: add a dead letter queue
 *         .maxEventAge(cdk.Duration.hours(2))// Otional: set the maxEventAge retry policy
 *         .retryAttempts(2)));
 * </pre></blockquote>
 * <p>
 * <h2>Log an event into a LogGroup</h2>
 * <p>
 * Use the <code>LogGroup</code> target to log your events in a CloudWatch LogGroup.
 * <p>
 * For example, the following code snippet creates an event rule with a CloudWatch LogGroup as a target.
 * Every events sent from the <code>aws.ec2</code> source will be sent to the CloudWatch LogGroup.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_logs;
 * import software.amazon.awscdk.aws_events;
 * import software.amazon.awscdk.aws_events_targets;
 * 
 * 
 * LogGroup logGroup = new LogGroup(this, "MyLogGroup", new LogGroupProps()
 *         .logGroupName("MyLogGroup"));
 * 
 * Rule rule = new Rule(this, "rule", new RuleProps()
 *         .eventPattern(new EventPattern()
 *                 .source(asList("aws.ec2"))));
 * 
 * rule.addTarget(new CloudWatchLogGroup(logGroup));
 * </pre></blockquote>
 * <p>
 * <h2>Start a CodeBuild build</h2>
 * <p>
 * Use the <code>CodeBuildProject</code> target to trigger a CodeBuild project.
 * <p>
 * The code snippet below creates a CodeCommit repository that triggers a CodeBuild project
 * on commit to the master branch. You can optionally attach a
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/rule-dlq.html">dead letter queue</a>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import aws.sdk.aws.codebuild.*;
 * import aws.sdk.aws.codecommit.*;
 * import aws.sdk.aws.sqs.*;
 * import software.amazon.awscdk.aws_events_targets;
 * 
 * 
 * Object repo = Repository.Builder.create(this, "MyRepo")
 *         .repositoryName("aws-cdk-codebuild-events")
 *         .build();
 * 
 * Object project = Project.Builder.create(this, "MyProject")
 *         .source(codebuild.Source.codeCommit(Map.of("repository", repo)))
 *         .build();
 * 
 * Object deadLetterQueue = new Queue(this, "DeadLetterQueue");
 * 
 * // trigger a build when a commit is pushed to the repo
 * Object onCommitRule = repo.onCommit("OnCommit", Map.of(
 *         "target", new CodeBuildProject(project, new CodeBuildProjectProps()
 *                 .deadLetterQueue(deadLetterQueue)),
 *         "branches", asList("master")));
 * </pre></blockquote>
 * <p>
 * <h2>Start a CodePipeline pipeline</h2>
 * <p>
 * Use the <code>CodePipeline</code> target to trigger a CodePipeline pipeline.
 * <p>
 * The code snippet below creates a CodePipeline pipeline that is triggered every hour
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import aws.sdk.aws.codepipeline.*;
 * import aws.sdk.aws.sqs.*;
 * 
 * 
 * Object pipeline = new Pipeline(this, "Pipeline");
 * 
 * Object rule = Rule.Builder.create(stack, "Rule")
 *         .schedule(events.Schedule.expression("rate(1 hour)"))
 *         .build();
 * 
 * rule.addTarget(new CodePipeline(pipeline));
 * </pre></blockquote>
 * <p>
 * <h2>Start a StepFunctions state machine</h2>
 * <p>
 * Use the <code>SfnStateMachine</code> target to trigger a State Machine.
 * <p>
 * The code snippet below creates a Simple StateMachine that is triggered every minute with a
 * dummy object as input.
 * You can optionally attach a
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/rule-dlq.html">dead letter queue</a>
 * to the target.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import aws.sdk.aws.iam.*;
 * import aws.sdk.aws.sqs.*;
 * import software.amazon.awscdk.aws_stepfunctions;
 * import software.amazon.awscdk.aws_events_targets;
 * 
 * 
 * Object rule = Rule.Builder.create(stack, "Rule")
 *         .schedule(events.Schedule.rate(cdk.Duration.minutes(1)))
 *         .build();
 * 
 * Object dlq = new Queue(stack, "DeadLetterQueue");
 * 
 * Object role = Role.Builder.create(stack, "Role")
 *         .assumedBy(new ServicePrincipal("events.amazonaws.com"))
 *         .build();
 * StateMachine stateMachine = new StateMachine(stack, "SM", new StateMachineProps()
 *         .definition(new Wait(stack, "Hello", new WaitProps().time(sfn.WaitTime.duration(cdk.Duration.seconds(10)))))
 *         .role(role));
 * 
 * rule.addTarget(new SfnStateMachine(stateMachine, new SfnStateMachineProps()
 *         .input(events.RuleTargetInput.fromObject(Map.of("SomeParam", "SomeValue")))
 *         .deadLetterQueue(dlq)));
 * </pre></blockquote>
 * <p>
 * <h2>Invoke a API Gateway REST API</h2>
 * <p>
 * Use the <code>ApiGateway</code> target to trigger a REST API.
 * <p>
 * The code snippet below creates a Api Gateway REST API that is invoked every hour.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import aws.sdk.aws.iam.*;
 * import aws.sdk.aws.sqs.*;
 * import software.amazon.awscdk.aws_apigateway;
 * import software.amazon.awscdk.aws_events_targets;
 * 
 * 
 * Object rule = Rule.Builder.create(stack, "Rule")
 *         .schedule(events.Schedule.rate(cdk.Duration.minutes(1)))
 *         .build();
 * 
 * Function fn = new Function(this, "MyFunc", new FunctionProps()
 *         .handler("index.handler")
 *         .runtime(lambda.Runtime.getNODEJS_12_X())
 *         .code(lambda.Code.fromInline("exports.handler = e =&gt; {}")));
 * 
 * LambdaRestApi restApi = new LambdaRestApi(this, "MyRestAPI", new LambdaRestApiProps().handler(fn));
 * 
 * Object dlq = new Queue(stack, "DeadLetterQueue");
 * 
 * rule.addTarget(
 * new ApiGateway(restApi, new ApiGatewayProps()
 *         .path("/*{@literal /}test")
 *         .mehod("GET")
 *         .stage("prod")
 *         .pathParameterValues(asList("path-value"))
 *         .headerParameters(Map.of(
 *                 "Header1", "header1"))
 *         .queryStringParameters(Map.of(
 *                 "QueryParam1", "query-param-1"))
 *         .deadLetterQueue(queue)));
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.events.targets;
