/**
 * <h1>AWS Secrets Manager Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_secretsmanager;
 * </pre></blockquote>
 * <p>
 * <h2>Create a new Secret in a Stack</h2>
 * <p>
 * In order to have SecretsManager generate a new secret value automatically,
 * you can get started with the following:
 * <p>
 * <a href="test/integ.secret.lit.ts">example of creating a secret</a>
 * <p>
 * The <code>Secret</code> construct does not allow specifying the <code>SecretString</code> property
 * of the <code>AWS::SecretsManager::Secret</code> resource (as this will almost always
 * lead to the secret being surfaced in plain text and possibly committed to
 * your source control).
 * <p>
 * If you need to use a pre-existing secret, the recommended way is to manually
 * provision the secret in <em>AWS SecretsManager</em> and use the <code>Secret.fromSecretArn</code>
 * or <code>Secret.fromSecretAttributes</code> method to make it available in your CDK Application:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object secret = secretsmanager.Secret.fromSecretAttributes(scope, "ImportedSecret", Map.of(
 *         "secretArn", "arn:aws:secretsmanager:&lt;region&gt;:&lt;account-id-number&gt;:secret:&lt;secret-name&gt;-&lt;random-6-characters&gt;",
 *         // If the secret is encrypted using a KMS-hosted CMK, either import or reference that key:
 *         "encryptionKey", encryptionKey));
 * </pre></blockquote>
 * <p>
 * SecretsManager secret values can only be used in select set of properties. For the
 * list of properties, see <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html">the CloudFormation Dynamic References documentation</a>.
 * <p>
 * A secret can set <code>RemovalPolicy</code>. If it set to <code>RETAIN</code>, that removing a secret will fail.
 * <p>
 * <h2>Grant permission to use the secret to a role</h2>
 * <p>
 * You must grant permission to a resource for that resource to be allowed to
 * use a secret. This can be achieved with the <code>Secret.grantRead</code> and/or <code>Secret.grantUpdate</code>
 * method, depending on your need:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Role role = new Role(stack, "SomeRole", new RoleProps().assumedBy(new AccountRootPrincipal()));
 * Object secret = new Secret(stack, "Secret");
 * secret.grantRead(role);
 * secret.grantWrite(role);
 * </pre></blockquote>
 * <p>
 * If, as in the following example, your secret was created with a KMS key:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object key = new Key(stack, "KMS");
 * Object secret = Secret.Builder.create(stack, "Secret").encryptionKey(key).build();
 * secret.grantRead(role);
 * secret.grantWrite(role);
 * </pre></blockquote>
 * <p>
 * then <code>Secret.grantRead</code> and <code>Secret.grantWrite</code> will also grant the role the
 * relevant encrypt and decrypt permissions to the KMS key through the
 * SecretsManager service principal.
 * <p>
 * <h2>Rotating a Secret</h2>
 * <p>
 * <h3>Using a Custom Lambda Function</h3>
 * <p>
 * A rotation schedule can be added to a Secret using a custom Lambda function:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Function fn = new Function(...);
 * Object secret = new Secret(this, "Secret");
 * 
 * secret.addRotationSchedule("RotationSchedule", Map.of(
 *         "rotationLambda", fn,
 *         "automaticallyAfter", Duration.days(15)));
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/rotating-secrets-lambda-function-overview.html">Overview of the Lambda Rotation Function</a> on how to implement a Lambda Rotation Function.
 * <p>
 * <h3>Using a Hosted Lambda Function</h3>
 * <p>
 * Use the <code>hostedRotation</code> prop to rotate a secret with a hosted Lambda function:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object secret = new Secret(this, "Secret");
 * 
 * secret.addRotationSchedule("RotationSchedule", Map.of(
 *         "hostedRotation", secretsmanager.HostedRotation.mysqlSingleUser()));
 * </pre></blockquote>
 * <p>
 * Hosted rotation is available for secrets representing credentials for MySQL, PostgreSQL, Oracle,
 * MariaDB, SQLServer, Redshift and MongoDB (both for the single and multi user schemes).
 * <p>
 * When deployed in a VPC, the hosted rotation implements <code>ec2.IConnectable</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object myHostedRotation = secretsmanager.HostedRotation.mysqlSingleUser(Map.of("vpc", myVpc));
 * secret.addRotationSchedule("RotationSchedule", Map.of("hostedRotation", myHostedRotation));
 * dbConnections.allowDefaultPortFrom(hostedRotation);
 * </pre></blockquote>
 * <p>
 * See also <a href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/integrating_cloudformation.html">Automating secret creation in AWS CloudFormation</a>.
 * <p>
 * <h2>Rotating database credentials</h2>
 * <p>
 * Define a <code>SecretRotation</code> to rotate database credentials:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * SecretRotation.Builder.create(this, "SecretRotation")
 *         .application(secretsmanager.SecretRotationApplication.getMYSQL_ROTATION_SINGLE_USER())// MySQL single user scheme
 *         .secret(mySecret)
 *         .target(myDatabase)// a Connectable
 *         .vpc(myVpc)// The VPC where the secret rotation application will be deployed
 *         .excludeCharacters(" %+:;{}")
 *         .build();
 * </pre></blockquote>
 * <p>
 * The secret must be a JSON string with the following format:
 * <p>
 * <blockquote><pre>
 * {
 *   "engine": "&lt;required: database engine&gt;",
 *   "host": "&lt;required: instance host name&gt;",
 *   "username": "&lt;required: username&gt;",
 *   "password": "&lt;required: password&gt;",
 *   "dbname": "&lt;optional: database name&gt;",
 *   "port": "&lt;optional: if not specified, default port will be used&gt;",
 *   "masterarn": "&lt;required for multi user rotation: the arn of the master secret which will be used to create users/change passwords&gt;"
 * }
 * </pre></blockquote>
 * <p>
 * For the multi user scheme, a <code>masterSecret</code> must be specified:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * SecretRotation.Builder.create(stack, "SecretRotation")
 *         .application(secretsmanager.SecretRotationApplication.getMYSQL_ROTATION_MULTI_USER())
 *         .secret(myUserSecret)// The secret that will be rotated
 *         .masterSecret(myMasterSecret)// The secret used for the rotation
 *         .target(myDatabase)
 *         .vpc(myVpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See also <a href="https://github.com/aws/aws-cdk/blob/master/packages/%40aws-cdk/aws-rds/README.md">aws-rds</a> where
 * credentials generation and rotation is integrated.
 * <p>
 * <h2>Importing Secrets</h2>
 * <p>
 * Existing secrets can be imported by ARN, name, and other attributes (including the KMS key used to encrypt the secret).
 * Secrets imported by name should use the short-form of the name (without the SecretsManager-provided suffx);
 * the secret name must exist in the same account and region as the stack.
 * Importing by name makes it easier to reference secrets created in different regions, each with their own suffix and ARN.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.aws_kms;
 * 
 * 
 * String secretCompleteArn = "arn:aws:secretsmanager:eu-west-1:111111111111:secret:MySecret-f3gDy9";
 * String secretPartialArn = "arn:aws:secretsmanager:eu-west-1:111111111111:secret:MySecret";// No Secrets Manager suffix
 * IKey encryptionKey = kms.Key.fromKeyArn(stack, "MyEncKey", "arn:aws:kms:eu-west-1:111111111111:key/21c4b39b-fde2-4273-9ac0-d9bb5c0d0030");
 * Object mySecretFromCompleteArn = secretsmanager.Secret.fromSecretCompleteArn(stack, "SecretFromCompleteArn", secretCompleteArn);
 * Object mySecretFromPartialArn = secretsmanager.Secret.fromSecretPartialArn(stack, "SecretFromPartialArn", secretPartialArn);
 * Object mySecretFromName = secretsmanager.Secret.fromSecretNameV2(stack, "SecretFromName", "MySecret");
 * Object mySecretFromAttrs = secretsmanager.Secret.fromSecretAttributes(stack, "SecretFromAttributes", Map.of(
 *         "secretCompleteArn", secretCompleteArn,
 *         "encryptionKey", encryptionKey));
 * </pre></blockquote>
 * <p>
 * <h2>Replicating secrets</h2>
 * <p>
 * Secrets can be replicated to multiple regions by specifying <code>replicaRegions</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Secret.Builder.create(this, "Secret")
 *         .replicaRegions(asList(Map.of(
 *                 "region", "eu-west-1"), Map.of(
 *                 "region", "eu-central-1",
 *                 "encryptionKey", myKey)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, use <code>addReplicaRegion()</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object secret = new Secret(this, "Secret");
 * secret.addReplicaRegion("eu-west-1");
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.secretsmanager;
